# -*- coding: utf-8 -*-
import json
import logging
import time

from passport.backend.api.common import simple_error_response
from passport.backend.api.common.logbroker import get_api_logbroker_writer
from passport.backend.core.am_pushes.push_request import (
    AppTargetingTypes,
    PushRequest,
    PushRequestRecipient,
)
from passport.backend.core.am_pushes.subscription_manager import (
    AM_CAPABILITY_PUSH_PROTOCOL,
    get_pushes_subscription_manager,
)
from passport.backend.core.builders.push_api import (
    get_push_api,
    PushApiForbiddenServiceError,
    PushApiInvalidCertError,
    PushApiInvalidTokenError,
    PushApiUnsupportedPlatformError,
)
from passport.backend.core.conf import settings
from passport.backend.core.crypto.utils import dumb_hash_string
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.protobuf.challenge_pushes.challenge_pushes_pb2 import ChallengePushRequest
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import noneless_dict

from ..base import BaseBundleView
from ..headers import (
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from ..mixins import BundleAccountGetterMixin
from .exceptions import (
    PushApiBundleForbiddenServiceError,
    PushApiBundleInvalidCertError,
    PushApiBundleInvalidTokenError,
    PushApiBundleNoSubscriptionsError,
    PushApiBundleUnsupportedAppPlatformError,
)
from .forms import (
    PushApiRegisterForm,
    PushApiSendAMPushForm,
    PushApiSubscribeForm,
    PushApiUnsubscribeForm,
)


log = logging.getLogger('passport.api.view.bundle.push')


class BasePushApiBundleView(BaseBundleView, BundleAccountGetterMixin):
    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            mode='push',
            user_agent=self.user_agent,
        )

    @cached_property
    def push_api(self):
        return get_push_api()

    def generate_uuid(self):
        return dumb_hash_string(self.form_values['deviceid'] + self.form_values['app_id'])


class PushApiRegisterBundleView(BasePushApiBundleView):
    """
    Зарегистрировать приложение в сервисе Push API
    """
    basic_form = PushApiRegisterForm
    required_grants = ['push_api.register']

    def process_request(self):
        self.process_basic_form()
        try:
            self.push_api.register(
                app_name=self.form_values['app_id'],
                cert=self.form_values['cert'],
                platform=self.form_values['app_platform'],
            )
        except PushApiInvalidCertError:
            raise PushApiBundleInvalidCertError()
        except PushApiUnsupportedPlatformError:
            raise PushApiBundleUnsupportedAppPlatformError()

        self.statbox.log(
            action='register',
            app_id=self.form_values['app_id'],
            app_platform=self.form_values['app_platform'],
        )


class PushApiSendAMPushBundleView(BasePushApiBundleView):
    """
    Отправить пользователю пуш по AM-протоколу
    """
    basic_form = PushApiSendAMPushForm
    required_grants = ['push_api.send']

    def error_response(self, errors, **extra_response_values):
        """Антифроду нужно, чтобы мы на ошибки в этой ручке отдавали статус 5хх"""
        if any(error.startswith('backend.') for error in errors):
            extra_response_values.update(code=504)
        if 'exception.unhandled' in errors:
            extra_response_values.update(code=500)
        return simple_error_response(errors, **extra_response_values)

    @cached_property
    def logbroker(self):
        return get_api_logbroker_writer('challenge_pushes')

    def send_push(self):
        context = None
        if self.form_values['context']:
            context = json.dumps(self.form_values['context'])
        expire_time = None
        if self.form_values['ttl']:
            expire_time = int(time.time() + self.form_values['ttl'])
        push_request = PushRequest(
            push_service=self.form_values['push_service'],
            event_name=self.form_values['event_name'],
            recipients=PushRequestRecipient(
                uid=self.form_values['uid'],
                app_targeting_type=AppTargetingTypes.ONE_APP_PER_DEVICE,
                required_am_capabilities=[AM_CAPABILITY_PUSH_PROTOCOL],
                require_trusted_device=self.form_values['require_trusted_device'],
                context=context,
            ),
            title=self.form_values['title'],
            body=self.form_values['body'],
            subtitle=self.form_values['subtitle'],
            webview_url=self.form_values['webview_url'],
            require_web_auth=self.form_values['require_web_auth'],
            expire_time=expire_time,
        )
        log.debug('Sending push message {} for uid {}: content={}'.format(
            push_request.push_id,
            self.form_values['uid'],
            trim_message(str(dict(
                title=self.form_values['title'],
                subtitle=self.form_values['subtitle'],
                body=self.form_values['body'],
            ))),
        ))
        self.logbroker.send(
            ChallengePushRequest(
                push_message_request=push_request.to_proto(),
            ),
        )

    @cached_property
    def manager(self):
        return get_pushes_subscription_manager(
            uid=self.uid,
            push_service=self.form_values['push_service'],
            event=self.form_values['event_name'],
        )

    def has_compatible_subscriptions(self):
        if self.form_values['require_trusted_device']:
            return self.manager.has_trusted_subscriptions()
        else:
            return self.manager.has_am_compatible_subscriptions()

    def process_request(self):
        self.process_basic_form()
        self.uid = self.form_values['uid']
        self.statbox.bind(
            consumer=self.consumer,
            host=self.host,
            mode='push_send_am',
            action='send',
            push_service=self.form_values['push_service'],
            event_name=self.form_values['event_name'],
            uid=self.form_values['uid'],
            check_subscriptions='yes' if self.form_values['check_subscriptions'] else 'no',
            require_trusted_device='yes' if self.form_values['require_trusted_device'] else 'no',
        )
        if (
            not self.form_values['check_subscriptions'] or
            self.has_compatible_subscriptions()
        ):
            self.send_push()
            self.statbox.log(status='ok')
        else:
            self.statbox.log(status='error', error='no_subscriptions')
            raise PushApiBundleNoSubscriptionsError()


class PushApiSubscribeBundleView(BasePushApiBundleView):
    """
    Подписать инсталяцию приложения на пуши
    """
    basic_form = PushApiSubscribeForm
    required_grants = ['push_api.subscribe']
    required_headers = (
        HEADER_CONSUMER_AUTHORIZATION,
        HEADER_CONSUMER_CLIENT_IP,
    )

    def get_app_name(self):
        platform = self.push_api.translate_platform(self.form_values['app_platform'])
        app_name = self.form_values['app_id']
        if (
            platform == 'gcm' and
            not app_name.endswith(settings.PUSH_APP_NAME_POSTFIX_ANDROID)
        ):
            app_name = app_name + settings.PUSH_APP_NAME_POSTFIX_ANDROID

        return app_name

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_oauth_token()
        try:
            uuid = self.generate_uuid()
            extra_fields = noneless_dict(
                am_version=self.form_values['am_version'],
                login_id=self.login_id,
                app_version=self.form_values['app_version'],
            )
            self.push_api.subscribe(
                app_name=self.get_app_name(),
                device_token=self.form_values['device_token'],
                device=self.form_values['deviceid'],
                platform=self.form_values['app_platform'],
                user=self.account.uid,
                uuid=uuid,
                extra_fields=extra_fields,
            )
        except PushApiInvalidTokenError:
            raise PushApiBundleInvalidTokenError()
        except PushApiForbiddenServiceError:
            raise PushApiBundleForbiddenServiceError()
        except PushApiUnsupportedPlatformError:
            raise PushApiBundleUnsupportedAppPlatformError()

        self.statbox.log(
            action='subscribed',
            app_id=self.form_values['app_id'],
            app_platform=self.form_values['app_platform'],
            service_name=settings.PUSH_API_SERVICE_NAME,
            uid=self.account.uid,
            uuid=uuid,
        )


class PushApiUnsubscribeBundleView(BasePushApiBundleView):
    """
    Отписать инсталяцию приложения от пушей
    """
    basic_form = PushApiUnsubscribeForm
    required_grants = ['push_api.subscribe']
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    def process_request(self):
        self.process_basic_form()
        try:
            uuid = self.generate_uuid()
            self.push_api.unsubscribe(
                user=self.form_values['uid'],
                uuid=uuid,
            )
        except PushApiInvalidTokenError:
            raise PushApiBundleInvalidTokenError()
        except PushApiForbiddenServiceError:
            raise PushApiBundleForbiddenServiceError()

        self.statbox.log(
            action='unsubscribed',
            service_name=settings.PUSH_API_SERVICE_NAME,
            uid=self.form_values['uid'],
            uuid=uuid,
        )
