# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    OAuthTokenDeviceNotTrustedError,
    RateLimitExceedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.am_pushes.subscription_manager import get_pushes_subscription_manager
from passport.backend.core.builders.push_api import get_push_api
from passport.backend.core.conf import settings
from passport.backend.core.logbroker.exceptions import BaseLogbrokerError
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import generate_random_code

from ...mixins.kolmogor import KolmogorMixin
from .exceptions import PushApiNoCompatibleSubscriptions


log = logging.getLogger('passport.api.views.bundle.push.2fa')


class BasePush2faView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundlePushMixin,
    KolmogorMixin,
):
    mode = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.mode,
            host=self.host,
            uid=self.account.uid,
            track_id=self.track.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
        )

    def process_request(self):
        raise NotImplementedError()


class SendAuthPush2faView(
    BasePush2faView,
):
    mode = 'push_2fa_send'
    require_track = True
    required_grants = ['push_2fa.send']
    methods = ['POST']
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    @cached_property
    def push_api(self):
        return get_push_api()

    def check_prerequisites(self):
        if not self.track.uid:
            log.debug('No uid in track')
            raise InvalidTrackStateError()
        self.get_account_by_uid(self.track_uid)
        if not self.user_has_push_2fa_subscriptions():
            log.debug('Account has no compatible subscriptions')
            raise PushApiNoCompatibleSubscriptions()
        if not self.assert_push_2fa_counters():
            log.debug('Auth push rate exceeded')
            raise RateLimitExceedError()

    def process_request(self):
        self.read_track()
        self.check_prerequisites()

        with self.track_transaction.rollback_on_error():
            self.track.push_otp = generate_random_code(settings.PUSH_2FA_CODE_LENGTH)

        try:
            self.send_push_2fa(context=json.dumps(dict(track=self.track_id)))
            self.statbox.bind_context(status='ok')
        except BaseLogbrokerError:
            log.exception('Error sending 2fa push')
            self.statbox.bind_context(status='failed')
            raise
        finally:
            self.statbox.log()


class GetPush2faAuthCodeView(BasePush2faView):
    mode = 'push_2fa_get_code'
    required_grants = ['push_2fa.get_code']
    require_track = True
    methods = ['GET']
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    def check_track(self):
        if not self.track.uid:
            log.debug('No uid in track')
            raise InvalidTrackStateError()
        if not self.track.push_otp:
            log.debug('No push otp code in track')
            raise InvalidTrackStateError()

    @cached_property
    def manager(self):
        return get_pushes_subscription_manager(
            self.account.uid,
            push_service='2fa',
            event='2fa_code',
        )

    def check_login_id(self):
        if self.login_id and self.login_id not in self.manager.get_trusted_login_ids(self.account.uid):
            log.debug('Login id {} is not from trusted xtoken')
            raise OAuthTokenDeviceNotTrustedError()

    def process_request(self):
        self.read_track()
        self.check_track()
        self.get_account_from_session(multisession_uid=self.track_uid)
        # self.check_login_id()  FIXME: раскомментировать после закрытия PASSP-33827

        self.response_values.update(otp=self.track.push_otp)
        self.statbox.log(status='ok')
