# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
    kolonkish_account,
    set_impossible_password,
    set_password_with_experiment,
    yambot_account,
)
from passport.backend.api.common.authorization import (
    authorize_oauth,
    SessionScope,
    set_authorization_track_fields,
)
from passport.backend.api.common.common import extract_tld
from passport.backend.api.common.grants import generate_subscription_grant
from passport.backend.api.common.login import (
    build_available_kolonkish_login,
    build_available_yambot_login,
)
from passport.backend.api.common.pdd import (
    does_domain_belong_to_pdd,
    is_domain_suitable_for_directory,
)
from passport.backend.api.common.processes import PROCESS_WEB_REGISTRATION
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.email_validator.exceptions import (
    EmailAlreadyConfirmedError,
    EmailIncorrectKeyError,
)
from passport.backend.api.email_validator.mixins import EmailValidatorMixin
from passport.backend.api.forms.register import (
    CAPTCHA_VALIDATION_METHOD,
    PHONE_VALIDATION_METHOD,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    AccountInvalidTypeError,
    ActionImpossibleError,
    DomainAlreadyExistsError,
    DomainInvalidTypeError,
    DomainNotFoundError,
    EmailConfirmationsLimitExceededError,
    EmailSendLimitExceededError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    OAuthUnavailableError,
    RateLimitExceedError,
    UserNotVerifiedError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAssertCaptchaMixin,
    BundleFederalMixin,
    BundleFrodoMixin,
    BundleNeophonishMixin,
    BundlePasswordValidationMixin,
    BundlePersistentTrackMixin,
    BundlePhoneMixin,
    BundleScholarMixin,
    MailSubscriptionsMixin,
    RegisterPhoneAliasManager,
    SecurePhoneBindAndAliasifyMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleCleanWebMixin
from passport.backend.api.views.bundle.register import exceptions
from passport.backend.api.views.bundle.register.forms import (
    AccountCreateForm,
    AccountRegisterByMiddlemanForm,
    AccountRegisterDirectoryForm,
    AccountRegisterEasyForm,
    AccountRegisterFederalForm,
    AccountRegisterIntranetForm,
    AccountRegisterKolonkishForm,
    AccountRegisterLiteCommitForm,
    AccountRegisterLiteSubmitForm,
    AccountRegisterNeophonishForm,
    AccountRegisterPddForm,
    AccountRegisterRequireConfirmedPhoneAndAliasify,
    AccountRegisterScholarForm,
    AccountRegisterWithHintForm,
    BaseAccountRegisterAlternativeForm,
    ConfirmEmailForm,
    SendRegistrationConfirmationCodeForm,
)
from passport.backend.api.views.bundle.register.helpers import get_auth_by_key_link_email_data
from passport.backend.api.views.bundle.utils import make_hint_question
from passport.backend.api.views.register import (
    incr_registration_sms_per_ip_counter,
    is_registration_sms_per_ip_counter_exceed,
)
from passport.backend.core import (
    language_detect,
    validators,
)
from passport.backend.core.builders.oauth import get_oauth
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    register_email_per_email,
    register_email_per_ip,
    register_kolonkish_per_creator_uid,
    sms_per_ip,
    sms_per_ip_for_consumer,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    login_shadower,
    MailInfo,
    make_email_context,
    render_to_sendmail,
    send_mail_for_account,
)
from passport.backend.core.models.alias import (
    AltDomainAlias,
    YambotAlias,
)
from passport.backend.core.models.domain import Domain
from passport.backend.core.models.persistent_track import TRACK_TYPE_AUTH_BY_KEY_LINK
from passport.backend.core.password.password_quality import get_password_qualifier
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.types.account.account import ACCOUNT_TYPE_LITE
from passport.backend.core.types.email.email import (
    get_default_native_emails,
    mask_email_for_statbox,
    unicode_email,
)
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import (
    generate_random_code,
    remove_none_values,
)
from passport.backend.utils.time import get_unixtime


log = logging.getLogger('passport.api.view.bundle.register')

PDD_DISPLAY_TEMPLATE = 't:%pdd_username%@%display_domain%'
IGNORE_STOPLIST_GRANT = 'ignore_stoplist'
ACCOUNT_IS_ENABLED_GRANT = 'account.is_enabled'
CREATE_GRANT = 'create'
ACCOUNT_RECOVERY_EMAIL_GRANT = 'account.recovery_email'


class RegistrationConfirmationEmailBaseView(BaseBundleView):
    process_name = PROCESS_WEB_REGISTRATION
    require_track = True
    require_process = True
    required_grants = ['account.register_mail']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_by_email',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
        )


class SendRegistrationConfirmationEmailView(RegistrationConfirmationEmailBaseView, BundleAssertCaptchaMixin, BundlePhoneMixin):
    """Отправка письма подтверждения регистрации. Работает без аккаунта, сохранённого в базу."""
    basic_form = SendRegistrationConfirmationCodeForm

    def check_counters(self, email):
        per_ip_counter = register_email_per_ip.get_counter(self.client_ip)

        if per_ip_counter.hit_limit(self.client_ip):
            raise EmailSendLimitExceededError(email)

        per_email_counter = register_email_per_email.get_counter()

        if per_email_counter.hit_limit(email):
            raise EmailSendLimitExceededError(email)

    def incr_counters(self, email):
        per_ip_counter = register_email_per_ip.get_counter(self.client_ip)
        per_email_counter = register_email_per_email.get_counter()

        per_ip_counter.incr(self.client_ip)
        per_email_counter.incr(email)

    def send_email(self, language, short_code):
        email = self.track.email_confirmation_address

        phrases = settings.translations.NOTIFICATIONS[language]
        user_tld = extract_tld(self.request.env.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD

        context = make_email_context(
            language=language,
            account=None,
            context={
                'TLD': user_tld,
                'ADDRESS': unicode_email(email),
                'SHORT_CODE': short_code,
                'greeting_key': 'greeting.noname',
            },
        )

        info = MailInfo(
            subject=phrases['emails.registration_confirmation_email_sent.subject'],
            from_=phrases['email_sender_display_name'],
            tld=user_tld,
        )

        log.debug(
            'Sending registration mail to %s.',
            email,
        )

        self.statbox.log(
            address=mask_email_for_statbox(email),
            action='send_confirmation_email',
            confirmation_checks_count=self.track.email_confirmation_checks_count.get(default=0),
        )

        render_to_sendmail(
            'mail/email_registration_confirmation_message.html',
            info,
            [email],
            context,
            is_plain_text=False,
        )

    def check_if_email_is_confirmed(self):
        if self.track.email_confirmation_passed_at:
            raise EmailAlreadyConfirmedError(
                address=self.track.email_confirmation_address,
                uid=None,
                code=None,
            )

    def check_passed_captcha_or_confirmed_phone(self):
        if not self.is_captcha_passed and not self.is_phone_confirmed_in_track(allow_by_flash_call=True):
            raise UserNotVerifiedError()

    def process_send(self, email):
        self.check_counters(email)

        short_code = self.track.email_confirmation_code

        is_new_address = self.track.email_confirmation_address != email
        limit = settings.ALLOWED_EMAIL_SHORT_CODE_FAILED_CHECK_COUNT
        limit_reached = self.track.email_confirmation_checks_count.get(default=0) >= limit
        if not short_code or is_new_address or limit_reached:
            with self.track_transaction.rollback_on_error():
                self.statbox.log(
                    address=mask_email_for_statbox(email),
                    action='new_code',
                    is_address_changed=is_new_address and bool(short_code),
                    is_limit_reached=limit_reached,
                )
                short_code = generate_random_code(settings.EMAIL_VALIDATOR_SHORT_CODE_LENGTH)
                self.track.email_confirmation_code = short_code
                self.track.email_confirmation_address = email
                self.track.email_confirmation_checks_count.reset()
                self.track.email_confirmation_passed_at = None

        self.incr_counters(email)
        language = language_detect.get_preferred_language(self.form_values['language'])
        self.send_email(language, short_code)

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        self.check_passed_captcha_or_confirmed_phone()

        email = self.form_values['email'].lower()
        if does_domain_belong_to_pdd(email):
            raise DomainInvalidTypeError()

        # NOTE Тут возможна ситуация, когда в email пришёл новый адрес, который ещё не подтверждён
        # и мы выкинем исключение, что другой (предыдущий) адрес подтверждён.
        self.check_if_email_is_confirmed()
        self.process_send(email)


class ConfirmRegistrationView(RegistrationConfirmationEmailBaseView):
    """Подтверждение почты кодом, без аккаунта, сохранённого в базу."""
    basic_form = ConfirmEmailForm

    def check_if_email_is_confirmed(self):
        if self.track.email_confirmation_passed_at:
            raise EmailAlreadyConfirmedError(
                address=self.track.email_confirmation_address,
                uid=None,
                code=None,
            )

    def get_confirmation_attempts_left(self):
        limit = settings.ALLOWED_EMAIL_SHORT_CODE_FAILED_CHECK_COUNT
        attempts_left = limit - self.track.email_confirmation_checks_count.get(default=0)
        return attempts_left

    def check_counters(self):
        attempts_left = self.get_confirmation_attempts_left()
        if attempts_left <= 0:
            raise EmailConfirmationsLimitExceededError()

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        if not self.track.email_confirmation_address or not self.track.email_confirmation_code:
            raise InvalidTrackStateError()

        self.check_if_email_is_confirmed()
        self.check_counters()

        masked_email = mask_email_for_statbox(self.track.email_confirmation_address)
        short_code = self.form_values['key']

        is_confirmed = short_code == self.track.email_confirmation_code

        with self.track_transaction.rollback_on_error():
            if is_confirmed:
                self.track.email_confirmation_passed_at = get_unixtime()
            self.track.email_confirmation_checks_count.incr()

        self.statbox.bind_context(
            address=masked_email,
            confirmation_checks_count=self.track.email_confirmation_checks_count.get(default=0),
        )

        if is_confirmed:
            self.statbox.log(action='email_confirmed')
        else:
            self.statbox.log(action='incorrect_key')
            self.response_values['email_confirmation_attempts_left'] = self.get_confirmation_attempts_left()
            raise EmailIncorrectKeyError(None, short_code)


class AccountRegisterPhoneAndAlisify(BaseBundleView,
                                     BundlePhoneMixin,
                                     BundlePasswordValidationMixin,
                                     BundleFrodoMixin,
                                     BundleCleanWebMixin,
                                     SecurePhoneBindAndAliasifyMixin):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )

    required_grants = ['account.register_phone_and_aliasify']

    require_track = True

    basic_form = AccountRegisterRequireConfirmedPhoneAndAliasify

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='digitsreg',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def remind_login(self, number):
        language = self.form_values['language']
        sms_text = settings.translations.NOTIFICATIONS[language]['login_reminder.sms'].replace(
            '%LOGIN%',
            self.account.login,
        )
        self.send_sms(number, sms_text, 'phone_registration_login_reminder')

    def send_greeting_email(self):
        RegisterPhoneAliasManager(
            consumer=self.consumer,
            environment=self.request.env,
        ).send_mail_about_alias_as_login_and_email_enabled(
            account=self.account,
            language=self.form_values['language'],
            phone_number=self.account.phonenumber_alias.number,
        )

    def process_request(self):
        self.statbox.log(
            action='submitted',
        )

        self.process_basic_form()
        self.clean_web_check_form_values()
        self.read_track()

        with self.track_transaction.rollback_on_error():
            track = self.track

            self.validate_password(
                uid='',
                login=self.form_values['login'],
                emails=get_default_native_emails(self.form_values['login']),
            )

            if track.is_successful_registered:
                raise AccountAlreadyRegisteredError()

            if not self.form_values['eula_accepted']:
                raise EulaIsNotAcceptedError()

            if not self.is_phone_confirmed_in_track(allow_by_flash_call=True):
                raise UserNotVerifiedError()

            if is_registration_sms_per_ip_counter_exceed(self.request.env, track, mode=self.statbox.all_values['mode']):
                raise exceptions.RegistrationSmsSendPerIPLimitExceededError()

            number = PhoneNumber.parse(track.phone_confirmation_phone_number)

            now = datetime.now()

            person_info = build_default_person_registration_info(self.client_ip)
            new_account = default_account(self.form_values['login'], now, self.form_values, person_info)
            new_account.emails = self.build_emails(number, new_account.registration_datetime)
            save_secure_phone = self.build_save_secure_phone(
                aliasify=True,
                language=self.form_values['language'],
                account=new_account,
                phone_number=number,
                should_ignore_binding_limit=False,
                is_new_account=True,
            )
            save_secure_phone.submit()

            events = {
                'action': 'account_register',
                'consumer': self.consumer,
            }
            with CREATE(new_account, self.request.env, events) as self.account:
                save_secure_phone.commit()
                add_subscription(self.account, get_service(slug='mail'))

            self.statbox.bind_context(uid=self.account.uid)
            save_secure_phone.after_commit()

            self.send_greeting_email()

            # отправляем смс-напоминалку с логином
            self.remind_login(number)

            track.is_successful_registered = True
            set_authorization_track_fields(
                self.account,
                track,
                allow_create_session=True,
                allow_create_token=True,
                password_passed=True,
                session_scope=SessionScope.xsession,
            )
            self.track.user_entered_login = self.form_values['login']

            self.frodo_check_spammer(action='phonereg', increment_counters=True)

            # http://wiki.yandex-team.ru/passport/statbox#chtopishem
            self.statbox.log(
                action='account_created',
                login=self.account.normalized_login,
                karma=self.account.karma.value,
                password_quality=self.form_values['quality'],
                is_suggested_login=self.account.login in track.suggested_logins.get(),
                suggest_generation_number=track.suggest_login_count.get(default=0),
                user_agent=self.user_agent,
                country=self.form_values['country'],
                language=track.language,
                retpath=track.retpath,
            )

            incr_registration_sms_per_ip_counter(self.request.env)

            self.response_values.update(uid=self.account.uid)

            process_env_profile(self.account, track=self.track)


class AccountRegisterByMiddleman(BaseBundleView,
                                 BundlePasswordValidationMixin):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    required_grants = ['account.register_by_middleman']

    require_track = False

    basic_form = AccountRegisterByMiddlemanForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_by_middleman',
            track_id=self.track_id,
        )

    def process_request(self):
        self.statbox.log(action='submitted')

        self.process_basic_form()
        self.read_or_create_track('register')

        with self.track_transaction.rollback_on_error():
            track = self.track

            self.validate_password(
                uid='',
                login=self.form_values['login'],
                emails=get_default_native_emails(self.form_values['login']),
            )

            if track.is_successful_registered:
                raise AccountAlreadyRegisteredError()

            now = datetime.now()
            person_info = build_default_person_registration_info(self.client_ip)

            with CREATE(default_account(self.form_values['login'], now, self.form_values, person_info),
                        self.request.env, {'action': 'account_register', 'consumer': self.consumer}) as self.account:
                self.account.password.is_creating_required = True
                add_subscription(self.account, get_service(slug='mail'))

            track.is_successful_registered = True
            set_authorization_track_fields(
                self.account,
                track,
                allow_create_session=False,
                allow_create_token=False,
                password_passed=True,
                session_scope=SessionScope.xsession,
            )

            # http://wiki.yandex-team.ru/passport/statbox#chtopishem
            self.statbox.log(
                action='account_created',
                track_id=self.track_id,
                uid=self.account.uid,
                login=self.account.normalized_login,
                karma=self.account.karma.value,
                password_quality=self.form_values['quality'],
                user_agent=self.user_agent,
                retpath=track.retpath,
                country=self.form_values['country'],
                language=track.language,
            )

            self.response_values.update(uid=self.account.uid)


class AccountRegisterAlternativeEasyViewV2(
    BundlePhoneMixin,
    BundlePasswordValidationMixin,
    BundleFrodoMixin,
    BaseBundleView,
    BundleCleanWebMixin,
    SecurePhoneBindAndAliasifyMixin,
):
    """
    Для Я.Денег требуется регистрация без ФИО

     * Привязывает цифровой алиас, если попросили
     * Не игнорирует ошибки YaSMS

    **Более того** валидация телефона производится на их стороне
    и нам надо принимать телефон как провалидированный
    """

    basic_form = BaseAccountRegisterAlternativeForm

    required_headers = [HEADER_CLIENT_USER_AGENT, HEADER_CONSUMER_CLIENT_IP]

    # Опасный грант - ТОЛЬКО для Я.Денег
    required_grants = ['account.register_alternative_easy']

    require_track = True

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    @cached_property
    def is_captcha_validation(self):
        return self.form_values['validation_method'] == CAPTCHA_VALIDATION_METHOD

    @cached_property
    def is_phone_validation(self):
        return self.form_values['validation_method'] == PHONE_VALIDATION_METHOD

    def is_verified_by_captcha(self):
        return (
            self.track.is_captcha_checked and
            self.track.is_captcha_recognized
        )

    def is_user_verified(self):
        if self.is_captcha_validation:
            return self.is_verified_by_captcha()
        elif self.is_phone_validation:
            # NOTE: Мы просто верим в то, что номер провалидирован на стороне ЯД
            return True

    def process_specific_form(self):
        """
        Это "Маскарад" - подменяем основную форму и снова обрабатываем параметры
        Проверяем входные параметры через одну из возможных форм
        * с валидацией через капчу - требуем ФИО и КВ/КО
        * с валидацией телефона по sms - без ФИО и КВ/КО (easy)
        """
        if self.is_captcha_validation:
            self.basic_form = AccountRegisterWithHintForm
        elif self.is_phone_validation:
            self.basic_form = AccountRegisterEasyForm

        self.process_basic_form()

    def check_state(self):
        if self.track.is_successful_registered:
            raise AccountAlreadyRegisteredError()

    def extra_validate(self):
        is_eula_accepted = self.form_values['eula_accepted']
        if not is_eula_accepted:
            raise EulaIsNotAcceptedError()

        phone_number = self.form_values['phone_number'].e164 if self.is_phone_validation else None
        self.validate_password(
            login=self.form_values['login'],
            phone_number=phone_number,
            emails=get_default_native_emails(self.form_values['login']),
        )

    def register_account(self):
        new_account = default_account(
            self.form_values['login'],
            datetime.now(),
            self.form_values,
            build_default_person_registration_info(self.client_ip),
        )
        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }

        if self.is_phone_validation:
            phone_number = self.form_values['phone_number']
            if self.form_values['create_phone_alias']:
                new_account.emails = self.build_emails(phone_number, new_account.registration_datetime)
            save_secure_phone = self.build_save_secure_phone(
                aliasify=self.form_values['create_phone_alias'],
                language=self.form_values['language'],
                account=new_account,
                phone_number=phone_number,
                should_ignore_binding_limit=False,
                is_new_account=True,
            )
            save_secure_phone.submit()

        with CREATE(new_account, self.request.env, events) as self.account:
            if self.is_phone_validation:
                save_secure_phone.commit()
            add_subscription(self.account, get_service(slug='mail'))

            if self.is_captcha_validation:
                self.account.hint.question = make_hint_question(
                    question=self.form_values['hint_question'],
                    question_id=self.form_values['hint_question_id'],
                    display_language=self.form_values['language'],
                )
                self.account.hint.answer = self.form_values['hint_answer']
        self.statbox.bind_context(uid=self.account.uid)
        if self.is_phone_validation:
            save_secure_phone.after_commit()
            if new_account.phonenumber_alias:
                self.send_greeting_email()

    def log_statbox(self, with_phone=False):
        if with_phone:
            self.statbox.bind(
                aliasify=self.form_values['create_phone_alias'],
            )
        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.normalized_login,
            karma=self.account.karma.value,
            password_quality=self.form_values['quality'],
            is_suggested_login=self.account.login in self.track.suggested_logins.get(),
            suggest_generation_number=self.track.suggest_login_count.get(default=0),
            country=self.form_values['country'],
            retpath=self.track.retpath,
        )

    def make_response(self):
        self.response_values['uid'] = self.account.uid

    def send_greeting_email(self):
        RegisterPhoneAliasManager(
            consumer=self.consumer,
            environment=self.request.env,
        ).send_mail_about_alias_as_login_and_email_enabled(
            account=self.account,
            language=self.form_values['language'],
            phone_number=self.account.phonenumber_alias.number,
        )

    def process_request(self):
        self.statbox.log(
            action='submitted',
        )

        self.process_basic_form()
        self.process_specific_form()
        self.clean_web_check_form_values()

        self.read_track()

        self.check_state()

        if not self.is_user_verified():
            raise UserNotVerifiedError()

        self.extra_validate()

        with self.track_transaction.commit_on_error():
            self.register_account()

            self.track.is_successful_registered = True
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=True,
                password_passed=True,
                session_scope=SessionScope.xsession,
            )
            self.track.user_entered_login = self.form_values['login']

            frodo_action = 'alternative_hint' if self.is_captcha_validation else 'alternative_phone'
            self.frodo_check_spammer(frodo_action, increment_counters=True)
            self.log_statbox(with_phone=self.is_phone_validation)

            self.make_response()

            process_env_profile(self.account, track=self.track)


class ShowLimits(BaseBundleView):
    """
    Отвечает на вопросы:
     * Сколько раз еще можно отправить СМС с кодом проверки на телефонный номер пользователя?
        Проверяет достижение лимитов на количество смс по отдельному треку и по отдельному ip-адресу.
        Указывает число смс, которое еще можно отправить до наступления блокировки.
     * Сколько раз еще можно ввести код валидации телефона?
    """

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['account.register_limits']

    require_track = True

    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            action='registration_limit_exceeded',
            limit='registration_with_sms_per_ip',
            track_id=self.track.track_id,
            user_ip=self.client_ip,
        )

    def get_sms_send_possible_count(self):
        """
        Проверяем флаги в треке и счетчики отправленных СМС:
         * Счетчик отправок смс с кодом проверки для этого ip-адреса в процессе регистрации
        @return: Сколько еще раз можно отправить смс для этого пользователя.
        """
        if sms_per_ip_for_consumer.exists(self.consumer):
            global_counter = sms_per_ip_for_consumer.get_counter(self.consumer)
        else:
            global_counter = sms_per_ip.get_counter(user_ip=self.client_ip)
        counter = sms_per_ip.get_registration_completed_with_phone_counter(user_ip=self.client_ip)

        if (self.track.phone_confirmation_send_count_limit_reached or
                self.track.phone_confirmation_send_ip_limit_reached):
            # Если пользователь уже помечен как достигший лимиты, ничего дополнительно не делаем
            result = 0

        elif counter.hit_limit_by_ip(self.client_ip):
            # Если пользователь достиг лимит отправки, запишем в statbox
            result = 0
            self.report_exceeded_limit(counter)

        else:
            # Расчитаем сколько раз еще можно отправить sms
            remained_sms_for_track = (settings.SMS_VALIDATION_MAX_SMS_COUNT -
                                      self.track.phone_confirmation_sms_count.get(default=0))
            remained_sms_for_ip = (counter.limit - counter.get(self.client_ip))
            remained_sms_for_ip_global = (global_counter.limit - global_counter.get(self.client_ip))

            result = min(remained_sms_for_track, remained_sms_for_ip, remained_sms_for_ip_global)
            result = result if result > 0 else 0

        return result

    def report_exceeded_limit(self, counter):
        self.statbox.log(
            counter_current_value=counter.get(self.client_ip),
            counter_limit_value=counter.limit,
        )

    def get_confirmation_possible_count(self):
        """
        Проверяем в треке счетчики подтверждения телефона(ввод кода подтверждения)
        @return: Сколько еще раз можно ввести код подтверждения телефона.
        """
        if self.track.phone_confirmation_confirms_count_limit_reached:
            result = 0
        else:
            result = (settings.SMS_VALIDATION_MAX_CHECKS_COUNT -
                      self.track.phone_confirmation_confirms_count.get(default=0))

        return result if result > 0 else 0

    def process_request(self):
        self.read_track()

        sms_remained = self.get_sms_send_possible_count()
        confirmation_remained = self.get_confirmation_possible_count()

        self.response_values = {
            'sms_remained_count': sms_remained,
            'confirmation_remained_count': confirmation_remained,
        }


class AccountRegisterIntranet(BaseBundleView,
                              EmailValidatorMixin):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['account.register_intranet']

    require_track = False

    basic_form = AccountRegisterIntranetForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_intranet',
            track_id=self.track_id,
        )

    def register_account(self):
        new_account = default_account(
            self.form_values['login'],
            datetime.now(),
            self.form_values,
            build_default_person_registration_info(self.client_ip),
        )
        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }

        external_email = self.form_values['external_email']
        with CREATE(new_account, self.request.env, events) as self.account:
            set_impossible_password(self.account)
            add_subscription(self.account, get_service(slug='mail'))
            if self.form_values['is_maillist']:
                # указываем, что аккаунт является рассылкой
                self.account.is_maillist = True
            else:
                # указываем, что аккаунт является сотрудником
                self.account.is_employee = True
            if external_email:
                self.account.default_email = self.form_values['external_email']
                confirmed_email = self.create_confirmed_email(self.form_values['external_email'])
                self.account.emails.add(confirmed_email)
            if self.form_values['altdomain_alias']:
                self.account.altdomain_alias = AltDomainAlias(
                    parent=self.account,
                    login=self.form_values['altdomain_alias'],
                )
            if self.form_values['firstname_global']:
                self.account.person.firstname_global = self.form_values['firstname_global']
            if self.form_values['lastname_global']:
                self.account.person.lastname_global = self.form_values['lastname_global']

    def process_request(self):
        self.process_basic_form()
        self.read_or_create_track('register')
        self.statbox.log(action='submitted')

        self.register_account()

        # http://wiki.yandex-team.ru/passport/statbox#chtopishem
        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.normalized_login,
        )

        self.response_values.update(uid=self.account.uid)


class AccountRegisterDirectory(BundleFrodoMixin,
                               BundlePasswordValidationMixin,
                               BundleCleanWebMixin,
                               BundlePhoneMixin,
                               BaseBundleView):
    """
    Создание ПДД-домена и ПДД-пользователя для Яндекс.Директории. Пользователь
    сразу же становится администратором домена.
    """
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    required_grants = ['account.register_directory']

    require_track = True

    basic_form = AccountRegisterDirectoryForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_register_directory',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.read_track()
        self.process_basic_form()
        self.clean_web_check_form_values()

        eula_accepted = self.form_values['eula_accepted']
        human_readable_domain = self.form_values['domain'].lower()
        registration_ts = datetime.now()

        new_login = '%s@%s' % (self.form_values['login'], human_readable_domain)

        self.statbox.log(
            action='submitted',
            consumer=self.consumer,
        )

        if not is_domain_suitable_for_directory(human_readable_domain):
            raise DomainInvalidTypeError()

        # Если пользователь не принял ПД, то ничего не делаем.
        if not eula_accepted:
            raise EulaIsNotAcceptedError()

        if self.track.is_successful_registered:
            raise AccountAlreadyRegisteredError()

        if not self.is_phone_confirmed_in_track(allow_by_flash_call=True):
            raise UserNotVerifiedError()

        # Проверяем, что регистрируемый домен ещё не занят
        if does_domain_belong_to_pdd(human_readable_domain):
            raise DomainAlreadyExistsError()

        validated_password, password_quality = self.validate_password(
            uid=None,
            login=self.form_values['login'],
            emails={new_login},
        )

        log.debug('Creating domain "%s" with placeholder admin...' % self.form_values['domain'])
        new_domain = Domain()
        with CREATE(new_domain, self.request.env, {'action': 'domain_create_directory', 'consumer': self.consumer}):
            new_domain.domain = human_readable_domain
            # Костыль из-за проблемы курицы и яица: мы ещё не знаем здесь
            # UID, т.к. для его регистрации нужен создаваемый домен.
            # Поэтому используем здесь несуществующий UID ПДДшника.
            new_domain.admin_uid = 1130000000000000
            new_domain.is_yandex_mx = True
            new_domain.is_enabled = True
            new_domain.registration_datetime = registration_ts
            new_domain.master_domain = None
            new_domain.default_uid = 0
            new_domain.organization_name = self.form_values['organization'] or human_readable_domain

        admin_account = default_account(
            new_login,
            registration_ts,
            {
                'lastname': self.form_values['lastname'],
                'firstname': self.form_values['firstname'],
                'password': validated_password,
                'quality': password_quality,
            },
            build_default_person_registration_info(self.client_ip),
            alias_type='pdd',
        )

        # По предварительному соглашению с Яндекс.Директорией, админом домена будет ПДДшник.
        # Стоит помнить, что это может вызвать всевозможные косяки в неожиданных местах
        # (к примеру, при удалении домена).
        log.debug('Registering domain administrator account...')
        with CREATE(admin_account, self.request.env, {'action': 'account_create_directory', 'consumer': self.consumer}):
            admin_account.domain = new_domain
            admin_account.is_connect_admin = True
            # Продолжаем подписывать на сид ПДД-админа - на это завязаны костыли в mdapi - Коннект может
            # создавать алиасы для своих доменов через mdapi
            admin_account.is_pdd_admin = True
            admin_account.is_pdd_agreement_accepted = True
            add_subscription(
                admin_account,
                get_service(slug='mail'),
            )

            phone_number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)
            save_secure_phone = self.build_save_secure_phone(
                account=admin_account,
                phone_number=phone_number,
                is_new_account=True,
                should_ignore_binding_limit=False,
            )
            save_secure_phone.submit()
            save_secure_phone.commit()
        self.account = admin_account

        self.statbox.bind_context(
            uid=admin_account.uid,
            login=admin_account.machine_readable_login,
        )

        self.statbox.log(
            action='account_created',
            country=admin_account.person.country,
            retpath=self.track.retpath,
            uid=admin_account.uid,
            domain=new_domain.punycode_domain,
        )

        save_secure_phone.after_commit()

        # После того, как мы создали аккаунт, наконец-то сделаем его
        # администратором домена.
        log.debug(
            'Assigning uid=%s to %s as domain administrator...' % (
                admin_account.uid,
                new_domain.domain,
            ),
        )
        with UPDATE(new_domain, self.request.env, {'action': 'account_create_directory', 'consumer': self.consumer}):
            new_domain.admin_uid = admin_account.uid

        self.frodo_check_spammer(
            action='directoryreg',
            increment_counters=True,
        )

        with self.track_transaction.rollback_on_error():
            # После регистрации.
            self.track.is_successful_registered = True
            set_authorization_track_fields(
                admin_account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                password_passed=True,
                session_scope=SessionScope.xsession,
            )

        process_env_profile(self.account, track=self.track)
        incr_registration_sms_per_ip_counter(self.request.env)

        self.response_values = {
            'uid': admin_account.uid,
            'domain': new_domain.punycode_domain,
            'track_id': self.track.track_id,
        }


class AccountRegisterPdd(BundlePasswordValidationMixin,
                         BundleCleanWebMixin,
                         BundleFederalMixin,
                         BaseBundleView,
                         ):
    """
    Создание ПДД-пользователя.
    """

    required_grants = ['account.register_pdd']

    basic_form = AccountRegisterPddForm

    @cached_property
    def statbox(self):
        return StatboxLogger(mode='account_register_pdd')

    def calculate_password_quality(self, login, password):
        # Вычислим качество пароля отдельно, т.к. пароль не валидируется в ручке.
        words = [login]
        qualifier = get_password_qualifier()

        # Отфильтруем символы, не поддерживаемые функцией вычисления качества пароля.
        filtered_password = filter(lambda char: ord(char) < 256, password)
        quality = qualifier.get_quality(
            filtered_password,
            words=words,
            subwords=words,
        )
        return quality['quality']

    def process_request(self):
        self.create_track('register')
        self.process_basic_form()
        self.clean_web_check_form_values()

        self.statbox.log(
            action='submitted',
            consumer=self.consumer,
        )

        # TODO: сountry и language из формы или фоллбэк на заголовок

        login = self.form_values['login']
        password = self.form_values['password']
        domain_name = self.form_values['domain']
        is_enabled = self.form_values['is_enabled']
        is_maillist = self.form_values['is_maillist']
        no_password = self.form_values['no_password']
        hint_answer = self.form_values['hint_answer']
        hint_question = self.form_values['hint_question']
        password_hash_version, password_hash = None, None
        if self.form_values['password_hash']:
            password_hash_version, password_hash = self.form_values['password_hash']
        weak_password_ok = self.form_values['weak_password']
        is_creating_required = self.form_values['is_creating_required']
        pdd_login = '%s@%s' % (login, domain_name)

        domain_info = self.blackbox.hosted_domains(domain=domain_name)
        if not domain_info['hosted_domains']:
            raise DomainNotFoundError()

        domain = Domain().parse(domain_info)
        if domain.master_domain or (not is_maillist and self.get_saml_settings(domain_id=domain.id, only_enabled=True) is not None):
            raise DomainInvalidTypeError()

        # TODO: Когда-нибудь написать достаточно замороченный валидатор,
        # который избавит от необходимости проверять заданность одного
        # из нескольких полей по значению в другом поле.
        if not no_password and not (password or password_hash):
            raise ValidationFailedError([
                'password.empty',
                'password_hash.empty',
            ])

        if not password_hash and not no_password:
            try:
                password, password_quality = self.validate_password(
                    login=pdd_login,
                    emails=set([pdd_login]),
                )
            except ValidationFailedError:
                if not weak_password_ok:
                    raise
                is_creating_required = True
                password_quality = self.calculate_password_quality(pdd_login, password)

        account_args = {
            'is_creating_required': is_creating_required,
        }
        person_args = {
            'firstname': self.form_values['firstname'],
            'lastname': self.form_values['lastname'],
            'gender': self.form_values['gender'],
            'birthday': self.form_values['birthday'],
            'language': self.form_values['language'],
            'country': self.form_values['country'],
        }

        new_account = default_account(
            pdd_login,
            datetime.now(),
            dict(account_args, **person_args),
            build_default_person_registration_info(self.client_ip),
            alias_type='pdd',
        )

        events = {
            'action': 'account_create_mdapi',
            'consumer': self.consumer,
        }
        with CREATE(new_account, self.request.env, events):
            new_account.domain = domain
            new_account.is_enabled = is_enabled
            new_account.is_maillist = is_maillist
            add_subscription(
                new_account,
                get_service(slug='mail'),
            )

            if hint_question and hint_answer:
                question = make_hint_question(question=hint_question)
                new_account.hint.question = question
                new_account.hint.answer = hint_answer

            if no_password:
                pass
            elif password_hash:
                new_account.password.set_hash(password_hash, version=password_hash_version)
            else:
                set_password_with_experiment(
                    new_account,
                    password,
                    password_quality,
                    login=new_account.login,
                )

            if self.form_values['with_yambot_alias']:
                yambot_alias = build_available_yambot_login(
                    settings.YAMBOT_LOGIN_GENERATION_RETRIES,
                    self.request.env,
                )
                new_account.yambot_alias = YambotAlias(
                    parent=new_account,
                    alias=yambot_alias,
                )

        self.statbox.log(
            action='account_created',
            is_enabled=new_account.is_enabled,
            retpath=self.track.retpath,
            uid=new_account.uid,
            login=new_account.login,
            domain=domain.punycode_domain,
            no_password=no_password is not None,
            password_hash=bool(password_hash),
        )

        self.response_values['uid'] = new_account.uid


class AccountCreate(BaseBundleView,
                    EmailValidatorMixin,
                    BundlePasswordValidationMixin,
                    BundleFrodoMixin,
                    BundleCleanWebMixin,
                    BundlePersistentTrackMixin):

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    required_grants = ['account.create']

    require_track = False

    basic_form = AccountCreateForm

    event_action = 'account_create'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_create',
            user_agent=self.user_agent,
        )

    def check_state(self):
        if self.track.is_successful_registered:
            raise AccountAlreadyRegisteredError()

    def register_account(self):
        new_account = default_account(
            self.form_values['login'],
            datetime.now(),
            self.form_values,
            build_default_person_registration_info(self.client_ip),
        )
        events = {
            'action': self.event_action,
            'consumer': self.consumer,
        }
        with CREATE(new_account, self.request.env, events) as self.account:
            if self.form_values['yastaff_login']:
                yastaff_service = get_service(slug='yastaff')
                add_subscription(self.account, yastaff_service, login=self.form_values['yastaff_login'])

            if self.form_values['subscriptions']:
                for service in self.form_values['subscriptions']:
                    add_subscription(self.account, service)

            if self.form_values['email']:
                confirmed_email = self.create_confirmed_email(self.form_values['email'])
                self.account.emails.add(confirmed_email)

            # Всегда подписываем на почту
            add_subscription(self.account, get_service(slug='mail'))

        self.statbox.bind_context(uid=self.account.uid)

    def send_email_notification(self, confirmed_email, persistent_track):
        template_name, mail_info, context = get_auth_by_key_link_email_data(
            self.account,
            persistent_track,
        )
        send_mail_for_account(
            template_name=template_name,
            mail_info=mail_info,
            context=context,
            account=self.account,
            context_shadower=login_shadower,
            send_to_external=False,
            send_to_native=False,
            specific_email=confirmed_email,
        )

    def process_request(self):
        self.statbox.log(action='submitted')

        self.process_basic_form()

        if self.form_values['ignore_stoplist']:
            self.check_grant(IGNORE_STOPLIST_GRANT)

        if not self.form_values['is_enabled']:
            self.check_grant(ACCOUNT_IS_ENABLED_GRANT)

        if self.form_values['email']:
            self.check_grant(ACCOUNT_RECOVERY_EMAIL_GRANT)

        services = self.form_values['subscriptions']
        if services is not None:
            grants_list = []
            for service in services:
                if service is not None:
                    grants_list.extend(generate_subscription_grant([CREATE_GRANT], service))

            self.check_grant(lambda: grants_list)

        self.read_or_create_track('register')

        self.statbox.bind_context(track_id=self.track_id)
        self.clean_web_check_form_values()

        self.check_state()

        with self.track_transaction.rollback_on_error():
            is_password_passed = bool(self.form_values['password'])

            if is_password_passed:
                self.validate_password(
                    uid='',
                    login=self.form_values['login'],
                    emails=set([self.form_values['email']]) if self.form_values['email'] is not None else None,
                )
            else:
                self.form_values['is_creating_required'] = True

            self.register_account()

            if not is_password_passed:
                persistent_track = self.create_persistent_track(
                    self.account.uid,
                    TRACK_TYPE_AUTH_BY_KEY_LINK,
                    expires_after=settings.AUTH_BY_KEY_LINK_LIFETIME_SECONDS,
                )
                self.send_email_notification(
                    confirmed_email=self.form_values['email'],
                    persistent_track=persistent_track,
                )

            self.track.is_successful_registered = True
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=False,
                allow_create_token=False,
                password_passed=True,
                session_scope=SessionScope.xsession,
            )

            self.frodo_check_spammer(
                action='admreg',
                increment_counters=False,
            )

            self.statbox.log(
                action='account_created',
                country=self.form_values['country'],
                karma=self.account.karma.value,
                login=self.account.normalized_login,
                password_quality=self.form_values.get('quality'),
            )
            process_env_profile(self.account, track=self.track)

            self.response_values.update(uid=self.account.uid)


class AccountRegisterYambot(BaseBundleView):

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    required_grants = ['account.register_yambot']
    require_track = False

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_register_yambot',
        )

    def register_account(self):
        alias = build_available_yambot_login(
            settings.YAMBOT_LOGIN_GENERATION_RETRIES,
            self.request.env,
        )
        new_account = yambot_account(
            alias=alias,
            registration_datetime=datetime.now(),
        )
        events = {
            'action': 'account_register_yambot',
            'consumer': self.consumer,
        }
        with CREATE(new_account, self.request.env, events) as self.account:
            pass

        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.login,
        )

    def issue_oauth_token(self):
        client_id = settings.OAUTH_APPLICATION_YAMB['client_id']
        client_secret = settings.OAUTH_APPLICATION_YAMB['client_secret']
        response = authorize_oauth(
            client_id=client_id,
            client_secret=client_secret,
            env=self.request.env,
            uid=self.account.uid,
        )
        if 'error' in response:
            raise OAuthUnavailableError(response['error'], response.get('description'))
        else:
            self.statbox.log(
                action='token_created',
                uid=self.account.uid,
                client_id=client_id,
            )
            return response['access_token']

    def process_request(self):
        self.statbox.log(action='submitted')

        self.register_account()
        token = self.issue_oauth_token()

        self.response_values.update(
            uid=self.account.uid,
            oauth_token=token,
        )


class AccountRegisterKolonkish(BaseBundleView, BundleAccountGetterMixin):
    basic_form = AccountRegisterKolonkishForm
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    required_grants = ['account.register_kolonkish']
    require_track = False

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_register_kolonkish',
        )

    def check_counters(self):
        short_term_counter = register_kolonkish_per_creator_uid.get_short_term_counter()
        long_term_counter = register_kolonkish_per_creator_uid.get_long_term_counter()

        creator_uid = self.account.uid

        if short_term_counter.hit_limit(creator_uid) or long_term_counter.hit_limit(creator_uid):
            raise RateLimitExceedError()

    def incr_counters(self):
        short_term_counter = register_kolonkish_per_creator_uid.get_short_term_counter()
        long_term_counter = register_kolonkish_per_creator_uid.get_long_term_counter()

        creator_uid = self.account.uid

        short_term_counter.incr(creator_uid)
        long_term_counter.incr(creator_uid)

    def register_account(self):
        alias = build_available_kolonkish_login(
            settings.KOLONKISH_LOGIN_GENERATION_RETRIES,
            self.request.env,
        )
        new_account = kolonkish_account(
            alias=alias,
            registration_datetime=datetime.now(),
            creator_uid=self.account.uid,
        )
        events = {
            'action': 'account_register_kolonkish',
            'consumer': self.consumer,
        }
        with CREATE(new_account, self.request.env, events) as new_account:
            if self.form_values['display_name']:
                new_account.person.display_name = self.form_values['display_name']

        self.statbox.log(
            action='account_created',
            uid=new_account.uid,
            login=new_account.login,
        )
        return new_account

    def issue_oauth_authorization_code(self, new_account):
        client_id = settings.OAUTH_APPLICATION_KOLONKA['client_id']
        client_secret = settings.OAUTH_APPLICATION_KOLONKA['client_secret']

        oauth = get_oauth()
        response = oauth.issue_authorization_code(
            code_strength='long',
            client_id=client_id,
            client_secret=client_secret,
            require_activation=False,
            uid=new_account.uid,
            by_uid=True,
            ttl=settings.OAUTH_KOLONKA_CODE_TTL,
            device_id=self.form_values['device_id'],
            device_name=self.form_values['device_name'],
            user_ip=self.client_ip,
        )

        if response['status'] == 'error':
            raise OAuthUnavailableError(response['errors'])
        else:
            self.statbox.log(
                action='oauth_code_issued',
                uid=new_account.uid,
                client_id=client_id,
                device_id=self.form_values['device_id'],
                device_name=self.form_values['device_name'],
            )
            return response['code']

    def check_creator_account_type(self):
        acceptable_account_type = (
            self.account.is_normal or
            self.account.is_lite or
            self.account.is_social or
            self.account.is_pdd or
            self.account.is_neophonish
        )
        if not acceptable_account_type:
            raise AccountInvalidTypeError()

    def process_request(self):
        self.statbox.log(action='submitted')
        self.process_basic_form()

        self.get_account_from_session_or_oauth_token(required_scope=settings.OAUTH_KOLONKA_KOLONKISH_SCOPE)
        self.check_creator_account_type()

        self.check_counters()

        new_account = self.register_account()
        code = self.issue_oauth_authorization_code(new_account)

        self.incr_counters()

        self.response_values.update(
            uid=new_account.uid,
            login=new_account.login,
            code=code,
        )


class AccountRegisterLiteMixin(object):
    def get_email_from_social_profile(self):
        if not self.track.social_task_data:
            return

        default_dict = dict()
        profile = self.track.social_task_data.get('profile', default_dict)

        provider = profile.get('provider', default_dict)
        if provider.get('code') not in settings.AUTH_ALLOWED_PROVIDERS:
            log.debug('Social provider not authorized to provide e-mail: %s' % provider.get('code'))
            return

        if profile.get('email'):
            try:
                return validators.LiteLogin().to_python(profile.get('email'))
            except validators.Invalid as e:
                log.debug('Social profile e-mail invalid: %s' % profile.get('email'))
                raise ValidationFailedError.from_invalid(e, field='login')

    def check_login_available(self):
        try:
            validators.Availability(ACCOUNT_TYPE_LITE).to_python(
                dict(login=self.email),
                validators.State(self.request.env),
            )
        except validators.Invalid as e:
            raise ValidationFailedError.from_invalid(e)


class AccountRegisterLiteSubmit(
    AccountRegisterLiteMixin,
    SendRegistrationConfirmationEmailView,
):
    basic_form = AccountRegisterLiteSubmitForm

    def __init__(self):
        super(AccountRegisterLiteSubmit, self).__init__()

        # E-Mail для которого будет создан аккаунт
        self.email = None

    def check_if_email_is_confirmed(self):
        if (
            self.track.email_confirmation_address == self.email and
            self.track.email_confirmation_passed_at
        ):
            raise EmailAlreadyConfirmedError(
                address=self.track.email_confirmation_address,
                uid=None,
                code=None,
            )

    def is_sending_code_required(self):
        social_profile_email = self.get_email_from_social_profile()
        if social_profile_email and self.email == social_profile_email:
            return False

        # Эвристики/походы в Спамооборону, позволяющие понять,
        # можно ли доверять этому адресу без подтверждения
        return not settings.CONFIRM_EMAIL_WITHOUT_CODE_ENABLED

    def confirm_without_sending_email(self):
        self.statbox.log(
            address=mask_email_for_statbox(self.email),
            action='fake_confirmation_email',
            confirmation_checks_count=self.track.email_confirmation_checks_count.get(default=0),
        )
        with self.track_transaction.rollback_on_error():
            self.track.email_confirmation_address = self.email
            self.track.email_confirmation_passed_at = get_unixtime()

    def update_response_personal_data_from_social_profile(self):
        if not self.track.social_task_data:
            return

        default_dict = dict()

        profile = self.track.social_task_data.get('profile', default_dict)
        provider = profile.get('provider', default_dict)
        personal_data = dict(
            firstname=profile.get('firstname'),
            lastname=profile.get('lastname'),
            social_provider_code=provider.get('code'),
        )
        personal_data = remove_none_values(personal_data)

        for key, val in personal_data.items():
            self.response_values.setdefault(key, val)

    def process_request(self):
        if not settings.ALLOW_LITE_REGISTRATION:
            raise ActionImpossibleError()

        self.process_basic_form()
        self.read_track()

        self.email = self.form_values.get('login')
        if not self.email:
            self.email = self.get_email_from_social_profile()
            if not self.email:
                raise ValidationFailedError(['login.empty'])
            self.check_login_available()

        self.check_track_for_captcha()

        self.check_if_email_is_confirmed()

        self.email = self.email.lower()
        if does_domain_belong_to_pdd(self.email):
            raise DomainInvalidTypeError()

        code_required = self.is_sending_code_required()
        if code_required:
            self.process_send(self.email)
        elif settings.CONFIRM_EMAIL_WITHOUT_CODE_ENABLED:
            self.confirm_without_sending_email()
        self.response_values.update(code_required=code_required)

        self.response_values.update(email=self.email)
        self.update_response_personal_data_from_social_profile()


class AccountRegisterLiteCommit(
    AccountRegisterLiteMixin,
    BaseBundleView,
    EmailValidatorMixin,
    BundlePasswordValidationMixin,
    BundleFrodoMixin,
    BundleCleanWebMixin,
    MailSubscriptionsMixin,
):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )
    required_grants = ['account.register_require_confirmed_email']
    process_name = PROCESS_WEB_REGISTRATION
    require_process = True
    require_track = True
    basic_form = AccountRegisterLiteCommitForm

    def __init__(self):
        super(AccountRegisterLiteCommit, self).__init__()

        # E-Mail для которого будет создан аккаунт
        self.email = None

    def check_email_is_confirmed(self):
        if (
            self.email and
            self.track.email_confirmation_address == self.email and
            self.track.email_confirmation_passed_at
        ):
            return

        if self.email and self.email == self.get_email_from_social_profile():
            return

        raise UserNotVerifiedError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_lite',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.statbox.log(
            action='submitted',
        )
        self.process_basic_form()
        self.clean_web_check_form_values()
        self.read_track()

        if self.track.is_successful_registered:
            raise AccountAlreadyRegisteredError()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        self.email = self.form_values.get('login')
        if self.email is None:
            self.email = self.get_email_from_social_profile()
            if not self.email:
                raise ValidationFailedError(['login.empty'])
            self.check_login_available()

        self.check_email_is_confirmed()

        with self.track_transaction.rollback_on_error():
            if self.form_values.get('password') is not None:
                password, quality = self.validate_password(
                    uid='',
                    login=self.email,
                )
            else:
                password, quality = None, None

            new_account = default_account(
                self.email,
                datetime.now(),
                self.form_values,
                build_default_person_registration_info(self.client_ip),
                alias_type='lite',
            )

            events = {
                'action': 'account_register_lite',
                'consumer': self.consumer,
            }

            with CREATE(new_account, self.request.env, events) as self.account:
                confirmed_email = self.create_confirmed_email(self.email)
                self.account.emails.add(confirmed_email)
                self.unsubscribe_from_maillists_if_nessesary()

            self.track.is_successful_registered = True
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                password_passed=bool(password),
                session_scope=SessionScope.xsession,
            )
            self.track.user_entered_login = self.email

            self.frodo_check_spammer(action='email_reg', increment_counters=True)

            self.statbox.log(
                uid=self.account.uid,
                login=self.account.normalized_login,
                action='account_created',
                karma=self.account.karma.value,
                country=self.form_values['country'],
                language=self.form_values['language'],
                password_quality=quality,
                process_uuid=self.track.process_uuid,
                user_agent=self.user_agent,
            )

            process_env_profile(self.account, track=self.track)

            self.response_values.update(uid=self.account.uid)


class AccountRegisterNeophonish(BaseBundleView,
                                BundlePhoneMixin,
                                BundleNeophonishMixin,
                                BundleCleanWebMixin,
                                MailSubscriptionsMixin):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )
    required_grants = ['account.register_neophonish']
    require_track = True
    basic_form = AccountRegisterNeophonishForm

    def check_phone_is_confirmed(self):
        if not (
            self.is_phone_confirmed_in_track(allow_by_flash_call=True) and
            self.track.phone_confirmation_phone_number
        ):
            raise UserNotVerifiedError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_neophonish',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.statbox.log(
            action='submitted',
        )

        if not settings.ALLOW_NEOPHONISH_REGISTRATION:
            raise ActionImpossibleError()

        self.process_basic_form()
        self.clean_web_check_form_values()
        self.read_track()

        if self.track.is_successful_registered:
            raise AccountAlreadyRegisteredError()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        self.check_phone_is_confirmed()

        is_name_defined = bool(self.form_values['firstname'] and self.form_values['lastname'])
        # сценарий без ФИО разрешён только если номер мониторится Сквоттером
        start_phone_tracking_in_squatter = (
            settings.USE_NEW_SUGGEST_BY_PHONE and
            settings.USE_PHONE_SQUATTER and
            not is_name_defined
        )

        with self.track_transaction.rollback_on_error():
            self.register_neophonish(
                person_args=self.form_values,
                start_phone_tracking_in_squatter=start_phone_tracking_in_squatter,
                ignore_squatter_errors=settings.PHONE_SQUATTER_DRY_RUN,
            )

            self.track.is_successful_registered = True
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                password_passed=True,
                session_scope=SessionScope.xsession,
            )

            self.statbox.log(
                uid=self.account.uid,
                login=self.account.normalized_login,
                action='account_created',
                user_agent=self.user_agent,
            )
            self.send_neophonish_registration_notification_to_messenger()

            process_env_profile(self.account, track=self.track)

            self.response_values.update(uid=self.account.uid)


class AccountRegisterScholar(
    BundleScholarMixin,
    BaseBundleView,
):
    basic_form = AccountRegisterScholarForm
    required_grants = ['account.register_scholar']
    required_headers = (
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    def process_request(self):
        self.statbox.log(action='submitted')
        self.process_basic_form()
        self.validate_scholar_password()

        self.register_scholar()

        self.statbox.log(
            action='account_created',
            login=self.account.normalized_login,
            uid=self.account.uid,
        )

        self.response_values.update(
            firstname=self.account.person.firstname,
            lastname=self.account.person.lastname,
            login=self.account.login,
            uid=self.account.uid,
        )

    def register_scholar(self):
        args = dict(self.form_values)
        password = args.pop('password', None)

        self.account = default_account(
            alias_type='scholar',
            args=args,
            default_person_info=build_default_person_registration_info(self.client_ip),
            login=self.form_values['login'],
            registration_datetime=datetime.now(),
        )

        with CREATE(
            environment=self.request.env,
            events=dict(action='account_register_scholar', consumer=self.consumer),
            model=self.account,
        ):
            self.set_scholar_password(password)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            mode='account_register_scholar',
            user_agent=self.user_agent,
        )


class AccountRegisterFederal(
    BundleFederalMixin,
    BaseBundleView,
):
    basic_form = AccountRegisterFederalForm
    required_grants = ['account.register_federal']
    required_headers = (
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    def process_request(self):
        self.statbox.log(action='submitted')
        self.process_basic_form()

        domain_info = self.blackbox.hosted_domains(domain_id=self.form_values['domain_id'])
        if not domain_info['hosted_domains']:
            raise DomainNotFoundError()
        domain = Domain().parse(domain_info)

        availability_info = self.blackbox.userinfo(
            login=self.make_alias(self.form_values['name_id'], domain).lower(),
            sid='federal',
        )
        is_user_exist = availability_info['uid'] is not None

        if is_user_exist:
            raise AccountAlreadyRegisteredError()

        self.register_federal(
            domain=domain,
            name_id=self.form_values['name_id'],
            email=self.form_values['email'],
            firstname=self.form_values['firstname'],
            lastname=self.form_values['lastname'],
            is_enabled=self.form_values['active'],
            display_name=self.form_values['display_name'],
        )

        self.statbox.log(
            action='account_created',
            login=self.account.normalized_login,
            uid=self.account.uid,
        )

        self.response_values.update(
            uid=self.account.uid,
        )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            mode='account_register_federal',
            user_agent=self.user_agent,
        )


__all__ = (
    'AccountCreate',
    'AccountRegisterPhoneAndAlisify',
    'AccountRegisterByMiddleman',
    'AccountRegisterAlternativeEasyViewV2',
    'AccountRegisterFederal',
    'AccountRegisterKolonkish',
    'AccountRegisterLiteCommit',
    'AccountRegisterLiteSubmit',
    'AccountRegisterNeophonish',
    'AccountRegisterIntranet',
    'AccountRegisterPdd',
    'AccountRegisterScholar',
    'AccountRegisterYambot',
    'ShowLimits',
)
