# -*- coding: utf-8 -*-
from passport.backend.api.forms.base import DeviceInfoForm
from passport.backend.api.forms.register import (
    AccountRegisterRequireConfirmedPhone,
    CAPTCHA_VALIDATION_METHOD,
    PHONE_VALIDATION_METHOD,
)
from passport.backend.api.validators import AvailabilityOnSid
from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.models.password import (
    PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON,
    PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON,
)
from passport.backend.core.services import Service
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_SCHOLAR,
)


class SendRegistrationConfirmationCodeForm(validators.Schema):
    email = validators.ComprehensiveEmailValidator(allow_native=False)
    language = validators.PortalLanguage(not_empty=True)


class ConfirmEmailForm(validators.Schema):
    key = validators.String(not_empty=True)


class AccountRegisterRequireConfirmedPhoneAndAliasify(AccountRegisterRequireConfirmedPhone):
    password = validators.String(not_empty=True)

    chained_validators = [
        validators.Availability(),
    ]


class AccountRegisterByMiddlemanForm(validators.Schema):
    login = validators.Login()
    password = validators.String(not_empty=True)

    firstname = validators.FirstName(if_missing=None)
    lastname = validators.LastName(if_missing=None)
    force_clean_web = validators.StringBool(if_missing=False)

    language = validators.PortalLanguage()
    country = validators.CountryCode()

    chained_validators = [
        validators.Availability(),
    ]


class BaseAccountRegisterAlternativeForm(validators.Schema):
    validation_method = validators.OneOf(
        [PHONE_VALIDATION_METHOD, CAPTCHA_VALIDATION_METHOD],
        not_empty=True,
    )


class AccountRegisterWithHintForm(BaseAccountRegisterAlternativeForm):
    login = validators.Login()
    password = validators.String(not_empty=True)

    firstname = validators.FirstName()
    lastname = validators.LastName()

    hint_question_id = validators.HintQuestionId()
    hint_question = validators.HintQuestion()
    hint_answer = validators.HintAnswer()

    language = validators.PortalLanguage()
    country = validators.CountryCode()
    gender = validators.Gender(if_missing=None)
    birthday = validators.Birthday(if_missing=None)
    timezone = validators.Timezone(if_missing=None)

    eula_accepted = validators.StringBool(not_empty=True, strip=True)

    chained_validators = [
        validators.Availability(),
        validators.RequireSet(
            allowed_sets=[
                ['hint_question_id', 'hint_answer'],
                ['hint_question', 'hint_answer'],
            ],
        ),
    ]


class AccountRegisterEasyForm(BaseAccountRegisterAlternativeForm):
    """
    Для Я.Денег нужна регистрация
    * без обязательных полей ФИО
    * с провалидированным на их стороне телефонным номером
    * как следствие, не обязательны КВ/КО
    """
    login = validators.Login()
    password = validators.String(not_empty=True)

    firstname = validators.FirstName(if_missing=None)
    lastname = validators.LastName(if_missing=None)

    phone_number = validators.String(not_empty=True)
    create_phone_alias = validators.StringBool(if_missing=False, strip=True)  # признак того, что нужен цифровой алиас

    language = validators.PortalLanguage()
    country = validators.CountryCode()
    gender = validators.Gender(if_missing=None)
    birthday = validators.Birthday(if_missing=None)
    timezone = validators.Timezone(if_missing=None)

    eula_accepted = validators.StringBool(not_empty=True, strip=True)

    chained_validators = [
        validators.Availability(),
        validators.PhoneNumber(),
    ]


class MD5Hash(validators.Regex):
    regex = '^[a-fA-F0-9]{32}$'


class AccountRegisterIntranetForm(validators.Schema):
    login = validators.YandexTeamLogin()
    is_maillist = validators.StringBool(if_missing=False)
    external_email = validators.SimpleEmailValidator(if_missing=None)
    altdomain_alias = validators.AltDomainAliasValidator(
        allowed_domains=settings.ALT_DOMAINS_AVAILABLE_AT_REGISTRATION,
        if_missing=None,
    )
    firstname_global = validators.String(if_missing=None, strip=True)
    lastname_global = validators.String(if_missing=None, strip=True)

    chained_validators = [
        validators.Availability(),
        validators.Availability(login_field='altdomain_alias'),
    ]


class AccountRegisterDirectoryForm(validators.Schema):
    """Форма создания организации"""
    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(
        not_empty=True,
        strip=True,
    )
    lastname = validators.AntiFraudLastName(
        not_empty=True,
        strip=True,
    )
    domain = validators.Hostname(
        not_empty=True,
        decode_punycode=True,
    )
    login = validators.PddLogin()
    organization = validators.String(if_missing=None, strip=True)
    password = validators.String(not_empty=True)
    eula_accepted = validators.StringBool(not_empty=True)


class BaseOAuthForm(DeviceInfoForm):
    client_id = validators.String(not_empty=True, strip=True)
    client_secret = validators.String(not_empty=True, strip=True)


class BaseGetOrCreateMailishForm(BaseOAuthForm):
    mailish_id = validators.MailishId(not_empty=True)
    task_id = validators.HexString(if_missing=None, strip=True)  # TODO: добавить not_empty=True, когда почта это поддержит
    code_verifier = validators.String(if_missing=None, strip=True)


class GetOrCreateMailishForm(BaseGetOrCreateMailishForm):
    email = validators.ComprehensiveEmailValidator(allow_native=False)


class GetOrCreateMailishAllowNativeForm(BaseGetOrCreateMailishForm):
    email = validators.ComprehensiveEmailValidator(allow_native=True)


class GetOrCreateUberUserForm(DeviceInfoForm):
    uber_id = validators.UberId()


class GetOrCreatePhonishForm(validators.Schema):
    phone_number = validators.String(not_empty=True, strip=True)

    chained_validators = [
        validators.PhoneNumber(),
    ]


class AccountCreateForm(validators.Schema):
    login = validators.Login()
    ignore_stoplist = validators.StringBool(if_missing=False, not_empty=True)
    is_enabled = validators.StringBool(if_missing=True, not_empty=True)

    # Может быть не указан, в этом случае отправляем ссылку для дорегистрации аккаунта на указанный email
    password = validators.String(if_missing=None)

    # Надо ли подписывать на 100 сид
    is_creating_required = validators.StringBool(if_missing=False, not_empty=True)

    # Создать подписку на 669 сиде с логином yastaff_login
    yastaff_login = validators.String(if_missing=None, not_empty=True, strip=True)

    # Персональные данные
    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(if_missing=None, not_empty=True)
    lastname = validators.AntiFraudLastName(if_missing=None, not_empty=True)
    language = validators.PortalLanguage(if_missing=None)
    country = validators.CountryCode(if_missing=None)
    gender = validators.Gender(if_missing=None)
    birthday = validators.Birthday(if_missing=None)
    timezone = validators.Timezone(if_missing=None)

    # Подтвержденный email - средство восстановления (может быть не указан)
    email = validators.SimpleEmailValidator(if_missing=None)

    subscriptions = validators.ListValidator(
        validator=validators.Service(),
        not_in=[Service.by_sid(100), Service.by_sid(669)],
        unique=True,
        if_missing=None,
        not_empty=True,
    )

    chained_validators = [
        validators.Availability(),
        AvailabilityOnSid(669, 'yastaff_login'),
        validators.RequireIfMissing('email', missing='password'),
    ]


class AccountRegisterPddForm(validators.Schema):
    login = validators.PddLogin(not_empty=True)
    domain = validators.Hostname(
        not_empty=True,
        decode_punycode=True,
    )
    is_enabled = validators.StringBool(
        if_missing=True,
        not_empty=True,
    )
    is_creating_required = validators.StringBool(
        if_missing=False,
        not_empty=True,
    )
    is_maillist = validators.StringBool(
        if_missing=False,
        not_empty=True,
    )
    with_yambot_alias = validators.StringBool(
        if_missing=False,
        not_empty=True,
    )

    password = validators.String(not_empty=True, if_missing=None)
    password_hash = validators.PasswordHash(
        allowed_hash_versions=[PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON, PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON],
        not_empty=True,
        if_missing=None,
    )
    weak_password = validators.StringBool(
        if_missing=False,
        not_empty=True,
    )
    no_password = validators.StringBool(
        if_missing=None,
        not_empty=True,
    )

    # Персональные данные
    firstname = validators.FirstName(if_missing=None, not_empty=True, strip=True)
    lastname = validators.LastName(if_missing=None, not_empty=True, strip=True)
    hint_question = validators.String(if_missing=None, not_empty=True, strip=True)
    hint_answer = validators.String(if_missing=None, not_empty=True, strip=True)
    gender = validators.Gender(if_missing=None)
    birthday = validators.Birthday(if_missing=None)
    language = validators.PortalLanguage(if_missing=None)
    country = validators.CountryCode(if_missing=None)
    force_clean_web = validators.StringBool(if_missing=False)

    chained_validators = [
        validators.RequireSome(
            [
                'password',
                'password_hash',
                'no_password',
            ],
        ),
        validators.RequireSet(
            allowed_sets=[
                ['hint_question', 'hint_answer'],
            ],
            allow_empty=True,
        ),
        validators.Availability(
            domain_field='domain',
            login_type=ACCOUNT_TYPE_PDD,
        ),
    ]


class AccountRegisterKolonkishForm(validators.Schema):
    display_name = validators.DisplayName(if_missing=None)
    device_id = validators.DeviceId()
    device_name = validators.DeviceName()

    pre_validators = [
        validators.GroupDisplayName(),
    ]


class AccountRegisterLiteSubmitForm(validators.Schema):
    login = validators.LiteLogin(if_missing=None)
    language = validators.PortalLanguage()

    chained_validators = [
        validators.Availability(login_type=ACCOUNT_TYPE_LITE),
    ]


class AccountRegisterLiteCommitForm(validators.Schema):
    login = validators.LiteLogin(if_missing=None)
    password = validators.String(if_missing=None, not_empty=True)

    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(if_missing=None, not_empty=True, strip=True)
    lastname = validators.AntiFraudLastName(if_missing=None, not_empty=True, strip=True)

    require_firstname = validators.StringBool(if_missing=False, not_empty=True)
    require_lastname = validators.StringBool(if_missing=False, not_empty=True)

    language = validators.PortalLanguage(if_missing=None)
    country = validators.CountryCode(if_missing=None)

    eula_accepted = validators.StringBool(not_empty=True, strip=True)

    unsubscribe_from_maillists = validators.StringBool(if_missing=False)
    origin = validators.String(if_missing=None)
    app_id = validators.String(if_missing=None)

    chained_validators = [
        validators.Availability(login_type=ACCOUNT_TYPE_LITE),
        validators.RequireIfEquals(['firstname'], 'require_firstname', True),
        validators.RequireIfEquals(['lastname'], 'require_lastname', True),
    ]


class AccountRegisterNeophonishForm(validators.Schema):
    force_clean_web = validators.StringBool(if_missing=False)

    language = validators.PortalLanguage(if_missing=None)
    country = validators.CountryCode(if_missing=None)

    eula_accepted = validators.StringBool(not_empty=True, strip=True)

    unsubscribe_from_maillists = validators.StringBool(if_missing=False)
    origin = validators.String(if_missing=None)
    app_id = validators.String(if_missing=None)

    def __init__(self):
        super(AccountRegisterNeophonishForm, self).__init__()
        name_kwargs = {}
        if settings.USE_NEW_SUGGEST_BY_PHONE:
            name_kwargs.update(if_missing=None)

        self.add_field(
            'firstname',
            validators.AntiFraudFirstName(
                strip=True,
                not_empty=True,
                **name_kwargs
            ),
        )
        self.add_field(
            'lastname',
            validators.AntiFraudLastName(
                strip=True,
                not_empty=True,
                **name_kwargs
            ),
        )


class AccountRegisterScholarForm(validators.Schema):
    firstname = validators.AntiFraudFirstName(not_empty=True)
    lastname = validators.AntiFraudLastName(not_empty=True)
    login = validators.ScholarLogin()
    password = validators.String(not_empty=True, strip=True)

    chained_validators = [
        validators.Availability(login_type=ACCOUNT_TYPE_SCHOLAR),
    ]


class AccountRegisterFederalForm(validators.Schema):
    pre_validators = [
        validators.GroupDisplayName(),
    ]
    firstname = validators.FirstName(if_missing=None, not_empty=True, strip=True)
    lastname = validators.LastName(if_missing=None, not_empty=True, strip=True)
    email = validators.SimpleEmailValidator()
    active = validators.StringBool(if_missing=True, not_empty=True)
    # равен полю name_id в SAMLResponse в ручке входа через SAML SSO
    name_id = validators.String(not_empty=True, strip=True)
    domain_id = validators.DomainId(not_empty=True)
    display_name = validators.UserDefinedDisplayName(antifraud_enabled=False)
