# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.account import (
    build_empty_person_info,
    default_account,
    uber_account,
)
from passport.backend.api.common.authorization import authorize_oauth
from passport.backend.api.common.login import build_available_phonish_login
from passport.backend.api.email_validator.mixins import EmailValidatorMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountNotFoundError,
    RateLimitExceedError,
    TaskNotFoundError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleLastAuthMixin
from passport.backend.core.authtypes import AUTH_TYPE_CALLER_ID
from passport.backend.core.builders.social_broker import (
    SocialBroker,
    SocialBrokerInvalidPkceVerifierError,
    SocialBrokerInvalidTaskIdError,
    SocialBrokerTaskNotFoundError,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    register_mailish,
    register_phonish_by_phone,
)
from passport.backend.core.historydb.entry import AuthEntry
from passport.backend.core.historydb.statuses import SUCCESSFUL
from passport.backend.core.host.host import get_current_host
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.email import Emails
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property

from .exceptions import (
    OauthClientAuthInvalidError,
    PkceInvalidError,
)
from .forms import (
    GetOrCreateMailishAllowNativeForm,
    GetOrCreateMailishForm,
    GetOrCreatePhonishForm,
    GetOrCreateUberUserForm,
)


SID_MAILISH = 'mailish'
SID_UBER = 'uber'

MAILISH_TEST_PREFIX = 'yndx-test-mailish-'
MAILISH_BASE_GRANT = 'account.get_or_create_mailish.base'
MAILISH_ALL_GRANT = 'account.get_or_create_mailish'

UBER_BASE_GRANT = 'account.get_or_create_uber_user'

auth_log = logging.getLogger('historydb.auth')


class BaseGetOrCreateView(BundlePhoneMixin, BundleAccountGetterMixin, BaseBundleView):
    """
    Базовый класс для операций создания неполноценных, но долгоживущих аккаунтов. Общая логика:
     - нам приносят какой-то отличительный признак (телефон, email,..)
     - ищем по нему аккаунт в базе; если не нашли - создаём
     - если нужно - финализируем аккаунт
     - выдаём для аккаунта oauth-токен и уид созданного/найденного аккаунта
    """
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    require_track = False

    mode = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.mode,
        )

    @property
    def oauth_device_info(self):
        return {
            name: self.form_values[name]
            for name in self.basic_form.DEVICE_INFO_FIELD_NAMES
        }

    @property
    def client_credentials(self):
        return {}   # pragma: no cover

    def check_preconditions(self):
        pass  # pragma: no cover

    def try_get_account(self):
        raise NotImplementedError()  # pragma: no cover

    def issue_oauth_token(self, uid):
        response = authorize_oauth(
            client_id=self.client_credentials['client_id'],
            client_secret=self.client_credentials['client_secret'],
            env=self.request.env,
            uid=uid,
            **self.oauth_device_info
        )
        if 'error' in response:
            raise OauthClientAuthInvalidError('Bad client_id or client_secret')
        else:
            self.statbox.log(
                action='token_created',
                uid=self.account.uid,
                client_id=self.client_credentials['client_id'],
            )
            return response['access_token']

    def register_account(self):
        raise NotImplementedError()  # pragma: no cover

    def finalize_account(self):
        pass  # pragma: no cover

    def process_request(self):
        self.statbox.log(action='submitted')
        self.process_basic_form()

        self.check_preconditions()

        is_new_account = False
        self.try_get_account()
        if self.account is None:
            is_new_account = True
            self.register_account()

        self.statbox.bind_context(
            uid=self.account.uid,
            login=self.account.login,
        )

        self.finalize_account()

        token = self.issue_oauth_token(uid=self.account.uid)

        self.response_values.update(
            uid=self.account.uid,
            oauth_token=token,
            is_new_account=is_new_account,
        )


class GetOrCreateMailish(BaseGetOrCreateView,
                         EmailValidatorMixin):
    required_grants = [MAILISH_BASE_GRANT]

    mode = 'account_get_or_create_mailish'

    basic_form = None  # определим позже в зависимости от настроек

    def __init__(self):
        super(GetOrCreateMailish, self).__init__()
        self.counter = register_mailish.get_per_consumer_counter()

    @property
    def client_credentials(self):
        return {
            'client_id': self.form_values['client_id'],
            'client_secret': self.form_values['client_secret'],
        }

    def process_basic_form(self):
        if settings.ALLOW_NATIVE_MAILISH_EMAILS:
            self.basic_form = GetOrCreateMailishAllowNativeForm
        else:
            self.basic_form = GetOrCreateMailishForm

        super(GetOrCreateMailish, self).process_basic_form()

    def check_additional_grants(self):
        # Для специальных тестовых логинов расширенный грант не проверяем
        email = self.form_values['email'].lower()
        if not email.startswith(MAILISH_TEST_PREFIX):
            self.check_grant(MAILISH_ALL_GRANT)

    def check_pkce(self):
        task_id = self.form_values['task_id']
        if not task_id:
            return  # TODO: выпилить, когда почта начнёт передавать
        try:
            SocialBroker().check_pkce(task_id, self.form_values['code_verifier'])
        except SocialBrokerInvalidTaskIdError:
            raise ValidationFailedError(['task_id.invalid'])
        except SocialBrokerTaskNotFoundError:
            raise TaskNotFoundError()
        except SocialBrokerInvalidPkceVerifierError:
            raise PkceInvalidError()

    def check_preconditions(self):
        self.check_additional_grants()
        self.check_pkce()

    def check_counter(self):
        if self.counter.hit_limit(self.consumer):
            raise RateLimitExceedError()

    def increment_counter(self):
        self.counter.incr(self.consumer)

    def try_get_account(self):
        try:
            self.account = None
            self.get_account_by_login(
                login=self.form_values['mailish_id'],
                sid=SID_MAILISH,
                emails=True,
                email_attributes='all',
            )
        except AccountNotFoundError:
            pass

    def _has_email(self, email_address):
        for email in self.account.emails.external:
            if email.address == email_address and email.is_rpop:
                return True
        return False

    def _set_email(self, email_address):
        email = self.create_confirmed_email(email_address)
        email.is_rpop = True
        email.is_unsafe = True

        self.account.emails = Emails(parent=self.account)
        self.account.emails.add(email)
        self.account.default_email = email_address
        self.account.person.display_name = DisplayName(name=email_address)

    def register_account(self):
        email = self.form_values['email'].lower()
        mailish_id = self.form_values['mailish_id']

        self.check_counter()

        new_account = default_account(
            login=mailish_id,
            registration_datetime=datetime.now(),
            args={},
            default_person_info={},
            alias_type='mailish',
        )

        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }
        with CREATE(new_account, self.request.env, events) as self.account:
            add_subscription(
                self.account,
                service=get_service(slug='mail'),
                login=mailish_id,
            )
            self._set_email(email)

        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.login,
        )

        self.increment_counter()

    def finalize_account(self):
        rpop_email = self.form_values['email'].lower()
        if self._has_email(rpop_email):
            return  # адрес уже добавлен, ничего делать не нужно

        self.check_counter()

        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self._set_email(rpop_email)

        self.increment_counter()


class GetOrCreateUberUserView(BaseGetOrCreateView):

    required_grants = [UBER_BASE_GRANT]

    basic_form = GetOrCreateUberUserForm

    mode = 'account_get_or_create_uber_user'

    @property
    def client_credentials(self):
        return {
            'client_id': settings.OAUTH_APP_UBER_CLIENT_ID,
            'client_secret': settings.OAUTH_APP_UBER_CLIENT_SECRET,
        }

    def try_get_account(self):
        try:
            self.account = None
            self.get_account_by_login(
                login=self.form_values['uber_id'],
                sid=SID_UBER,
            )
        except AccountNotFoundError:
            pass

    def register_account(self):
        uber_id = self.form_values['uber_id']

        new_account = uber_account(
            uber_id=uber_id,
            registration_datetime=datetime.now(),
        )
        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }
        with CREATE(new_account, self.request.env, events) as self.account:
            pass

        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.login,
        )


class GetOrCreatePhonishView(BaseGetOrCreateView, BundleLastAuthMixin):
    """Регистрируем нового фониша или возвращаем uid фониша по номеру телефона.
       Ручка нужна Такси для авторизации фониша при звонке в колцентр. PASSP-19036
    """
    basic_form = GetOrCreatePhonishForm

    require_track = False
    required_grants = ['account.get_or_create_phonish']

    def __init__(self, *args, **kwargs):
        super(GetOrCreatePhonishView, self).__init__()

        self.counter = register_phonish_by_phone.get_per_consumer_counter()
        self.account = None
        self.is_new_account = False

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_phonish',
            consumer=self.consumer,
        )

    def log_statbox(self, login, account):
        self.statbox.log(
            action='account_created',
            uid=account.uid,
            login=login,
        )

    def update_lastauth(self):
        """Подновляем lastauth через запись в HistoryDB"""
        entry = AuthEntry(
            uid=self.account.uid,
            login=self.account.login,
            type=AUTH_TYPE_CALLER_ID,
            status=SUCCESSFUL,
            host_id=get_current_host().get_id(),
            client_name='passport',
            user_ip=self.request.env.user_ip,
        )
        auth_log.debug(str(entry))

    def try_get_account(self):
        phone_number = self.form_values['phone_number']
        self.account, _ = self.get_account_by_phone_number(
            phone_number,
            phonish_namespace=settings.PHONISH_NAMESPACE_DEFAULT,
        )

    def register_account(self):
        phone_number = self.form_values['phone_number']

        login = build_available_phonish_login(
            settings.PHONISH_LOGIN_GENERATION_RETRIES,
            self.request.env,
        )

        # Если пользователь заводит фониша используя старый номер, для которого
        # уже действует новый, то создаём фониша с новым номером, для того чтобы
        # прекратить появление новых фонишей со старыми номерами.
        phone_number = PhoneNumber.from_deprecated(phone_number)

        time_now = datetime.now()

        new_account = default_account(
            login,
            time_now,
            self.form_values,
            build_empty_person_info(),
            alias_type='phonish',
        )
        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }

        save_simple_phone = self.build_save_simple_phone(
            account=new_account,
            phone_number=phone_number,
            is_new_account=True,
            should_ignore_binding_limit=True,
        )

        with CREATE(new_account, self.request.env, events, datetime_=time_now) as self.account:
            save_simple_phone.submit()
            save_simple_phone.commit()
            add_subscription(
                self.account,
                service=get_service(slug='phonish'),
                login=login,
            )
            self.is_new_account = True

        self.statbox.bind_context(uid=self.account.uid)
        self.log_statbox(login, self.account)
        save_simple_phone.after_commit()

    def check_counter(self):
        if self.counter.hit_limit(self.consumer):
            raise RateLimitExceedError()

    def increment_counter(self):
        self.counter.incr(self.consumer)

    def process_request(self):
        self.check_counter()
        self.process_basic_form()

        self.try_get_account()
        if self.account is None:
            self.register_account()
        elif not self.account.is_enabled:
            raise AccountDisabledError()

        self.update_lastauth()

        self.response_values['uid'] = self.account.uid
        self.response_values['new_account'] = self.is_new_account

        self.increment_counter()
