# -*- coding: utf-8 -*-
from passport.backend.api.common.processes import PROCESS_RESTORE
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountNotFoundError,
    BaseBundleError,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.account import ACCOUNT_DISABLED_ON_DELETION
from passport.backend.core.types.email.email import normalize_email
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid

from .. import exceptions
from ..mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleFamilyMixin,
)
from ..states import (
    DomainNotServed,
    PasswordChangeForbidden,
    RedirectToCompletion,
    RedirectToPDDCompletion,
    RedirectToSocialCompletion,
)
from .semi_auto.base import (
    MULTISTEP_FORM_VERSION,
    STEP_1_PERSONAL_DATA,
)


# Идентификаторы поддерживаемых методов восстановления
RESTORE_METHOD_PHONE = 'phone'
RESTORE_METHOD_PHONE_AND_2FA_FACTOR = 'phone_and_2fa_factor'
RESTORE_METHOD_HINT = 'hint'
RESTORE_METHOD_EMAIL = 'email'
RESTORE_METHOD_LINK = 'link'  # Специальный тип для случаев, когда по ссылке от саппорта сразу вводятся новые данные
RESTORE_METHOD_SEMI_AUTO_FORM = 'semi_auto'

# Средства восстановления, которые пользователь может выбирать в интерфейсе
SELECTABLE_RESTORE_METHODS = {
    RESTORE_METHOD_PHONE,
    RESTORE_METHOD_PHONE_AND_2FA_FACTOR,
    RESTORE_METHOD_EMAIL,
    RESTORE_METHOD_HINT,
    RESTORE_METHOD_SEMI_AUTO_FORM,
}

ALL_RESTORE_METHODS = SELECTABLE_RESTORE_METHODS.union({RESTORE_METHOD_LINK})

PHONE_BASED_RESTORE_METHODS = {
    RESTORE_METHOD_PHONE,
    RESTORE_METHOD_PHONE_AND_2FA_FACTOR,
}

NON_PHONE_BASED_RESTORE_METHODS = ALL_RESTORE_METHODS - PHONE_BASED_RESTORE_METHODS

RESTORE_STEP_CHECK_2FA_FORM = 'check_2fa_form'
RESTORE_STEP_CHECK_PIN = 'check_pin'

# Состояния трека при восстановлении
RESTORE_STATE_SUBMIT_PASSED = 'submit_passed'
RESTORE_STATE_METHOD_SELECTED = 'method_selected'
RESTORE_STATE_METHOD_PASSED = 'method_passed'
RESTORE_STATE_COMMIT_PASSED = 'commit_passed'


# Получение MDB для передачи в почтовое API
# TODO: это древнее поле, которое Почта давно обещает забрать к себе
BLACKBOX_MDB_DBFIELD = 'hosts.db_id.2'
BLACKBOX_FIELDS_WITH_MDB = settings.BLACKBOX_FIELDS + (BLACKBOX_MDB_DBFIELD,)


class GetAccountForRestoreMixin(
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleFamilyMixin,
):
    def get_and_validate_account(self, login=None, uid=None, check_autoregistered=True, check_domain_support=False,
                                 check_child_family=False, check_disabled_status=True, skip_validation=False,
                                 allow_missing_password_with_portal_alias=False,
                                 allow_social_missing_password_with_portal_alias=False,
                                 skip_statbox_log=False, **blackbox_kwargs):
        """
        Получить аккаунт и проверить применимость процедуры восстановления для аккаунта.
        Возвращает признак необходимости перенаправления пользователя для выполнения других шагов (дорегистрация,
        обращение к администратору домена и т.п.).
        Работает либо по логину, либо по UID.
        @param login: логин пользователя
        @param uid: UID пользователя
        @param check_autoregistered: проверить требование смены пароля для автозарегистрированного пользователя
        @param check_domain_support: проверить обслуживание ПДД-домена саппортами
        @param check_child_family: проверить возможность вернуть child-аккаунт в семью
        @param check_disabled_status: проверить возможность восстановления заблокированного аккаунта
        @param skip_validation: не выполнять никаких проверок после получения аккаунта
        @param allow_missing_password_with_portal_alias: пропускать пользователя с портальным алиасом, но без пароля
        @param allow_social_missing_password_with_portal_alias: пропускать пользователя с социальным и портальным
        алиасами, но без пароля
        @param blackbox_kwargs: дополнительные опции для ЧЯ
        """
        if not login and not uid:
            raise ValueError('Login and UID are both empty')
        try:
            if self.account is None:
                self._get_account(login=login, uid=uid, get_family_info=True, **blackbox_kwargs)
            if not skip_statbox_log:
                self.statbox.bind_context(
                    uid=self.account.uid,
                    login=self.account.login,
                )
            if skip_validation:
                return False
            processing_finished = self._validate_account(
                check_disabled_status=check_disabled_status,
                check_autoregistered=check_autoregistered,
                check_domain_support=check_domain_support,
                check_child_family=check_child_family,
                allow_missing_password_with_portal_alias=allow_missing_password_with_portal_alias,
                allow_social_missing_password_with_portal_alias=allow_social_missing_password_with_portal_alias,
            )
            if processing_finished and not skip_statbox_log:
                self.statbox.log(
                    action='finished_with_state',
                    state=self.state.state,
                )
            return processing_finished
        except BaseBundleError as e:
            if not skip_statbox_log:
                self.statbox.log(
                    action='finished_with_error',
                    error=e.error,
                )
            raise

    def _get_account(self, login=None, uid=None, **blackbox_kwargs):
        if login:
            try:
                response = self.get_account_by_login(
                    login,
                    enabled_required=False,
                    dbfields=BLACKBOX_FIELDS_WITH_MDB,
                    find_by_phone_alias=None,
                    **blackbox_kwargs
                )
            except AccountNotFoundError:
                if '@' not in login:
                    # Проверяем лайт-пользователя, должна быть собака в логине
                    raise AccountNotFoundError()
                response = self.get_account_by_login(
                    login,
                    enabled_required=False,
                    sid='mk',
                    dbfields=BLACKBOX_FIELDS_WITH_MDB,
                    find_by_phone_alias=None,
                    **blackbox_kwargs
                )
        elif uid:
            response = self.get_account_by_uid(
                uid,
                enabled_required=False,
                dbfields=BLACKBOX_FIELDS_WITH_MDB,
                **blackbox_kwargs
            )
        self.userinfo_response = response

    def _validate_account(self, check_disabled_status, check_autoregistered,
                          check_domain_support, check_child_family, allow_missing_password_with_portal_alias,
                          allow_social_missing_password_with_portal_alias):
        """
        Выполнить проверку применимости процедуры восстановления для аккаунта.
        Возвращает признак необходимости перенаправления пользователя для выполнения других шагов (дорегистрация,
        обращение к администратору домена и т.п.).
        """
        if check_disabled_status:
            if not self.account.is_user_enabled and not self.can_restore_disabled_account(self.account):
                raise AccountDisabledError()

        if self.account.is_pdd:
            # ПДД-пользователь может быть недорегистрирован
            if not self.account.is_complete_pdd:
                self.state = RedirectToPDDCompletion()
                return True

            # Некоторые ПДД-пользователи не имеют права изменять свой пароль
            response = self.blackbox.hosted_domains(domain=self.account.domain.domain)
            self.account.parse(response)
            if not self.account.domain.can_users_change_password:
                self.state = PasswordChangeForbidden()
                return True

            if (check_domain_support and
                    self.account.domain.unicode_domain not in settings.DOMAINS_SERVED_BY_SUPPORT):
                self.state = DomainNotServed()
                return True
        elif self.account.is_incomplete_autoregistered:
            # Автозарегистрированный пользователь, требуется создание пароля и принятие ПС
            if check_autoregistered:
                self.state = RedirectToCompletion()
                return True
            else:
                self.statbox.bind_context(is_autoregistered_completion_required=True)

        if not self.account.have_password:
            if (allow_missing_password_with_portal_alias and
                    self.account.portal_alias):
                # Опционально пропускаем пользователя с портальным алиасом и без пароля
                if self.account.social_alias:
                    # Случай пользователя с социальным алиасом - требуется дорегистрация
                    if allow_social_missing_password_with_portal_alias:
                        self.statbox.bind_context(is_social_completion_required=True)
                    else:
                        self.state = RedirectToSocialCompletion()
                        return True
                else:
                    self.statbox.bind_context(is_password_missing=True)
                return False
            elif self.account.social_alias:
                # Пользователь без пароля, с социальным алиасом
                self.state = RedirectToSocialCompletion()
                return True
            raise exceptions.AccountWithoutPasswordError()

        if (
            check_child_family and
            self.account.is_child and
            self.account.disabled_status == ACCOUNT_DISABLED_ON_DELETION and
            not self.account.has_family
        ):
            if not self.account.last_child_family:
                raise ValueError('Child is disabled on deletion, but last_child_family is empty')
            self.load_family_info_by_family_id(self.account.last_child_family)
            self.get_family_member_free_place()

        return False


class RestoreSemiAutoBaseMixin(object):
    """
    Общий код, обеспечивающий взаимодействие автовосстановления и анкеты восстановления.
    """
    def fill_track_with_account_data_for_semi_auto_form(self):
        """
        Установка полей трека, используемых анкетой, на основании данных аккаунта.
        """
        self.track.uid = self.account.uid
        self.track.login = self.account.login
        self.track.country = settings.DEFAULT_COUNTRY
        if self.account.person.country:
            self.track.country = self.account.person.country
        if self.account.emails is not None:
            self.track.emails = ' '.join([normalize_email(email.address) for email in self.account.emails.native])
        if self.account.is_pdd:
            self.track.domain = self.account.domain.domain

    def setup_multistep_process(self):
        """
        Настройка многошагового процесса анкеты.
        """
        # Установить первый шаг для многошаговой анкеты
        self.track.semi_auto_step = STEP_1_PERSONAL_DATA
        # Установить версию многошаговой анкеты для проверки консистентности на последующих шагах
        self.track.version = MULTISTEP_FORM_VERSION
        self.statbox.bind_context(version=MULTISTEP_FORM_VERSION)

    def setup_track_for_semi_auto_form(self, login, request_source, is_unconditional_pass=False):
        """
        Установка всех необходимых полей трека для анкеты восстановления.
        """
        self.setup_multistep_process()
        self.track.user_entered_login = login
        self.track.request_source = request_source
        self.track.is_unconditional_pass = is_unconditional_pass
        self.track.is_for_learning = self.is_form_suitable_for_learning(request_source, uid=self.account.uid)
        self.fill_track_with_account_data_for_semi_auto_form()

    def setup_track_after_semi_auto_form_passed(self):
        """
        Установка состояния в треке для перехода на ввод новых данных после автоматического "Да" на анкете.
        Ввод новых данных происходит в контексте автоматического восстановления.
        """
        self.track.process_name = PROCESS_RESTORE  # FIXME: перевести анкету на процесс restore
        self.track.login = self.account.login
        self.track.uid = self.account.uid
        self.track.current_restore_method = RESTORE_METHOD_SEMI_AUTO_FORM
        self.mark_method_passed()

    def mark_method_passed(self):
        """
        Установим в треке признак успешного прохождения способа восстановления, настроим трек для последующего
        шага ввода и валидации пароля
        """
        self.track.restore_state = RESTORE_STATE_METHOD_PASSED
        self.fill_track_for_password_validation()

    def fill_track_for_password_validation(self):
        self.track.login = self.account.login
        self.track.is_strong_password_policy_required = self.account.is_strong_password_required

    def is_form_suitable_for_learning(self, request_source, uid=None):
        """
        Логика отделения потока на обучение.
        Без UID - случаи инициализации анкеты серверной ручкой submit.
        """
        if request_source not in settings.RESTORE_SEMI_AUTO_LEARNING_DENOMINATORS or uid is None:
            return False

        return is_experiment_enabled_by_uid(
            uid,
            settings.RESTORE_SEMI_AUTO_LEARNING_DENOMINATORS[request_source],
        )
