# -*- coding: utf-8 -*-

import logging
import time

from passport.backend.api.common import looks_like_yandex_email
from passport.backend.api.common.account import default_revokers
from passport.backend.api.common.authorization import (
    SessionScope,
    set_authorization_track_fields,
)
from passport.backend.api.common.processes import (
    PROCESS_LOGIN_RESTORE,
    PROCESS_RESTORE,
)
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.challenge.controllers import reset_challenge_counters
from passport.backend.api.views.bundle.exceptions import (
    AccountGlobalLogoutError,
    AccountInvalidTypeError,
    InvalidTrackStateError,
    PhoneCompromisedError,
    RateLimitExceedError,
    SecretKeyInvalidError,
    UserNotVerifiedError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BindRelatedPhonishAccountMixin,
    BundleAccountFlushMixin,
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleFixPDDRetpathMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
    BundlePersistentTrackMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.common import BundleAdminActionMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.api.views.bundle.restore.base import (
    ALL_RESTORE_METHODS,
    GetAccountForRestoreMixin,
    PHONE_BASED_RESTORE_METHODS,
    RESTORE_METHOD_EMAIL,
    RESTORE_METHOD_HINT,
    RESTORE_METHOD_LINK,
    RESTORE_METHOD_PHONE,
    RESTORE_METHOD_PHONE_AND_2FA_FACTOR,
    RESTORE_METHOD_SEMI_AUTO_FORM,
    RESTORE_STATE_COMMIT_PASSED,
    RESTORE_STATE_METHOD_PASSED,
    RESTORE_STATE_METHOD_SELECTED,
    RESTORE_STATE_SUBMIT_PASSED,
    RestoreSemiAutoBaseMixin,
)
from passport.backend.api.views.bundle.restore.exceptions import (
    EmailChangedError,
    RestoreMethodNotAllowedError,
)
from passport.backend.api.views.bundle.restore.forms import (
    RestoreCommitForm,
    RestoreCommitNewMethodForm,
    RestoreCreateLinkForm,
    RestoreKeySubmitForm,
    RestoreSelectMethodForm,
    RestoreSubmitForm,
)
from passport.backend.api.views.bundle.restore.helpers import (
    get_2fa_email_notification_data,
    get_restore_passed_email_notification_data,
    restore_passed_message_context_shadower,
    SupportLinkOptions,
)
from passport.backend.api.views.bundle.states import (
    DomainNotServed,
    PhoneAliasProhibited,
    RateLimitExceeded,
    RedirectToForcedLiteCompletion,
    Show2FAPromo,
)
from passport.backend.api.views.bundle.utils import (
    make_hint_question,
    write_phone_to_log,
)
from passport.backend.core.builders.blackbox.exceptions import BaseBlackboxError
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_l import (
    CookieL,
    CookieLUnpackError,
)
from passport.backend.core.counters import restore_counter
from passport.backend.core.counters.change_password_counter import get_per_phone_number_buckets
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.historydb.events import (
    ACTION_RESTORE_ENTITIES_FLUSHED,
    ACTION_RESTORE_SUPPORT_LINK_CREATED,
    EVENT_ACTION,
    EVENT_INFO_FLUSHED_ENTITIES,
    EVENT_INFO_SUPPORT_LINK_TYPE,
)
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    StatboxLogger,
    to_statbox,
)
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    send_mail_for_account,
)
from passport.backend.core.models.account import (
    Account,
    ACCOUNT_DISABLED_ON_DELETION,
    get_preferred_language,
)
from passport.backend.core.models.hint import Hint
from passport.backend.core.models.password import (
    get_sha256_hash,
    PASSWORD_CHANGING_REASON_HACKED,
)
from passport.backend.core.models.persistent_track import (
    TRACK_TYPE_AUTH_BY_KEY_LINK,
    TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK,
    TRACK_TYPE_RESTORATION_SUPPORT_LINK,
)
from passport.backend.core.models.phones.phones import (
    RemoveSecureOperation,
    ReplaceSecurePhoneWithBoundPhoneOperation,
    ReplaceSecurePhoneWithNonboundPhoneOperation,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.support_link_types import (
    BLOCKING_SUPPORT_LINK_TYPES,
    SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD,
    SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION,
    SUPPORT_LINK_TYPE_FORCE_PHONE_RESTORATION,
    SUPPORT_LINK_TYPE_REDIRECT_TO_COMPLETION,
)
from passport.backend.core.types.email.email import (
    is_yandex_email,
    punycode_email,
)
from passport.backend.core.types.login.login import (
    extract_phonenumber_alias_candidate_from_login,
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.phone import PhoneNumber
from passport.backend.core.types.phone_number.phone_number import parse_phone_number
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import (
    datetime_to_unixtime,
    unixtime_to_datetime,
)


log = logging.getLogger('passport.api.views.bundle.restore.controllers')


RESTORE_BASE_GRANT = 'restore.base'
RESTORE_CREATE_LINK_GRANT = 'restore.create_link'


RESTORE_TRACK_TYPE = 'restore'


class BaseRestoreView(BaseBundleView, GetAccountForRestoreMixin, BundlePhoneMixin, RestoreSemiAutoBaseMixin):
    process_name = PROCESS_RESTORE

    require_process = True

    require_track = True

    required_grants = [RESTORE_BASE_GRANT]

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
    )

    restore_step = None

    @cached_property
    def statbox(self):
        statbox_params = dict(
            mode='restore',
            track_id=self.track_id,
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            step=self.restore_step,
            consumer=self.consumer,
        )
        if self.track:
            statbox_params.update(
                login=self.track.login,
                is_suggested_login=self.track.user_entered_login in self.track.suggested_logins.get(),
                uid=self.track.uid,
                retpath=self.track.retpath,
                current_restore_method=self.track.current_restore_method,
                support_link_type=self.track.support_link_type,
                **self.get_statistics_for_statbox()
            )
        return StatboxLogger(**statbox_params)

    @cached_property
    def antifraud_logger(self):
        # Аналогично PASSP-36049
        scenario = None
        if self.track.scenario:
            scenario = self.track.scenario
        elif self.track.track_type:
            scenario = self.track.track_type

        return AntifraudLogger(
            external_id='track-{}'.format(self.track_id),
            channel='pharma',
            status='OK',
            uid=self.track.uid,
            sub_channel=self.consumer,
            phone_confirmation_method=self.track.phone_confirmation_method,
            request_path=self.request.env.request_path,
            scenario=scenario,
        )

    def get_statistics_for_statbox(self):
        selected_methods_counters = self.track.restore_methods_select_counters or {}
        statbox_params = {
            '%s_select_count' % method: count for (method, count) in selected_methods_counters.items()
        }
        selected_methods_order = self.track.restore_methods_select_order.get()
        statbox_params.update({
            'selected_methods_order': ','.join(selected_methods_order),
            # данные о числе неудачных проверок для всех поддерживаемых методов восстановления
            'phone_checks_count': self.track.secure_phone_checks_count.get(),
            'answer_checks_count': self.track.answer_checks_count.get(),
            '2fa_form_checks_count': self.track.restore_2fa_form_checks_count.get(),
            'pin_checks_count': self.track.pin_check_errors_count.get(),
            'email_checks_count': self.track.email_checks_count.get(),
            # число отправленных email-сообщений
            'restoration_emails_count': self.track.restoration_emails_count.get(),
        })
        return statbox_params

    def raise_error_with_logging(self, exception_type):
        self.statbox.log(action='finished_with_error', error=exception_type.error)
        raise exception_type()

    def fill_basic_response(self, user_entered_login=None):
        """
        Данные, которые всегда должны быть в ответе для фронтенда
        """
        if user_entered_login or (self.track and self.track.user_entered_login):
            self.response_values['user_entered_login'] = user_entered_login or self.track.user_entered_login
        if self.track_id:
            self.response_values['track_id'] = self.track_id

    def check_global_counters(self, finish_with_state=False):
        """
        Проверка глобальных счетчиков, делающих невозможной процедуру восстановления.
        @param finish_with_state: Признак необходимости вернуть состояние в случае превышения глобального лимита
        @return Признак необходимости редиректа
        """
        limit = None
        if self.track.support_link_type:
            # Для случая саппортских ссылок лимит увеличен - это дает пользователю доп. число попыток
            # пройти процесс, но не дает возможности перебора средства восстановления.
            limit = settings.RESTORE_PER_IP_COUNTER_LIMIT_FOR_SUPPORT_LINK
        global_limit_reached = restore_counter.is_ip_limit_exceeded(self.client_ip, only_check=True, limit=limit)
        if global_limit_reached and finish_with_state:
            self.state = RateLimitExceeded()
            self.statbox.log(action='finished_with_state', state=self.state.state)
            return True
        elif global_limit_reached:
            self.raise_error_with_logging(RateLimitExceedError)
        return False

    def increase_global_counters(self):
        restore_counter.get_per_ip_buckets().incr(self.client_ip)

    def assert_track_valid(self, allowed_states, allowed_methods=None):
        """
        "Забор" от некорректных треков. Ошибки не логгируются в статбокс, так как означают некорректное
        использование ручек (или нас пытаются сломать?).
        """
        if self.track.track_type != RESTORE_TRACK_TYPE or not self.track.uid:
            raise InvalidTrackStateError('Bad track type or no UID in track')
        if self.track.restore_state not in allowed_states:
            raise InvalidTrackStateError('Invalid restore track state (%s)' % self.track.restore_state)
        current_method = self.track.current_restore_method
        if current_method and allowed_methods and current_method not in allowed_methods:
            raise InvalidTrackStateError('Invalid restore method (%s)' % current_method)

    @cached_property
    def is_phonenumber_alias_used_as_login(self):
        """
        Отвечает на вопрос, использовался ли в качестве логина цифровой алиас.
        """
        if self.account.phonenumber_alias.alias:
            user_entered_login = self.track.user_entered_login or self.form_values['login']
            alias_candidate = extract_phonenumber_alias_candidate_from_login(user_entered_login)
            if alias_candidate:
                # На аккаунте есть цифровой алиас, в логине пользователя есть значение, похожее на алиас -
                # нужно сходить в ЧЯ с sid=restore. Если использовался алиас, аккаунт не будет найден.
                response = self.blackbox.userinfo(sid='restore', login=user_entered_login)
                if response['uid'] is None:
                    self.statbox.bind_context(is_phonenumber_alias_used_as_login=True)
                    return True

    def get_suitable_restore_methods(self):
        """
        Получить доступные для аккаунта способы восстановления (без учета состояния счетчиков).
        """
        methods = []
        if self.track.current_restore_method == RESTORE_METHOD_LINK:
            # Специальный случай - восстановление по ссылке от саппортов, сразу ведущей
            # на страницу ввода новых данных
            methods = [RESTORE_METHOD_LINK]
        elif self.account.is_maillist:
            # Для аккаунтов-рассылок разрешена только анкета (либо ввод новых данных по ссылке от саппорта)
            pass
        elif self.account.totp_secret.is_set:
            if self.support_link_options.force_2fa_phone_restoration:
                methods = [RESTORE_METHOD_PHONE]
            elif not self.is_phonenumber_alias_used_as_login:
                methods = [RESTORE_METHOD_PHONE_AND_2FA_FACTOR]
        elif self.support_link_options.force_hint_restoration:
            if self.account.hint.is_set:
                methods = [RESTORE_METHOD_HINT]
        else:
            has_suitable_phone = bool(self.account_phones.suitable_for_restore)
            if has_suitable_phone and not self.is_phonenumber_alias_used_as_login:
                methods.append(RESTORE_METHOD_PHONE)
            has_suitable_email = bool(self.account.emails.suitable_for_restore)
            if has_suitable_email and not self.account.password.is_changing_required:
                methods.append(RESTORE_METHOD_EMAIL)
            if (self.account.hint.is_set and
                    not has_suitable_email and
                    not has_suitable_phone and
                    not self.account.is_strong_password_required and
                    not self.account.password.is_changing_required):
                methods.append(RESTORE_METHOD_HINT)
            elif self.account.hint.is_set:
                self.statbox.bind_context(is_hint_masked=True)

        if self.is_semi_auto_form_suitable(other_methods_available=bool(methods)):
            methods.append(RESTORE_METHOD_SEMI_AUTO_FORM)

        self.statbox.bind_context(suitable_restore_methods=','.join(methods))
        return methods

    def is_semi_auto_form_suitable(self, other_methods_available):
        if self.account.is_pdd:
            if self.account.domain.unicode_domain not in settings.DOMAINS_SERVED_BY_SUPPORT:
                if not other_methods_available:
                    self.state = DomainNotServed()
                    self.statbox.log(action='finished_with_state', state=self.state.state)
                return False
        is_semi_auto_form_passed = (
            self.track.current_restore_method == RESTORE_METHOD_SEMI_AUTO_FORM and
            self.track.restore_state == RESTORE_STATE_METHOD_PASSED
        )
        can_show_semi_auto_form = (
            # Безусловно показываем анкету пользователям без средств восстановления
            not other_methods_available or
            # Если к нам пришли после автоматического ДА анкеты, не применяем правило показа анкеты
            # на долю аудитории
            is_semi_auto_form_passed or
            # В остальных случаях показываем анкету на долю аудитории
            is_experiment_enabled_by_uid(
                self.account.uid,
                settings.SHOW_SEMI_AUTO_FORM_ON_AUTO_RESTORE_DENOMINATOR,
            )
        )
        return can_show_semi_auto_form

    def assert_restore_method_suitable(self, method=None, suitable_methods=None):
        """
        Проверяем, что метод восстановления все еще доступен для аккаунта.
        Кидаем общую ошибку. Появление такой ошибки - повод отправить пользователя на анкету.
        """
        method = method or self.track.current_restore_method
        suitable_methods = suitable_methods or self.get_suitable_restore_methods()
        if method not in suitable_methods:
            self.raise_error_with_logging(RestoreMethodNotAllowedError)

    def is_captcha_required_for_hint(self):
        return self.track.answer_checks_count.get(default=0) >= settings.ANSWER_CHECK_ERRORS_CAPTCHA_THRESHOLD

    def is_captcha_required_for_2fa_form(self):
        threshold = settings.RESTORE_2FA_FORM_CHECK_ERRORS_CAPTCHA_THRESHOLD
        return self.track.restore_2fa_form_checks_count.get(default=0) >= threshold

    def get_pin_checks_left(self):
        return max(
            settings.ALLOWED_PIN_CHECK_FAILS_COUNT - self.account.totp_secret.failed_pin_checks_count,
            0,
        ) if self.account.totp_secret.is_set else 0

    def assert_logout_not_occurred(self, restoration_key_created_at=None):
        """
        Проверка состояния логаута аккаунта:
        1) Трек был создан не позднее чем произошел логаут
        2) БД-трек, если он используется, был создан не позднее логаута
        """
        check_datetimes = []
        if self.track:
            check_datetimes.append(unixtime_to_datetime(self.track.logout_checkpoint_timestamp))
        if restoration_key_created_at:
            check_datetimes.append(restoration_key_created_at)
        if check_datetimes and self.account.is_logouted_after(min(check_datetimes)):
            self.raise_error_with_logging(AccountGlobalLogoutError)

    def assert_support_link_type_valid_for_account(self, link_type):
        """
        Проверяем, что саппортская ссылка данного типа подходит для аккаунта.
        Возвращает особый тип ошибки, так как процесс нельзя продолжить.
        """
        if self.account.totp_secret.is_set and link_type == SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION:
            # Восстановление по КВ недоступно для 2ФА пользователей
            self.raise_error_with_logging(AccountInvalidTypeError)
        if not self.account.totp_secret.is_set and link_type == SUPPORT_LINK_TYPE_FORCE_PHONE_RESTORATION:
            # Возможность восстановления только по телефону доступна только для 2ФА пользователей
            self.raise_error_with_logging(AccountInvalidTypeError)
        if link_type == SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION and not self.account.hint.is_set:
            self.raise_error_with_logging(AccountInvalidTypeError)
        if (link_type == SUPPORT_LINK_TYPE_REDIRECT_TO_COMPLETION and
                not (self.is_social_completion_required or self.account.is_incomplete_autoregistered)):
            # Возможна дорегистрация пользователя с портальным и социальным алиасами, но без пароля; дорегистрация
            # автозарегистрированного пользователя
            self.raise_error_with_logging(AccountInvalidTypeError)

    def assert_track_and_restoration_key_valid_for_account(self):
        """
        Проверяем, что на аккаунте не произошел логаут после создания трека и ссылки восстановления
         (если ссылка применялась в процессе восстановления).
        """
        restoration_key_created_at = None
        if self.track.restoration_key_created_at:
            restoration_key_created_at = unixtime_to_datetime(float(self.track.restoration_key_created_at))
        self.assert_logout_not_occurred(restoration_key_created_at=restoration_key_created_at)
        if self.track.support_link_type:
            self.assert_support_link_type_valid_for_account(self.track.support_link_type)

    @cached_property
    def support_link_options(self):
        return SupportLinkOptions(self.track.support_link_type)

    def get_and_validate_account_from_track(self, **params):
        return self.get_and_validate_account(
            uid=self.track.uid,
            emails=True,
            need_phones=True,
            check_disabled_status=not self.support_link_options.is_disabled_account_allowed,
            allow_missing_password_with_portal_alias=self.support_link_options.is_missing_password_with_portal_alias_allowed,
            **params
        )

    @cached_property
    def is_social_completion_required(self):
        return bool(
            not self.account.have_password and
            self.account.portal_alias and
            self.account.social_alias,
        )

    def is_email_suitable_for_restore(self, encoded_email):
        """
        Проверяет, что заданный email, закодированный в punycode, подходит для восстановления.
        Проверяет совпадение без учета регистра. Учитывает нормализацию при сравнении нативных адресов.
        """
        normalized_login = normalize_login(raw_login_from_email(encoded_email))
        is_native_email = is_yandex_email(encoded_email)
        matched_emails = []
        for email in self.account.emails.all:
            if encoded_email.lower() == email.address.lower():
                matched_emails.append(email)
                break
            if is_native_email and is_yandex_email(email.address):
                # Оба адреса являются нативными, логинная часть совпадает с точностью до нормализации
                if normalized_login == normalize_login(raw_login_from_email(email.address)):
                    # В теории может быть привязано несколько email-адресов с разными Яндекс-доменами,
                    # часть из которых пригодна для восстановления, часть нет - сохраняем все.
                    matched_emails.append(email)

        self.statbox.bind_context(
            matched_emails_count=len(matched_emails),
        )
        if matched_emails:
            matched_email = sorted(matched_emails, key=lambda email: email.is_suitable_for_restore)[-1]
            is_email_suitable = matched_email.is_suitable_for_restore

            self.statbox.bind_context(
                is_email_suitable=is_email_suitable,
                is_email_external=matched_email.is_external,
                is_email_confirmed=matched_email.is_confirmed,
                is_email_unsafe=matched_email.is_unsafe,
                is_email_rpop=matched_email.is_rpop,
                is_email_silent=matched_email.is_silent,
            )
            return is_email_suitable
        return False

    @cached_property
    def allowed_methods_to_bind(self):
        """
        Определяем набор способов восстановления, один из которых можно привязать в процессе восстановления.
        В случае работы по саппортской ссылке, этот набор определяется свойствами ссылки.
        В противном случае, возможна привязка телефона в следующих случаях:
         - при восстановлении через большую анкету;
         - при восстановлении не по телефону при наличии требования смены пароля;
         - при отсутствии защищенного телефона.
        в других случаях рекомендуем привязать телефон, если его нет.
        """
        if self.track.support_link_type:
            return self.support_link_options.allowed_methods_to_bind
        current_method = self.track.current_restore_method
        if (
            not self.account_phones.suitable_for_restore or
            current_method == RESTORE_METHOD_SEMI_AUTO_FORM or
            current_method != RESTORE_METHOD_PHONE and self.account.password.is_changing_required
        ):
            return [RESTORE_METHOD_PHONE]
        return []

    @cached_property
    def is_method_binding_required(self):
        """
        Определяем, требуется ли привязка нового средства восстановления.
        В случае работы по саппортской ссылке, требование привязки определяется типом ссылки.
        В противном случае, привязка необходима в следующих случаях:
         - при восстановлении через большую анкету;
         - при восстановлении не по телефону при наличии требования смены пароля.
        """
        if self.track.support_link_type:
            return self.support_link_options.is_method_binding_required
        current_method = self.track.current_restore_method
        return (
            current_method == RESTORE_METHOD_SEMI_AUTO_FORM or
            current_method != RESTORE_METHOD_PHONE and self.account.password.is_changing_required
        )

    def assert_phone_not_compromised(self, phone_number):
        """
        Проверяем, что привязываемый телефон может быть использован как защищенный.
        """
        counter = get_per_phone_number_buckets()
        is_exceeded = counter.hit_limit(phone_number)

        if is_exceeded:
            self.raise_error_with_logging(PhoneCompromisedError)

    @cached_property
    def is_glogout_required(self):
        return (
            self.account.is_strong_password_required or
            self.account.is_yandexoid or
            self.account.is_corporate or
            self.account.is_betatester or
            self.account.password.forced_changing_reason == PASSWORD_CHANGING_REASON_HACKED or
            self.track.current_restore_method == RESTORE_METHOD_LINK
        )


class RestoreSubmitView(BaseRestoreView, BundleFixPDDRetpathMixin, BundleAssertCaptchaMixin):
    """
    Проверяем капчу и применимость аккаунта
    """
    basic_form = RestoreSubmitForm
    restore_step = 'submit'

    def assert_track_valid(self):
        if self.track.track_type != RESTORE_TRACK_TYPE or self.track.restore_state:
            # В ручку нужно приходить с чистым треком типа restore
            raise InvalidTrackStateError()

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid()
        login = self.form_values['login']
        self.fill_basic_response(user_entered_login=login)
        if looks_like_yandex_email(login):
            self.response_values.update(looks_like_yandex_email=True)

        self.statbox.bind_context(
            login=login,
            is_suggested_login=login in self.track.suggested_logins.get(),
            captcha_check_count=self.track.captcha_check_count.get(default=0),
            retpath=self.form_values['retpath'],
        )

        if not self.is_captcha_passed:
            self.raise_error_with_logging(UserNotVerifiedError)

        redirect_required = self.get_and_validate_account(login, check_child_family=True)
        if redirect_required:
            return
        if self.is_phonenumber_alias_used_as_login:
            # Временная заглушка: не пускаем аккаунты с логином в виде телефонного алиаса
            self.state = PhoneAliasProhibited()
            self.statbox.log(action='finished_with_state', state=self.state.state)
            return
        self.assert_track_and_restoration_key_valid_for_account()  # Отсекаем протухшие треки для данного аккаунта

        with self.track_transaction.rollback_on_error():
            self.track.user_entered_login = login
            self.track.login = self.account.login
            self.track.uid = self.account.uid
            self.track.restore_state = RESTORE_STATE_SUBMIT_PASSED
            self.track.gps_package_name = self.form_values['gps_package_name']
            if self.form_values['retpath']:
                if self.account.is_pdd:
                    self.fix_pdd_retpath()
                self.track.retpath = self.form_values['retpath']
                self.statbox.bind_context(retpath=self.track.retpath)
        self.fill_response_with_track_fields('track_id', 'user_entered_login')
        self.statbox.log(action='passed')


ALLOWED_PERSISTENT_TRACK_TYPES = {
    TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK,
    TRACK_TYPE_RESTORATION_SUPPORT_LINK,
}


class RestoreKeySubmitView(BaseRestoreView, BundlePersistentTrackMixin):
    """
    Проверяем код восстановления, обновляем или создаем трек, задаем состояние, предусмотренное кодом.
    """
    require_track = False
    basic_form = RestoreKeySubmitForm
    restore_step = 'key_submit'

    def read_and_validate_restoration_persistent_track(self, parsed_track_key, is_track_passed=False):
        uid = parsed_track_key['uid']
        track_id = parsed_track_key['track_id']
        persistent_track = self.read_persistent_track(uid, track_id)
        self.statbox.bind_context(
            does_key_exist=persistent_track.exists,
            is_key_expired=persistent_track.is_expired if persistent_track.exists else None,
            key_type=persistent_track.type if persistent_track.exists else None,
        )
        # Базовые проверки persistent-трека на валидность
        if (not persistent_track.exists or
                persistent_track.is_expired or
                persistent_track.type not in ALLOWED_PERSISTENT_TRACK_TYPES):
            self.raise_error_with_logging(SecretKeyInvalidError)
        # Дополнительные проверки в случае работы с существующим redis-треком
        if is_track_passed and (
                persistent_track.type != TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK or
                str(uid) != self.track.uid
        ):
            self.raise_error_with_logging(SecretKeyInvalidError)
        return persistent_track

    def assert_email_still_suitable_for_restore(self, email):
        email = punycode_email(email)
        if not self.is_email_suitable_for_restore(email):
            self.raise_error_with_logging(EmailChangedError)

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['secret_key']['uid']
        is_track_passed = bool(self.track_id)
        if is_track_passed:
            # С существующим треком можно прийти только с восстановления по email-адресу, в случае если
            # пользователь вводит код восстановления в форме
            self.read_track()
            self.assert_track_valid(
                allowed_states=[RESTORE_STATE_METHOD_SELECTED],
                allowed_methods=[RESTORE_METHOD_EMAIL],
            )
        self.statbox.bind_context(
            uid=self.track.uid if is_track_passed else uid,
            track_id=self.track_id,
            is_track_passed=is_track_passed,
        )

        persistent_track = self.read_and_validate_restoration_persistent_track(
            self.form_values['secret_key'],
            is_track_passed=is_track_passed,
        )
        self.fill_basic_response(user_entered_login=self.get_login_from_persistent_track(persistent_track))

        allow_disabled_account = allow_missing_password_with_portal_alias = False
        if persistent_track.type == TRACK_TYPE_RESTORATION_SUPPORT_LINK:
            link_options = SupportLinkOptions(persistent_track.content['support_link_type'])
            allow_disabled_account = link_options.is_disabled_account_allowed
            allow_missing_password_with_portal_alias = link_options.is_missing_password_with_portal_alias_allowed
        redirect_required = self.get_and_validate_account(
            uid=uid,
            emails=True,
            need_phones=True,
            check_child_family=True,
            check_disabled_status=not allow_disabled_account,
            allow_missing_password_with_portal_alias=allow_missing_password_with_portal_alias,
        )
        if redirect_required:
            return
        self.assert_logout_not_occurred(restoration_key_created_at=persistent_track.created)
        if not is_track_passed:
            self.create_track(RESTORE_TRACK_TYPE, process_name=PROCESS_RESTORE)
            self.statbox.bind_context(track_id=self.track_id)

        if persistent_track.type == TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK:
            self.process_auto_email_restoration(persistent_track, is_track_passed)
        elif persistent_track.type == TRACK_TYPE_RESTORATION_SUPPORT_LINK:
            self.process_support_link_restoration(persistent_track)
        self.statbox.log(action='passed')

    def process_auto_email_restoration(self, persistent_track, is_track_passed):
        """
        Восстановление по ссылке / коду восстановления из автоматического email-сообщения
        """
        with self.track_transaction.commit_on_error():
            if not is_track_passed:
                # Заполним необходимые поля для нового трека
                self.track.user_entered_login = persistent_track.content['user_entered_login']
                self.track.login = self.account.login
                self.track.uid = self.account.uid
                self.track.retpath = persistent_track.content['retpath']
                self.track.restore_state = RESTORE_STATE_METHOD_SELECTED
                self.track.current_restore_method = RESTORE_METHOD_EMAIL
            email = persistent_track.content['user_entered_email']
            self.track.user_entered_email = email

            self.fill_response_with_track_fields('track_id', 'user_entered_login')
            self.statbox.bind_context(
                login=self.track.login,
                retpath=self.track.retpath,
                current_restore_method=self.track.current_restore_method,
                initiator_track_id=persistent_track.content['initiator_track_id'],
            )
            self.assert_restore_method_suitable(method=RESTORE_METHOD_EMAIL)
            self.assert_email_still_suitable_for_restore(email)

            self.track.is_email_check_passed = True
            self.track.restoration_key_created_at = int(datetime_to_unixtime(persistent_track.created))
            self.mark_method_passed()

    def process_support_link_restoration(self, persistent_track):
        """
        Восстановление по ссылке от саппорта
        """
        link_type = persistent_track.content['support_link_type']
        self.statbox.bind_context(support_link_type=link_type)
        self.assert_support_link_type_valid_for_account(link_type)

        with self.track_transaction.commit_on_error():
            self.track.user_entered_login = self.track.login = self.account.login
            self.track.uid = self.account.uid
            self.track.support_link_type = link_type
            self.track.restoration_key_created_at = int(datetime_to_unixtime(persistent_track.created))
            self.track.restore_state = RESTORE_STATE_METHOD_SELECTED
            self.track.current_restore_method = {
                SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION: RESTORE_METHOD_HINT,
                SUPPORT_LINK_TYPE_FORCE_PHONE_RESTORATION: RESTORE_METHOD_PHONE,
                SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD: RESTORE_METHOD_LINK,
            }[link_type]
            self.statbox.bind_context(current_restore_method=self.track.current_restore_method)
            self.fill_response_with_track_fields('track_id', 'user_entered_login')

            if link_type == SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD:
                self.mark_method_passed()

    def get_login_from_persistent_track(self, persistent_track):
        if persistent_track.type == TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK:
            return persistent_track.content['user_entered_login']


class RestoreGetStateView(BaseRestoreView):
    """
    Получить информацию о процессе восстановления:
    - доступные методы восстановления
    - текущий выбранный метод восстановления
    - состояние для текущего метода
    - состояние для страницы ввода новых данных (при условии что метод восстановления успешно пройден)
    """
    restore_step = 'get_state'

    def get_phone_state(self):
        return {
            'phone_entered': self.track.user_entered_phone_number,
            'is_phone_suitable_for_restore': bool(self.track.phone_confirmation_code),
            'is_phone_confirmed': bool(self.is_phone_confirmed_in_track()),
        }

    def get_phone_and_2fa_factor_state(self):
        state = self.get_phone_state()
        state.update({
            # Показывает последний шаг пользователя - актуально для второго фактора 2ФА-восстановления
            'last_method_step': self.track.last_restore_method_step,
            # Единственный счетчик, который отдаем наружу, так как возможность проверки пина
            # блокируется после исчерпания попыток.
            'pin_checks_left': self.get_pin_checks_left(),
            'is_captcha_required': self.is_captcha_required_for_2fa_form(),
            'question': None,
        })
        if self.account.hint.is_set:
            state['question'] = self.account.hint.question.text
        return state

    def get_email_state(self):
        return {
            'email_entered': self.track.user_entered_email,
            'is_email_suitable_for_restore': bool(self.track.is_email_check_passed),
        }

    def get_hint_state(self):
        return {
            'question': self.account.hint.question.text,
            'is_captcha_required': self.is_captcha_required_for_hint(),
        }

    def fill_response_with_method_state(self, current_method):
        method_state_getter = getattr(self, 'get_%s_state' % current_method, lambda *args: {})
        self.response_values.update(
            current_restore_method=current_method,
            method_state=method_state_getter(),
        )

    def get_new_phone_state(self):
        new_phone = is_new_phone_confirmed = None
        if RESTORE_METHOD_PHONE in self.allowed_methods_to_bind:
            is_new_phone_confirmed = bool(self.is_phone_confirmed_in_track())
            # В процессе восстановления подтверждение телефона может происходить при восстановлении
            # по телефону либо при привязке нового телефона. В треке состояние отличается тем, что
            # при привязке нового телефона заполняется поле phone_confirmation_phone_number, а при
            # подтверждении защищенного номера - secure_phone_number.
            if self.track.phone_confirmation_phone_number:
                number = parse_phone_number(self.track.user_entered_phone_number, country=self.track.country)
                new_phone = dump_number(number)
        return new_phone, is_new_phone_confirmed

    def fill_response_with_new_auth_state(self):
        flushed_entities = []
        if self.account.totp_secret.is_set:
            # Предупреждаем о выключении 2ФА
            flushed_entities.append('2fa')

        new_phone, is_new_phone_confirmed = self.get_new_phone_state()
        self.response_values.update(
            new_auth_state=dict(
                # Нужно ли показывать страницу с предупреждением о сбросе - при восстановлении
                # по ссылке от саппортов не нужно
                show_method_passed_page=not bool(self.track.support_link_type),
                # Предупреждение о сбросе сущностей на аккаунте
                flushed_entities=flushed_entities,
                # Информация о том, что будет отозвано по умолчанию
                revokers=default_revokers(
                    allow_select=not self.is_glogout_required,
                ),
                # Является ли привязка средства восстановления обязательной
                is_method_binding_required=self.is_method_binding_required,
                # Допустимые к привязке средства восстановления
                allowed_methods_to_bind=self.allowed_methods_to_bind,
                # Состояние привязки нового телефона (если применимо)
                is_new_phone_confirmed=is_new_phone_confirmed,
                new_phone=new_phone,
                # Выставлено ли на аккаунте требование смены пароля
                is_forced_password_changing_pending=bool(self.account.password.is_changing_required),
            ),
            restore_method_passed=True,
        )

    def fill_response_with_restore_finished_state(self):
        response = dict(
            next_track_id=self.track.next_track_id,
            retpath=self.track.retpath,
            state=self.track.state,
        )
        self.response_values.update(
            current_restore_method=self.track.current_restore_method,
            restore_finished=True,
            has_secure_phone_number=bool(self.track.has_secure_phone_number),
            has_restorable_email=bool(self.track.has_restorable_email),
            is_restore_passed_by_support_link=bool(self.track.support_link_type),
            **{key: value for (key, value) in response.items() if value}
        )

    def process_request(self):
        self.read_track()
        self.assert_track_valid(
            allowed_states=(
                RESTORE_STATE_SUBMIT_PASSED,
                RESTORE_STATE_METHOD_SELECTED,
                RESTORE_STATE_METHOD_PASSED,
                RESTORE_STATE_COMMIT_PASSED,
            ),
        )
        self.fill_basic_response()
        if self.track.restore_state == RESTORE_STATE_COMMIT_PASSED:
            # После прохождения восстановления, показываем набор данных из трека
            self.fill_response_with_restore_finished_state()
            return

        redirect_required = (
            self.check_global_counters(finish_with_state=True) or
            self.get_and_validate_account_from_track()
        )
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()

        suitable_methods = self.get_suitable_restore_methods()
        if not suitable_methods:
            # Нет доступных средств восстановления - возвращаем состояние, установленное в get_suitable_restore_method
            # Такая ситуация может возникнуть для ПДД-пользователя с требованием сильного пароля и с необслуживаемым
            # саппортами доменом
            return

        # Эти поля обязательно должны быть в ответе для того, чтобы фронтенд мог что-то предложить
        # пользователю в случае ошибки
        self.response_values.update(
            # Возвращаем подходящие для аккаунта методы восстановления
            suitable_restore_methods=suitable_methods,
            restore_method_passed=False,
        )

        current_method = self.track.current_restore_method
        if current_method:
            self.assert_restore_method_suitable(suitable_methods=suitable_methods)

            # Теперь мы уверены, что пользователь может продолжать процедуру восстановления данным методом
            # При этом мы не проверяем, что для текущего метода доступны попытки, так как пользователь может быть
            # в середине процесса
            self.fill_response_with_method_state(current_method)

            method_passed = self.track.restore_state == RESTORE_STATE_METHOD_PASSED
            if method_passed:
                # Способ восстановления успешно пройден, нужно в ответе выдать информацию
                # для страницы ввода новых данных
                self.fill_response_with_new_auth_state()

        self.statbox.log(action='passed')


class RestoreSelectMethodView(BaseRestoreView):
    """
    Пользователь выбрал, каким способом восстановления он хочет воспользоваться (возможно, это сделал неявно фронтенд).
    """
    basic_form = RestoreSelectMethodForm
    restore_step = 'select_method'

    def update_restore_method_select_statistics(self, method):
        """
        Обновить статистику выбора метода восстановления.
        """
        selected_methods_counters = self.track.restore_methods_select_counters or {}
        selected_methods_counters[method] = selected_methods_counters.get(method, 0) + 1
        self.track.restore_methods_select_counters = selected_methods_counters
        self.track.restore_methods_select_order.append(method)
        self.statbox.bind_context(self.get_statistics_for_statbox(), current_restore_method=method)

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=(
                RESTORE_STATE_SUBMIT_PASSED,
                RESTORE_STATE_METHOD_SELECTED,
                RESTORE_STATE_METHOD_PASSED,
            ),
        )
        self.fill_basic_response()

        method = self.form_values['method']
        self.statbox.bind_context(previous_restore_method=self.track.current_restore_method)
        self.check_global_counters()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()

        if self.track.current_restore_method == method:
            # Данный способ восстановления и так активен, ничего не делаем
            self.fill_response_with_track_fields('track_id')
            return

        if self.track.current_restore_method == RESTORE_METHOD_SEMI_AUTO_FORM:
            # Если анкета уже выбрана, другой способ восстановления нельзя выбрать
            self.raise_error_with_logging(RestoreMethodNotAllowedError)
        suitable_methods = self.get_suitable_restore_methods()
        self.assert_restore_method_suitable(method=method, suitable_methods=suitable_methods)

        with self.track_transaction.rollback_on_error():
            self.update_restore_method_select_statistics(method)
            self.track.restore_state = RESTORE_STATE_METHOD_SELECTED
            self.track.current_restore_method = method
            if method == RESTORE_METHOD_SEMI_AUTO_FORM:
                # Настраиваем трек для анкеты восстановления
                request_source = settings.RESTORE_REQUEST_SOURCE_FOR_AUTO_RESTORE_WITH_METHODS
                if suitable_methods == [RESTORE_METHOD_SEMI_AUTO_FORM]:
                    # У пользователя нет средств восстановления помимо анкеты
                    request_source = settings.RESTORE_REQUEST_SOURCE_FOR_AUTO_RESTORE
                self.setup_track_for_semi_auto_form(
                    self.track.user_entered_login,
                    request_source,
                )
        self.fill_response_with_track_fields('track_id')
        self.statbox.log(action='passed')


class RestoreCommitView(BaseRestoreView,
                        BundlePasswordChangeMixin,
                        BundlePasswordValidationMixin,
                        BundleFixPDDRetpathMixin,
                        BundleBaseAuthorizationMixin,
                        BundleAccountResponseRendererMixin,
                        BindRelatedPhonishAccountMixin,
                        BundleAccountFlushMixin,
                        KolmogorMixin,
                        BundlePushMixin,
                        MailMixin,
                        UserMetaDataMixin):
    """
    Что происходит в ручке commit.
    Глобально нам нужно сменить пароль на аккаунте и подготовиться к выписыванию новой сессии.
    Основные шаги следующие:
    1. Проверить предусловия - метод восстановления пройден и всё еще применим для аккаунта.
    2. Провалидировать введенный пароль и другие данные.
    3. Выключить 2ФА (если включена), сменить пароль, сделав glogout. При этом протухнут
    OAuth-токены и пароли приложений. Установить на аккаунте новые данные при необходимости (телефон, КВ/КО и т.п.).
    4. Подготовить трек для вызова ручки создания сессии.
    5. Записать в логи.
    """

    basic_form = RestoreCommitForm
    restore_step = 'commit'

    def send_email_notifications(self, is_it_2fa_restore, are_emails_flushed):
        passed_method = self.track.current_restore_method
        if is_it_2fa_restore:
            language = get_preferred_language(self.account)
            template_name, info, context = get_2fa_email_notification_data(self.host, self.account, language)
            send_mail_for_account(
                template_name,
                info,
                context,
                self.account,
                login_shadower,
                # Если данные были сброшены, на аккаунте это не отражено. В этом случае отправляем письмо только
                # на нативные адреса (если среди них есть телефонный алиас и он был удален, письмо не дойдет)
                send_to_external=not are_emails_flushed,
            )
        elif (
            not self.track.support_link_type and
            passed_method in {
                RESTORE_METHOD_EMAIL,
                RESTORE_METHOD_HINT,
                RESTORE_METHOD_PHONE,
                RESTORE_METHOD_SEMI_AUTO_FORM,
            }
        ):
            # Письмо о прохождении восстановления отправляем для случаев автоматического прохождения по
            # КВ/КО, телефону, email-у и анкете.
            language = get_preferred_language(self.account)
            template_name, info, context = get_restore_passed_email_notification_data(
                self.host,
                self.account,
                language,
                passed_method,
                email=self.track.user_entered_email,
                phone=self.track.secure_phone_number,
            )
            send_mail_for_account(
                template_name,
                info,
                context,
                self.account,
                restore_passed_message_context_shadower,
            )

    def get_used_method_event_item(self):
        """
        Возвращает ключ и значение использованного способа восстановления для записи в event-лог
        """
        method = self.track.current_restore_method
        entity = method
        value = None
        if method in PHONE_BASED_RESTORE_METHODS:
            entity = 'phone'
            value = self.track.secure_phone_number
        elif method == RESTORE_METHOD_EMAIL:
            value = self.track.user_entered_email
        elif method == RESTORE_METHOD_HINT:
            entity = 'question'
            value = smart_text(self.account.hint.question)
        if value:
            return {'info.used_%s' % entity: value}

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_PASSED],
            allowed_methods=ALL_RESTORE_METHODS,
        )
        self.fill_basic_response()

        redirect_required = self.get_and_validate_account_from_track(check_child_family=True)
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        self.assert_new_method_passed_if_required()

        # Валидируем пароль. Трек для валидации заполняется ранее, при успешном прохождении способа восстановления.
        password, password_quality = self.validate_password()

        is_it_2fa_restore = self.account.totp_secret.is_set
        entities_to_flush = self.get_flushed_entities()

        # Получаем информацию о привязываемом средстве восстановления
        is_hint_passed = new_phone = None
        new_method = self.form_values['new_method']
        if new_method == RESTORE_METHOD_PHONE:
            new_phone = PhoneNumber.parse(self.track.phone_confirmation_phone_number)
        elif new_method == RESTORE_METHOD_HINT:
            is_hint_passed = True

        used_method_event_item = self.get_used_method_event_item()
        with self.track_transaction.rollback_on_error():
            # Удаление сущностей: телефоны, email-адреса, соц. профили, КВ/КО
            # TODO: Выпилить этот уродливый хак после исправления send_mail_for_account
            # Проблема заключается в том, что send_mail_for_account
            # предполагает, что на аккаунте всегда есть адреса. Это
            # было обоснованно в старой модели поведения, адреса были
            # внешней сущностью. Теперь же адреса на аккаунте напрямую
            # связаны с адресами в базе.
            account_emails = self.account.emails
            self.flush_entities(entities_to_flush)
            self.account.emails = account_emails
            # Начало процесса привязки нового телефона, если он задан
            if new_phone:
                phone_binder = self.build_bind_securest_possible_phone(
                    account=self.account,
                    phone_number=self.saved_number,
                )
                phone_binder.submit()

            # Смена пароля, установка новых данных
            events = {
                EVENT_ACTION: 'restore_passed_by_%s' % self.track.current_restore_method,
                'consumer': self.consumer,
            }
            if self.track.support_link_type:
                events[EVENT_INFO_SUPPORT_LINK_TYPE] = self.track.support_link_type
            if used_method_event_item:
                events.update(used_method_event_item)
            with UPDATE(self.account, self.request.env, events):
                if is_it_2fa_restore:
                    # Выключаем 2ФА на аккаунте (могли уже выключить, если выполнялся сброс телефонов)
                    if self.account.totp_secret is not None:
                        self.send_account_modification_push(
                            event_name='login_method_change',
                            context=dict(track_id=self.track_id),
                        )
                        self.send_account_modification_mail(
                            event_name='login_method_change',
                        )
                    self.account.totp_secret = None
                self.change_password(
                    password,
                    password_quality,
                    global_logout=self.is_glogout_required,
                    revoke_web_sessions=self.form_values['revoke_web_sessions'],
                    revoke_tokens=self.form_values['revoke_tokens'],
                    revoke_app_passwords=self.form_values['revoke_app_passwords'],
                )
                if self.track.current_restore_method in PHONE_BASED_RESTORE_METHODS:
                    # Отменим операцию, удаляющую защищенный номер, при восстановлении по этому номеру
                    self.cancel_quarantined_phone_operations()

                if new_phone:
                    phone_binder.commit()

                    if (
                        phone_binder.is_binding_secure() or
                        phone_binder.is_starting_replacement() or
                        phone_binder.is_replacing()
                    ):
                        # Если телефон удалось привязать как защищенный (в т.ч. через карантин), увеличиваем счетчик.
                        get_per_phone_number_buckets().incr(new_phone.e164)

                    self.statbox.bind_context(self.phone_binding_stats(phone_binder))
                    self.response_values['new_phone_status'] = self.securest_possible_phone_binding_response(phone_binder)

                if is_hint_passed:
                    self.account.hint = Hint(parent=self.account)
                    self.account.hint.question = make_hint_question(
                        question=self.form_values['question'],
                        question_id=self.form_values['question_id'],
                        display_language=self.form_values['display_language'],
                    )
                    self.account.hint.answer = self.form_values['answer']
                if not self.account.is_user_enabled:
                    # Разблокируем аккаунт в следующих случаях:
                    # - восстановление по блокирующей саппортской ссылке
                    # - восстановление заблокированного при удалении аккаунта; при этом операция удаления
                    #   будет подчищена в сборщике удаленных аккаунтов
                    if self.account.disabled_status == ACCOUNT_DISABLED_ON_DELETION:
                        if self.account.is_child and not self.account.has_family:
                            # Ребёнка нужно вернуть в семью, а потом уже разблокировать
                            place = self.get_family_member_free_place()
                            with UPDATE(
                                self.family_info,
                                self.request.env,
                                events=dict(
                                    action='family_restore_child',
                                    consumer=self.consumer,
                                ),
                            ):
                                self.family_info.add_member_uid(self.account.uid, place)

                            self.account.last_child_family = None

                        self.statbox.bind_context(restored_from_deletion=True)
                    self.account.is_enabled = True
                process_env_profile(self.account, track=self.track)
                reset_challenge_counters(self.account)

                if self.track.support_link_type:
                    if self.account.karma.prefix in settings.KARMA_PREFIX_TO_WASH_BY_SUPPORT_LINK:
                        self.account.karma.prefix = settings.KARMA_PREFIX_WASHED_BY_SUPPORT

            # Окончание процесса привязки телефона
            if new_phone:
                phone_binder.after_commit()

            # Выставляем флаги в треке для последующего выписывания сессии
            # Лайтам не разрешаем выписать сессию, потребуем дорегистрацию (в зависимости от настройки)
            is_forced_lite_completion_pending = (
                settings.RESTORE_TO_FORCED_LITE_COMPLETION_ENABLED and self.account.is_lite
            )
            if not is_forced_lite_completion_pending:
                set_authorization_track_fields(
                    self.account,
                    self.track,
                    allow_create_session=True,
                    allow_create_token=False,
                    is_2fa_restore_passed=is_it_2fa_restore,
                    password_passed=True,
                    session_scope=SessionScope.xsession,
                )

            # Записываем в трек признак успешного завершения восстановления
            self.track.restore_state = RESTORE_STATE_COMMIT_PASSED

            # Идем явно в self.account.phones.secure, т.к. обёртка account_phones не учитывает
            # возможные изменения телефонов.
            if is_it_2fa_restore and self.account.phones.secure:
                # Промо упрощенного включения 2ФА
                self.setup_2fa_promo()
            elif is_forced_lite_completion_pending:
                # Принудительная дорегистрация лайта
                self.setup_forced_lite_completion()

            # Сохраняем флаги, на основе которых фронтенд сможет показывать промо привязки телефона после
            # окончания процесса восстановления
            self.setup_new_restore_method_promo(entities_to_flush)

            # Если мы в процессе переподтверждали уже привязанный телефон - запомним это в треке
            if (
                self.track.phone_confirmation_is_confirmed and
                self.track.secure_phone_number and
                not self.track.phone_confirmation_phone_number
            ):
                self.track.phone_confirmation_phone_number = self.track.secure_phone_number

        self.send_email_notifications(
            is_it_2fa_restore=is_it_2fa_restore,
            are_emails_flushed='emails' in entities_to_flush,
        )
        write_phone_to_log(self.account, self.cookies, phone=new_phone.e164 if new_phone else None)
        self.send_account_modification_push(event_name='changed_password')
        self.send_account_modification_push(
            event_name='restore',
            context=dict(track_id=self.track_id),
        )
        self.try_bind_related_phonish_account()
        self.statbox.log(action='passed')

    def validate_password(self):
        try:
            phone_number_for_validation = (
                # Новый защищенный номер, привязываемый на восстановлении
                self.track.phone_confirmation_phone_number or
                # Защищенный номер, использованный в качестве средства восстановления
                self.track.secure_phone_number or
                # Защищенный номер на аккаунте
                (self.account_phones.secure.number if self.account_phones.secure else None)
            )
            return super(RestoreCommitView, self).validate_password(
                is_strong_policy=self.account.is_strong_password_required,
                old_password_hash=self.account.password.serialized_encrypted_password,
                required_check_password_history=True,
                emails=set([email.address for email in self.account.emails.native]),
                phone_number=phone_number_for_validation,
                country=self.account.person.country,
            )
        except ValidationFailedError:
            self.statbox.log(action='finished_with_error', error='password.invalid')
            raise

    def assert_new_method_passed_if_required(self):
        """
        Проверяем, что в случае необходимости привязки нового средства восстановления, это средство передано.
        В случае добровольной привязки нового средства, проверяем, что состояние в треке корректное.
        """
        form = RestoreCommitNewMethodForm(
            self.allowed_methods_to_bind,
            is_new_method_required=self.is_method_binding_required,
        )
        self.form_values.update(self.process_form(form, self.all_values))
        new_method = self.form_values['new_method']
        if (
            new_method == RESTORE_METHOD_PHONE and
            (not self.is_phone_confirmed_in_track() or not self.track.phone_confirmation_phone_number)
        ):
            raise InvalidTrackStateError()
        if new_method == RESTORE_METHOD_PHONE:
            self.assert_phone_not_compromised(self.track.phone_confirmation_phone_number)
        self.statbox.bind_context(
            new_method=new_method,
            is_new_method_required=self.is_method_binding_required,
            allowed_methods_to_bind=','.join(self.allowed_methods_to_bind),
        )

    def get_flushed_entities(self):
        """
        Определяем сущности, которые необходимо сбросить. В случае восстановления по саппортской
        ссылке, сущности определяются ссылкой. В случае восстановления по анкете, сбрасываем все
        средства входа в аккаунт, кроме телефона, который (при наличии) заменяем через карантин.
        """
        if self.track.support_link_type:
            return self.support_link_options.entities_to_flush
        if self.track.current_restore_method == RESTORE_METHOD_SEMI_AUTO_FORM:
            return {'emails', 'social_profiles', 'hint'}
        return set()

    def flush_entities(self, entities):
        if not entities:
            return
        events = {
            EVENT_ACTION: ACTION_RESTORE_ENTITIES_FLUSHED,
            EVENT_INFO_FLUSHED_ENTITIES: ','.join(sorted(entities)),
            'consumer': self.consumer,
        }

        with UPDATE(self.account, self.request.env, events, force_history_db_external_events=True):
            if 'emails' in entities:
                self.drop_emails()
            if 'phones' in entities:
                if self.account.totp_secret and self.account.totp_secret.is_set:
                    # Выключаем 2ФА на аккаунте вместе со сбросом телефонов
                    self.account.totp_secret = None
                    self.send_account_modification_push(
                        event_name='login_method_change',
                        context=dict(track_id=self.track_id),
                    )
                    self.send_account_modification_mail(
                        event_name='login_method_change',
                    )
                if self.account.sms_2fa_on:
                    # Выключаем смс-2ФА на аккаунте вместе со сбросом телефонов
                    self.account.sms_2fa_on = False  # игнорируем атрибут forbid_disabling_sms_2fa
                self.drop_phones(drop_bank_phone=False, unsecure_bank_phone=True)
            if 'hint' in entities:
                self.account.hint = None
            if 'social_profiles' in entities:
                self.drop_social_profiles()

    def cancel_quarantined_phone_operations(self):
        confirmed_phone = self.account.phones.by_number(self.track.secure_phone_number)
        if confirmed_phone is self.account.phones.secure:
            logical_op = confirmed_phone.get_logical_operation(self.statbox)
            if type(logical_op) in (
                ReplaceSecurePhoneWithNonboundPhoneOperation,
                ReplaceSecurePhoneWithBoundPhoneOperation,
                RemoveSecureOperation,
            ):
                logical_op.cancel()

    def setup_2fa_promo(self):
        # Для случая 2ФА восстановления, при наличии защищенного телефона, создаем авторизационный
        # трек с признаком прохождения восстановления, для упрощения повторного включения 2ФА.
        auth_track = self.track_manager.create('authorize', self.consumer)
        with self.track_manager.transaction(track=auth_track).rollback_on_error() as auth_track:
            auth_track.is_otp_restore_passed = True
            auth_track.retpath = self.track.retpath

        self.track.next_track_id = auth_track.track_id
        self.track.state = Show2FAPromo.state

    def setup_forced_lite_completion(self):
        # Запустим процедуру принудительной дорегистрации лайта
        auth_track = self.track_manager.create('authorize', self.consumer)
        with self.track_manager.transaction(track=auth_track).rollback_on_error() as auth_track:
            auth_track.is_force_complete_lite = True
            auth_track.uid = self.account.uid
            auth_track.login = self.account.login
            auth_track.have_password = True
            auth_track.is_password_passed = True
            auth_track.password_hash = get_sha256_hash(self.form_values['password'])
            auth_track.retpath = self.track.retpath

        self.track.next_track_id = auth_track.track_id
        self.track.state = RedirectToForcedLiteCompletion.state

    def setup_new_restore_method_promo(self, entities_to_flush):
        has_restorable_email = (
            'emails' not in entities_to_flush and
            bool(self.account.emails.suitable_for_restore)
        )

        self.track.has_secure_phone_number = bool(self.account.phones.secure)
        if self.track.has_secure_phone_number:
            self.track.secure_phone_number = self.account.phones.secure.number.e164
        self.track.has_restorable_email = has_restorable_email

    def phone_binding_stats(self, phone_binder):
        stats = self.securest_possible_phone_binding_response(phone_binder)
        stats.pop('secure_phone_pending_until', None)
        return stats


class RestoreLoginSuggestViewV1(BaseBundleView, BundleAccountGetterMixin):
    """
    Саджест логина при восстановлении (аккаунта или логина)
    """
    require_track = True

    allowed_processes = [PROCESS_RESTORE, PROCESS_LOGIN_RESTORE]

    required_grants = [RESTORE_BASE_GRANT]

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
    )

    def _get_logins_from_session(self):
        """
        Попробуем получить логины из сессионной куки.
        Не проверяем валидность пользователей в ответе, нас интересуют любые логины.
        Не проверяем применимость процедуры восстановления для логина - сообщения об ошибках
        на фронтенде содержат ценную информацию о дальнейших шагах пользователя.
        """
        logins = set()
        sessionid, ssl_sessionid, host = self.get_effective_session_and_host()
        if not sessionid:
            return logins
        try:
            to_statbox({
                'mode': 'check_cookies',
                'host': self.request.env.host,
                'consumer': self.consumer,
                'have_sessguard': self.get_sessguard() is not None,
                'sessionid': mask_sessionid(sessionid),
            })
            bb_response = self.blackbox.sessionid(
                sessionid=sessionid,
                ip=self.client_ip,
                host=host,
                sslsessionid=ssl_sessionid,
                sessguard=self.get_sessguard(),
                multisession=True,
                request_id=self.request.env.request_id,
            )
        except BaseBlackboxError:
            return logins

        multi_session_users = bb_response.get('users', {})
        for user_info in multi_session_users.values():
            try:
                account = Account().parse(user_info)
                if account.display_login:
                    logins.add(account.display_login)
            except UnknownUid:
                pass
        return logins

    def _get_suggested_logins(self):
        """
        Саджест получаем на основе сессионной куки и куки L; yandex_login не используем - см. PASSP-12573
        """
        suggested_logins = set()
        try:
            cookie_info = CookieL().unpack(self.cookies.get('L'))
            l_login = cookie_info['login']
        except CookieLUnpackError as e:
            log.debug('Error while unpacking cookie: %s', e.message)
            l_login = None
        if l_login:
            suggested_logins.add(l_login)
        suggested_logins.update(self._get_logins_from_session())

        return list(suggested_logins)

    def _write_suggest_info_to_track(self, logins):
        with self.track_transaction.commit_on_error():
            self.track.suggested_logins.append(*logins)
            self.track.suggest_login_count.incr()
            if self.track.suggest_login_first_call is None:
                self.track.suggest_login_first_call = time.time()
            self.track.suggest_login_last_call = time.time()

    def process_request(self):
        self.read_track()

        suggested_logins = self._get_suggested_logins()
        self._write_suggest_info_to_track(suggested_logins)

        self.response_values['suggested_logins'] = suggested_logins


class RestoreLoginSuggestView(RestoreLoginSuggestViewV1):
    def _update_suggest_with_account(self, suggest, account):
        suggest_name = account.display_login
        if not suggest_name and account.person.display_name:
            suggest_name = account.person.display_name.name

        if suggest_name:
            suggest_login = account.display_login or account.login
            info = {
                'login': suggest_login,
                'suggest_name': suggest_name,
            }
            if account.person and account.person.default_avatar:
                info['default_avatar'] = account.person.default_avatar
            suggest[suggest_login] = info

    def _suggest_from_session(self):
        """
        Попробуем получить логины из сессионной куки.
        Не проверяем валидность пользователей в ответе, нас интересуют любые логины.
        Не проверяем применимость процедуры восстановления для логина - сообщения об ошибках
        на фронтенде содержат ценную информацию о дальнейших шагах пользователя.
        """
        suggest = {}
        sessionid, ssl_sessionid, host = self.get_effective_session_and_host()
        if not sessionid:
            return suggest
        try:
            to_statbox({
                'mode': 'check_cookies',
                'host': self.request.env.host,
                'consumer': self.consumer,
                'have_sessguard': self.get_sessguard() is not None,
                'sessionid': mask_sessionid(sessionid),
            })
            bb_response = self.blackbox.sessionid(
                sessionid=sessionid,
                ip=self.client_ip,
                host=host,
                sslsessionid=ssl_sessionid,
                sessguard=self.get_sessguard(),
                multisession=True,
                request_id=self.request.env.request_id,
            )
        except BaseBlackboxError:
            return suggest

        multi_session_users = bb_response.get('users', {})
        for user_info in multi_session_users.values():
            try:
                account = Account().parse(user_info)
                self._update_suggest_with_account(suggest, account)
            except UnknownUid:
                pass
        return suggest

    def _suggest_from_cookie_l(self, suggest):
        cookie_l_value = self.cookies.get('L')
        if not cookie_l_value:
            return suggest
        try:
            cookie_info = CookieL().unpack(cookie_l_value)
            l_login = cookie_info['login']
            if not l_login or l_login in suggest:
                return suggest
            self.get_account_by_login(
                l_login,
                enabled_required=False,
                find_by_phone_alias=None,
            )
            self._update_suggest_with_account(suggest, self.account)
        except CookieLUnpackError as e:
            log.debug('Error while unpacking cookie: %s', e.message)
        except BaseBlackboxError:
            pass

        return suggest

    def _suggest_logins(self):
        suggest = self._suggest_from_session()
        return self._suggest_from_cookie_l(suggest)

    def process_request(self):
        self.read_track()

        suggest = self._suggest_logins()
        self._write_suggest_info_to_track(suggest.keys())

        self.response_values['suggested_logins'] = sorted(
            suggest.values(),
            key=lambda item: item['suggest_name'],
        )


class RestoreCreateLinkView(BaseRestoreView, BundlePersistentTrackMixin, BundleAdminActionMixin):
    require_track = False

    require_process = False

    required_grants = [RESTORE_CREATE_LINK_GRANT, RESTORE_BASE_GRANT]

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_HOST,
    )

    basic_form = RestoreCreateLinkForm
    restore_step = 'create_link'

    def process_request(self):
        self.process_basic_form()
        self.statbox.bind_context(
            uid=self.form_values['uid'],
            support_link_type=self.form_values['link_type'],
        )

        link_type = self.form_values['link_type']
        link_options = SupportLinkOptions(link_type)
        is_completion_link = link_type == SUPPORT_LINK_TYPE_REDIRECT_TO_COMPLETION
        redirect_required = self.get_and_validate_account(
            uid=self.form_values['uid'],
            emails=True,
            need_phones=True,
            check_disabled_status=not link_options.is_disabled_account_allowed,
            allow_missing_password_with_portal_alias=link_options.is_missing_password_with_portal_alias_allowed,
            # Два случая дорегистрации допускаются только для ссылки соответствующего типа
            allow_social_missing_password_with_portal_alias=is_completion_link,
            check_autoregistered=not is_completion_link,
            # Саппорты не имеют права выписывать ссылки ПДД-пользователям не из списка обслуживаемых саппортами
            check_domain_support=True,
        )
        if redirect_required:
            return
        self.assert_support_link_type_valid_for_account(link_type)

        persistent_track_type = TRACK_TYPE_RESTORATION_SUPPORT_LINK
        if is_completion_link:
            # Для дорегистрации создаем ссылку на существующий механизм дорегистраций
            persistent_track_type = TRACK_TYPE_AUTH_BY_KEY_LINK
        persistent_track = self.create_persistent_track(
            self.account.uid,
            persistent_track_type,
            expires_after=settings.RESTORATION_MANUAL_LINK_LIFETIME_SECONDS,
            support_link_type=link_type,
        )

        events = {
            EVENT_ACTION: ACTION_RESTORE_SUPPORT_LINK_CREATED,
            EVENT_INFO_SUPPORT_LINK_TYPE: link_type,
        }
        self.mark_admin_action(events)
        with UPDATE(self.account, self.request.env, events, force_history_db_external_events=True):
            if link_type in BLOCKING_SUPPORT_LINK_TYPES:
                self.account.is_enabled = False

        tld = get_tld_by_country(self.account.person.country)
        self.response_values.update(secret_link=link_options.key_link_getter(persistent_track.track_key, tld))
        self.statbox.log(action='passed')
