# -*- coding: utf-8 -*-
# Описание данных, генерируемых обработчиками в данном модуле, доступно в тестах:
# tests/views/bundle/restore/semi_auto/test_restore_data_with_documentation.py
import base64
from collections import OrderedDict
from datetime import (
    datetime,
    timedelta,
)
import logging
import re
import time

from namedlist import namedlist
from passport.backend.api.common.account import safe_detect_timezone
from passport.backend.api.common.historydb_api import (
    AuthsAggregatedRuntimeInfo,
    find_password_in_history,
    get_account_registration_unixtime,
    get_historydb_events_info,
)
from passport.backend.api.common.mail_api import (
    get_blackwhite_lists,
    get_collectors,
    get_outbound_emails,
    get_user_email_folders,
)
from passport.backend.api.common.social_api import (
    get_profiles_with_person_by_uid,
    get_unique_social_profiles_by_task_ids,
)
from passport.backend.api.views.bundle.headers import HEADER_CLIENT_USER_AGENT
from passport.backend.api.yasms.api import Yasms
from passport.backend.core import (
    geobase,
    Undefined,
)
from passport.backend.core.builders.blackbox.blackbox import get_blackbox
from passport.backend.core.builders.yasms import get_yasms
from passport.backend.core.compare import (
    BIRTHDAYS_FACTOR_INEXACT_MATCH,
    BIRTHDAYS_FACTOR_NO_MATCH,
    calculate_timestamp_depth_factor,
    calculate_timestamp_interval_factor,
    compare_answers,
    compare_birthdays,
    compare_birthdays_inexact,
    compare_dates_loose,
    compare_emails,
    compare_ip_with_subnet,
    compare_ips,
    compare_lastname_with_names,
    compare_names,
    compare_phones,
    compare_strings,
    compare_uas,
    compress_string_factor,
    default_fuzzy_names_factor,
    FACTOR_BOOL_MATCH,
    FACTOR_BOOL_NO_MATCH,
    FACTOR_FLOAT_MATCH,
    FACTOR_NOT_SET,
    find_best_string_factor_index,
    serialize_names_factor,
    STRING_FACTOR_INEXACT_MATCH,
    STRING_FACTOR_NO_MATCH,
    string_result_to_simple_factor,
    UA_FACTOR_FULL_MATCH,
)
from passport.backend.core.conf import settings
from passport.backend.core.env.env import parse_uatraits_value
from passport.backend.core.historydb.analyzer.event_handlers.phone import flatten_phones_mapping
from passport.backend.core.historydb.analyzer.event_handlers.question_answer import (
    flatten_question_answer_mapping,
    serialize_question_answer_mapping,
)
from passport.backend.core.historydb.events import PASSWORD_CHANGE_TYPE_FORCED
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.person import MOSCOW_TIMEZONE
from passport.backend.core.portallib import detect_user_agent
from passport.backend.core.services import Service
from passport.backend.core.types.email.email import punycode_email
from passport.backend.core.types.mobile_device_info import MobileDeviceInfo
from passport.backend.core.types.phone import build_phones
from passport.backend.core.types.phone_number.phone_number import parse_phone_number
from passport.backend.core.types.question import Question
from passport.backend.utils.common import unique_preserve_order
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import (
    DATE_WITH_TZ_FORMAT,
    datetime_to_string,
    datetime_to_unixtime,
    safe_local_datetime_to_date,
)


log = logging.getLogger('passport.api.views.bundle.restore.factors')

# Имена сервисов, которые можно передать в ручку анкеты
ALLOWED_INPUT_SERVICE_NAMES = ['mail', 'yandsearch', 'disk', 'market', 'music', 'metrika']

# Отображение имен сервисов в сервисы, использование которых мы можем определить по наличию подписки.
# Подробнее см. PASSP-10415.
SERVICE_NAME_TO_SUBSCRIBED_SERVICE = {
    'mail': Service.by_slug('mail'),
    'disk': Service.by_slug('cloud'),  # Выставляется при посещении Я.Диска
    'metrika': Service.by_slug('metrika'),
}

SUBSCRIBED_SERVICE_TO_SERVICE_NAME = dict((service, service_name) for (
    service_name,
    service,
) in SERVICE_NAME_TO_SUBSCRIBED_SERVICE.items())

# Списки сервисов, которые мы обрабатываем
PROCESSED_SERVICES = set(SERVICE_NAME_TO_SUBSCRIBED_SERVICE.values())

MAIL_SID = Service.by_slug('mail').sid

MAX_PASSWORD_FIELDS = 3

PERSONAL_DATA_CHANGE_INDICES = (0, -2, -1)  # Для личных данных смотрим на первую и две последних смены
RESTORE_METHODS_CHANGE_INDICES = (-3, -2, -1)  # Для средств восстановления смотрим на последние три смены
# Для средств восстановления также смотрим первое и последнее окружение задания значения
# для найденных совпадений
MATCHED_RESTORE_METHODS_CHANGE_INDICES = (0, -1)


NamesBirthdayResult = namedlist(
    'CalculateFactorsResult',
    ('check_passed', 'matches'),
    default=None,
)

# Тип для хранения информации об окружении.
UserEnv = namedlist(
    'UserEnv',
    ('ip', 'subnet', 'ua', 'timestamp', 'entity', 'device_info'),
    default=None,
)

# Имена сущностей, используемые для сохранения факторов
NAMES_ENTITY_NAME = 'names'
BIRTHDAYS_ENTITY_NAME = 'birthday'
PASSWORDS_ENTITY_NAME = 'passwords'
ANSWERS_ENTITY_NAME = 'answer'
PHONE_NUMBERS_ENTITY_NAME = 'phone_numbers'


def encode_emails(emails):
    """
    Закодируем доменную часть в punycode.
    Если доменной части нет, ничего не делаем.
    """
    if not emails:
        # допустимы значения None (не помню) и ''(точно не было)
        return emails
    return [punycode_email(email) for email in emails]


def get_names_birthday_matches(factors):
    """
    Получение списка совпадений и признака совпадения ФИО/ДР.
    Необходимо для обратной совместимости.
    """
    statuses = [
        factors[NAMES_ENTITY_NAME]['account_status'],
        factors[NAMES_ENTITY_NAME]['history_status'],
        factors[BIRTHDAYS_ENTITY_NAME]['account_factor'] == FACTOR_BOOL_MATCH,
        factors[BIRTHDAYS_ENTITY_NAME]['history_factor'] == FACTOR_BOOL_MATCH,
    ]
    fields = [
        'names_current',
        'names_registration',
        'birthday',
        'birthday_registration',
    ]
    check_passed = any(statuses)
    matches = [
        field
        for (field, match_found) in zip(fields, statuses)
        if match_found
    ]

    return NamesBirthdayResult(
        check_passed=check_passed,
        matches=matches,
    )


def get_names_check_status(factors):
    return (
        factors[NAMES_ENTITY_NAME]['factor']['registration'][1] >= STRING_FACTOR_INEXACT_MATCH or
        factors[NAMES_ENTITY_NAME]['factor']['intermediate'][1] >= STRING_FACTOR_INEXACT_MATCH or
        factors[NAMES_ENTITY_NAME]['factor']['current'][1] >= STRING_FACTOR_INEXACT_MATCH
    )


def get_names_birthday_check_status(factors):
    return (
        get_names_check_status(factors) or
        factors[BIRTHDAYS_ENTITY_NAME]['factor']['registration'] >= BIRTHDAYS_FACTOR_INEXACT_MATCH or
        factors[BIRTHDAYS_ENTITY_NAME]['factor']['intermediate'] >= BIRTHDAYS_FACTOR_INEXACT_MATCH or
        factors[BIRTHDAYS_ENTITY_NAME]['factor']['current'] >= BIRTHDAYS_FACTOR_INEXACT_MATCH
    )


def get_user_env_check_status(factors):
    return any(
        factors['user_env_auths']['factor']['%s_first_auth_depth' % field] != FACTOR_NOT_SET
        for field in ('ip', 'subnet', 'ua')
    )


def names_factor_to_statbox_dict(prefix, factors):
    to_bind = {}
    for name in ('firstname', 'lastname'):
        to_bind['%s_%s' % (prefix, name)] = compress_string_factor(factors[name])
    return to_bind


def factor_list_to_statbox_dict(prefix, factor):
    to_bind = {}
    for index, value in enumerate(factor):
        key = '%s_%d' % (prefix, index)
        to_bind[key] = value
    return to_bind


def factor_dict_to_statbox_dict(prefix, factor, key_format='%s_factor_%s'):
    to_bind = {}
    for field, value in factor.items():
        key = key_format % (prefix, field)
        if isinstance(value, (tuple, list)):
            to_bind.update(factor_list_to_statbox_dict(key, value))
        elif isinstance(value, dict):
            to_bind.update(factor_dict_to_statbox_dict(key, value, key_format='%s_%s'))
        else:
            to_bind[key] = value
    return to_bind


def build_parsed_user_agent(yandexuid, headers=None, user_agent=None):
    """
    Построить представление UA (ОС, браузер, yandexuid)
    """
    headers = headers if headers is not None else {HEADER_CLIENT_USER_AGENT.name: user_agent}
    user_agent_info = detect_user_agent(headers)
    browser_name = parse_uatraits_value(user_agent_info.get('BrowserName'))
    os_name = parse_uatraits_value(user_agent_info.get('OSName', user_agent_info.get('OSFamily')))
    return {
        'os.name': os_name.lower() if os_name else os_name,
        'browser.name': browser_name.lower() if browser_name else browser_name,
        'yandexuid': yandexuid,
    }


def build_user_env(headers, yandexuid, user_ip, timestamp=None, entity=None, track=None):
    """
    Построить информацию об окружении в момент времени timestamp. Если timestamp не задан, будет использовано
    текущее время.
    """
    parsed_ua = build_parsed_user_agent(yandexuid, headers=headers)
    ip_lookup = do_ip_lookup(user_ip)
    device_info = MobileDeviceInfo.from_track(track) if track else None
    return UserEnv(
        ip=user_ip,
        subnet=ip_lookup.subnet if ip_lookup is not None else None,
        ua=parsed_ua,
        timestamp=timestamp if timestamp is not None else time.time(),
        entity=entity,
        device_info=device_info,
    )


def build_registration_env(events_info, events_info_cache, account):
    """
    Построить информацию об окружении в момент регистрации. Данные берутся либо из результатов работы HistoryDBApi,
    либо из кеша.
    """
    if 'registration_env' in events_info_cache:
        reg_env = events_info_cache['registration_env']
    else:
        reg_env = events_info.registration_env if events_info else {}
        if reg_env:
            # обновим значение в кеше
            events_info_cache['registration_env'] = reg_env
    return build_user_env(
        {HEADER_CLIENT_USER_AGENT.name: reg_env.get('user_agent')},
        reg_env.get('yandexuid'),
        reg_env.get('user_ip'),
        timestamp=get_account_registration_unixtime(account.registration_datetime),
    )


def build_change_env(point, entity=None):
    """
    Построить окружение на основании данных смены
    @param point: данные смены
    @param entity: имя изменяемой сущности, например, 'names'
    """
    return build_user_env(
        {HEADER_CLIENT_USER_AGENT.name: point.get('user_agent')},
        point.get('yandexuid'),
        point.get('user_ip'),
        timestamp=point['timestamp'],
        entity=entity,
    )


def do_ip_lookup(user_ip):
    return geobase.get_as_lookup().ip_lookup(str(user_ip)) if user_ip is not None else None


def make_depth_factor(now_ts, registration_ts, event_ts, fixed_threshold=None):
    """
    Вычислить фактор глубины для события
    """
    # могут быть ситуации, когда timestamp не известен
    if event_ts is None:
        return FACTOR_NOT_SET
    # event_ts может быть округлен до дня
    if event_ts < registration_ts and registration_ts - registration_ts % timedelta(days=1).total_seconds() == event_ts:
        event_ts = registration_ts
    return calculate_timestamp_depth_factor(
        registration_ts,
        now_ts,
        event_ts,
        max_depth_threshold=timedelta(days=settings.RESTORE_SEMI_AUTO_IP_CHECK_DEPTH).total_seconds(),
        fixed_threshold=fixed_threshold,
    )


class UserDataHandler(object):
    """
    Базовый класс для получения факторов для различных данных, введенных пользователем.
    """
    def __init__(self, form_values, account, user_env, statbox, calculated_factors, **kwargs):
        """
        Сохраним необходимые данные для работы.
        @param form_values: данные формы
        @param account: модель аккаунта
        @param user_env: информация об окружении пользователя (объект типа UserEnv)
        @param statbox: экземпляр StatboxLogger
        @param calculated_factors: содержит результаты вычисления всех факторов
        """
        self.form_values = form_values
        self.account = account
        self.user_env = user_env
        self.statbox = statbox
        self.calculated_factors = calculated_factors

    def handle(self):
        """
        Метод handle выполняет сравнение и получение факторов для конкретных данных, должен
        вернуть словарь, содержащий введенные данные и факторы в json-сериализуемом виде,
        а также записать необходимые данные в статбокс (сделать bind).
        """
        raise NotImplementedError()  # pragma: no cover


class HistoryDBEventsDataHandler(UserDataHandler):
    """
    Обработчик пользовательских данных, использующий данные о событиях HistoryDB.
    """
    def __init__(self, *args, **kwargs):
        """
        @param events_info: структурированные данные о событиях HistoryDB
        @param registration_env: информация об окружении при регистрации (объект типа UserEnv)
        """
        self.events_info = kwargs.pop('events_info')
        self.registration_env = kwargs.pop('registration_env')
        super(HistoryDBEventsDataHandler, self).__init__(*args, **kwargs)


class NamesHandler(HistoryDBEventsDataHandler):
    """
    Обработчик для сравнения ФИО с аккаунтом и историей.
    """
    def handle(self):
        result = {}
        names = self.form_values.get('firstname', ''), self.form_values['lastname']
        lastname = names[-1]
        person = self.account.person
        account_names = person.firstname, person.lastname

        account_result = compare_lastname_with_names(
            account_names,
            lastname,
            person.language,
        )

        history_firstname = self.events_info.firstnames[0] if self.events_info and self.events_info.firstnames else None
        history_lastname = self.events_info.lastnames[0] if self.events_info and self.events_info.lastnames else None
        historydb_events_found = history_firstname and history_lastname
        if historydb_events_found:
            historydb_names = history_firstname, history_lastname
            historydb_result = compare_lastname_with_names(
                historydb_names,
                lastname,
                person.language,
            )

        result['entered'] = names
        result['account'] = account_names
        result['account_status'] = account_result.status
        result['account_factor'] = serialize_names_factor(account_result.factors)
        self.statbox.bind(
            names_factor_to_statbox_dict('names_account_factor', account_result.factors),
            names_account_status=account_result.status,
            names_account_reason=', '.join(account_result.reasons),
            names_history_found=bool(historydb_events_found),
        )

        if historydb_events_found:
            result['history'] = historydb_names
            result['history_status'] = historydb_result.status
            result['history_factor'] = serialize_names_factor(historydb_result.factors)
            self.statbox.bind(
                names_factor_to_statbox_dict('names_history_factor', historydb_result.factors),
                names_history_status=historydb_result.status,
                names_history_reason=', '.join(historydb_result.reasons),
            )
        else:
            result['history'] = None
            result['history_status'] = False
            result['history_factor'] = serialize_names_factor(default_fuzzy_names_factor())

        return {NAMES_ENTITY_NAME: result}


def _make_default_change_factors(factors_count, compare_with_reg=True, prefix='change'):
    """
    Подготовить значения факторов по-умолчанию для смен с заданными индексами
    """
    def _make_not_set_values():
        return [FACTOR_NOT_SET] * factors_count

    required_env_names = ['user', 'reg'] if compare_with_reg else ['user']
    factors = {'%s_depth' % prefix: _make_not_set_values()}
    for env_name in required_env_names:
        factors.update({
            '%s_ip_eq_%s' % (prefix, env_name): _make_not_set_values(),
            '%s_subnet_eq_%s' % (prefix, env_name): _make_not_set_values(),
            '%s_ua_eq_%s' % (prefix, env_name): _make_not_set_values(),
        })
    return factors


def _extract_values_by_indices(values, indices):
    """
    Из списка values вытаскивает доступные значения по индексам в списке indices,
    каждый элемент списка вытаскивается не более одного раза (индексы могут быть отрицательными).
    Для индексов, которым не соответствует значение, в результат записывается None
    """
    accessed_values = [False] * len(values)
    extracted_values = []
    for index in indices:
        try:
            if accessed_values[index]:
                extracted_values.append(None)
                continue
            extracted_values.append(values[index])
            accessed_values[index] = True
        except IndexError:
            extracted_values.append(None)
            pass
    return extracted_values


def _calculate_factors_for_change(
        factors,
        factor_index,
        change_origin_info,
        user_env,
        registration_env,
        compare_with_reg=True,
        prefix='change',
):
    """
    Вычислить факторы для окружения, в котором было задано (изменено) значение.
    Вычисляются факторы сравнения с заданными окружениями, фактор глубины для времени задания значения.
    @param factor: словарь с подготовленными факторами для смены
    @param factor_index: номер фактора в списках значений факторов
    @param change_origin_info: данные окружения, в котором было задано (изменено) значение
    @param user_env: информация об окружении пользователя, пришедшего на анкету (объект типа UserEnv)
    @param registration_env: информация об окружении при регистрации (объект типа UserEnv)
    @param compare_with_reg: признак необходимости сравнения с регистрационным окружением
    @param prefix: префикс для записи результата в словарь факторов
    """
    required_env_names = ['user', 'reg'] if compare_with_reg else ['user']
    factors['%s_depth' % prefix][factor_index] = make_depth_factor(
        user_env.timestamp,
        registration_env.timestamp,
        change_origin_info['timestamp'],
    )
    parsed_change_ua = build_parsed_user_agent(
        change_origin_info.get('yandexuid'),
        user_agent=change_origin_info.get('user_agent'),
    )
    change_ip = change_origin_info.get('user_ip')
    for name in required_env_names:
        env = registration_env if name == 'reg' else user_env
        factors['%s_ip_eq_%s' % (prefix, name)][factor_index] = compare_ips(change_ip, env.ip)
        factors['%s_subnet_eq_%s' % (prefix, name)][factor_index] = compare_ip_with_subnet(change_ip, env.subnet)
        factors['%s_ua_eq_%s' % (prefix, name)][factor_index] = compare_uas(parsed_change_ua, env.ua)
    return factors


def _calculate_factors_for_values_changes(
        values,
        indices,
        user_env,
        registration_env,
        compare_with_reg=True,
        prefix='change',
):
    """
    Вычислить факторы для заданных смен значений
    TODO: не считаем удаление за смену, пока так
    @param values: значения, содержащие информацию об интервалах актуальности
    @param indices: индексы значений, для которых требуется вычислить факторы
    @param user_env: информация об окружении пользователя, пришедшего на анкету (объект типа UserEnv)
    @param registration_env: информация об окружении при регистрации (объект типа UserEnv)
    @param compare_with_reg: признак необходимости сравнения с регистрационным окружением
    @param prefix: префикс для записи результата в словарь факторов
    """
    factors = _make_default_change_factors(len(indices), compare_with_reg=compare_with_reg, prefix=prefix)
    target_values = _extract_values_by_indices(values, indices)
    for index, value in enumerate(target_values):
        if value is not None:
            _calculate_factors_for_change(
                factors,
                index,
                value['interval']['start'],
                user_env,
                registration_env,
                compare_with_reg=compare_with_reg,
                prefix=prefix,
            )
    return factors


def _compare_multiple_names(firstnames, lastnames, account_names, language):
    entered_names_to_indices = {
        (firstname.lower(), lastname.lower()): (first_index, last_index)
        for (first_index, firstname) in enumerate(firstnames) for (last_index, lastname) in enumerate(lastnames)
    }
    compare_info = []
    for names in account_names:
        # в этом цикле сравниваем введенные варианты имени и фамилии с каждым из значений в истории,
        # находим лучшее совпадение для каждого варианта из истории,
        # записываем лучший результат сравнения и соответствующие индексы введенных имени и фамилии
        name = (names['firstname'], names['lastname'])
        compare_results = sorted(
            [
                (entered_name, compare_names(name, entered_name, language))
                for entered_name in entered_names_to_indices.keys()
            ],
            key=lambda entered_name_and_factor: entered_name_and_factor[1],
        )
        compare_info.append({
            'factor': compare_results[-1][1],
            'input_indices': entered_names_to_indices[compare_results[-1][0]],
        })
    return compare_info


def _calculate_factors_and_indices_for_names(
        firstnames,
        lastnames,
        account_names,
        language,
        are_names_set_at_registration,
        user_env,
        registration_env,
):
    """
    Сравнить введенные варианты имени и фамилии со списком ФИО account_names.
    Вычислить факторы по результатам сравнения, а также индексы лучших совпадений.
    Считаем следующие факторы:
     - фактор сравнения с ФИО, указанными при регистрации (параметр are_names_set_at_registration)
     - фактор сравнения с текущими актуальными ФИО
     - фактор лучшего совпадения с промежуточными ФИО, найденными в истории
     - факторы совпадения окружений смен значений ФИО и окружений текущего пользователя / при регистрации
    Индекс совпадения - пара ((firstname_index, lastname_index), account_index), где firstname_index -
    номер введенного варианта имени, lastname_index - номер введенного варианта фамилии,
    account_index - номер в списке ФИО аккаунта.
    """
    compare_info = _compare_multiple_names(firstnames, lastnames, account_names, language)

    reg_factor = cur_factor = intermediate_factor = [FACTOR_NOT_SET, FACTOR_NOT_SET]
    intermediate_depth_factor = FACTOR_NOT_SET
    reg_best_indices = cur_best_indices = intermediate_best_index = None
    intermediate_info = compare_info[:-1]
    if are_names_set_at_registration:
        reg_factor = compare_info[0]['factor']
        reg_best_indices = compare_info[0]['input_indices']
        intermediate_info = intermediate_info[1:]
    if account_names:
        cur_factor = compare_info[-1]['factor']
        cur_best_indices = compare_info[-1]['input_indices']
    if intermediate_info:
        intermediate_factor, intermediate_best_index = sorted(
            [
                (info['factor'], index)
                for (index, info) in enumerate(intermediate_info)
            ],
            key=lambda factor_and_index: factor_and_index[0],
        )[-1]
        if are_names_set_at_registration:
            intermediate_best_index += 1
        if intermediate_factor > [STRING_FACTOR_NO_MATCH, STRING_FACTOR_NO_MATCH]:
            intermediate_depth_factor = make_depth_factor(
                user_env.timestamp,
                registration_env.timestamp,
                account_names[intermediate_best_index]['interval']['start']['timestamp'],
            )

    change_factor = _calculate_factors_for_values_changes(
        account_names[1:],  # первое значение не считаем сменой
        PERSONAL_DATA_CHANGE_INDICES,
        user_env,
        registration_env,
    )

    return dict(
        factor=dict(
            change_factor,
            registration=reg_factor,
            current=cur_factor,
            intermediate=intermediate_factor,
            intermediate_depth=intermediate_depth_factor,
            change_count=len(account_names) - 1 if account_names else 0,
        ),
        indices=dict(
            registration=(reg_best_indices, 0) if reg_best_indices is not None else None,
            current=(cur_best_indices, len(account_names) - 1) if cur_best_indices is not None else None,
            intermediate=(
                compare_info[intermediate_best_index]['input_indices'],
                intermediate_best_index,
            ) if intermediate_best_index is not None else None,
        ),
    )


ALLOWED_TIMESTAMP_DELTA = 1.0


def _is_timestamp_at_registration(account, timestamp):
    """
    Проверим, что указанный unixtime timestamp равен дате регистрации аккаунта. Учитываем округление
    даты регистрации до целых секунд.
    """
    if timestamp is not None:
        reg_timestamp = get_account_registration_unixtime(account.registration_datetime)
        event_timestamp = float(timestamp)
        return abs(reg_timestamp - event_timestamp) < ALLOWED_TIMESTAMP_DELTA
    else:
        return False


def _make_interval_point(timestamp=None):
    """
    Начало или конец интервала актуальности, построенный на основании текущих данных (нет информации об
    окружении, в котором устанавливалось значение).
    """
    return {'timestamp': timestamp}


class MultipleNamesHandler(HistoryDBEventsDataHandler):
    """
    Обработчик для сравнения нескольких введённых вариантов ФИО с историей ФИО (в т.ч. текущими ФИО).
    """
    def _get_account_names(self, person):
        account_names = dict(
            firstname=person.firstname,
            lastname=person.lastname,
            interval=dict(start=_make_interval_point(), end=None),
        ) if person.firstname or person.lastname else None
        history_names = self.events_info.names if self.events_info else None
        all_names = history_names or []
        if account_names and history_names:
            current_names_from_history = history_names[-1]
            if any(current_names_from_history[name] != account_names[name] for name in ('firstname', 'lastname')):
                # последние ФИО в истории не совпадают с аккаунтом, учтем ФИО из базы при сравнении
                log.warning(u'Current account names not found in history for UID %d', self.account.uid)
                # завершим интервал актуальности последних ФИО в истории
                all_names[-1]['interval']['end'] = _make_interval_point()
                all_names.append(account_names)
        elif account_names:
            # ФИО в истории не найдены, учтем ФИО из базы при сравнении
            all_names = [account_names]
        return all_names

    def handle(self):
        person = self.account.person
        account_names = self._get_account_names(person)
        are_names_set_at_registration = _is_timestamp_at_registration(
            self.account,
            account_names[0]['interval']['start']['timestamp'] if account_names else None,
        )

        result = _calculate_factors_and_indices_for_names(
            self.form_values['firstnames'],
            self.form_values['lastnames'],
            account_names,
            person.language,
            are_names_set_at_registration,
            self.user_env,
            self.registration_env,
        )

        result['entered'] = dict(firstnames=self.form_values['firstnames'], lastnames=self.form_values['lastnames'])
        result['account'] = account_names

        self.statbox.bind(factor_dict_to_statbox_dict(NAMES_ENTITY_NAME, result['factor']))
        return {NAMES_ENTITY_NAME: result}


class SimpleBirthdayHandler(HistoryDBEventsDataHandler):
    """
    Обработчик для сравнения ДР с аккаунтом и историей.
    """
    def handle(self):
        result = {}
        birthday = self.form_values['birthday']
        person = self.account.person
        account_birthday = person.birthday

        result['entered'] = str(birthday)
        result['account'] = str(account_birthday) if account_birthday else None
        result['account_factor'] = FACTOR_NOT_SET
        if account_birthday:
            result['account_factor'] = int(compare_birthdays(account_birthday, birthday))

        if self.events_info and self.events_info.birthdays:
            history_birthday = self.events_info.birthdays[0]['value']
            result['history'] = str(history_birthday)
            result['history_factor'] = int(compare_birthdays(
                history_birthday,
                birthday,
            ))
        else:
            result['history'] = None
            result['history_factor'] = FACTOR_NOT_SET

        self.statbox.bind(
            birthday_account_factor=result['account_factor'],
            birthday_history_factor=result['history_factor'],
        )
        return {BIRTHDAYS_ENTITY_NAME: result}


def _calculate_factors_for_birthdays(
        birthday,
        account_birthdays,
        is_birthday_set_at_registration,
        user_env,
        registration_env,
):
    """
    Сравнить введенный ДР с найденными в истории, которые когда-либо устанавливал пользователь.
    Получить факторы по результатам сравнения, индексы лучших совпадений.
    Считаем следующие факторы:
     - фактор сравнения с ДР, указанным при регистрации (параметр is_birthday_set_at_registration)
     - фактор сравнения с текущим актуальным ДР
     - фактор лучшего совпадения с промежуточными ДР, найденными в истории
     - факторы совпадения окружений для смен значений ДР
    """
    factors = [compare_birthdays_inexact(account_birthday['value'], birthday) for account_birthday in account_birthdays]

    reg_factor = cur_factor = intermediate_factor = FACTOR_NOT_SET
    intermediate_depth_factor = FACTOR_NOT_SET
    cur_index = reg_index = intermediate_best_index = None
    intermediate_factors = factors
    if is_birthday_set_at_registration:
        reg_factor = factors[0]
        reg_index = 0
        intermediate_factors = intermediate_factors[1:]
    if account_birthdays:
        last_birthday = account_birthdays[-1]
        if last_birthday['interval']['end'] is None:
            cur_factor = factors[-1]
            cur_index = len(account_birthdays) - 1
            intermediate_factors = intermediate_factors[:-1]
    if intermediate_factors:
        intermediate_best_index = 0
        intermediate_factor = BIRTHDAYS_FACTOR_NO_MATCH
        for index, factor in enumerate(intermediate_factors):
            if factor > intermediate_factor:
                intermediate_best_index = index
                intermediate_factor = factor
        if is_birthday_set_at_registration:
            intermediate_best_index += 1
        if intermediate_factor > BIRTHDAYS_FACTOR_NO_MATCH:
            intermediate_depth_factor = make_depth_factor(
                user_env.timestamp,
                registration_env.timestamp,
                account_birthdays[intermediate_best_index]['interval']['start']['timestamp'],
            )

    change_factor = _calculate_factors_for_values_changes(
        account_birthdays[1:],  # первое значение не считаем сменой
        PERSONAL_DATA_CHANGE_INDICES,
        user_env,
        registration_env,
    )

    return dict(
        factor=dict(
            change_factor,
            registration=reg_factor,
            current=cur_factor,
            intermediate=intermediate_factor,
            intermediate_depth=intermediate_depth_factor,
            change_count=len(account_birthdays) - 1 if account_birthdays else 0,
        ),
        indices=dict(
            registration=reg_index,
            current=cur_index,
            intermediate=intermediate_best_index,
        ),
    )


class BirthdayHandler(HistoryDBEventsDataHandler):
    """
    Обработчик для сравнения ДР с аккаунтом и историей, вычисляет факторы для сравнения ДР с регистрационными,
    текущими и промежуточными данными.
    """
    def _get_account_birthdays(self, person):
        account_birthday = person.birthday
        all_birthdays = self.events_info.birthdays if self.events_info else []
        if account_birthday and (not all_birthdays or all_birthdays[-1]['value'] != account_birthday):
            # ДР в истории не найден. Добавим ДР аккаунта в список всех ДР.
            if all_birthdays:
                log.warning(u'Current account birthday not found in history for UID %d', self.account.uid)
                # Для поддержания корректности, завершим интервал актуальности последнего ДР в истории.
                last_birthday_intervals = all_birthdays[-1]['interval']
                if last_birthday_intervals['end'] is None:
                    last_birthday_intervals['end'] = _make_interval_point()
            all_birthdays.append(dict(
                value=account_birthday,
                interval={'start': _make_interval_point(), 'end': None},
            ))
        return all_birthdays

    def handle(self):
        account_birthdays = self._get_account_birthdays(self.account.person)
        is_birthday_set_at_registration = _is_timestamp_at_registration(
            self.account,
            account_birthdays[0]['interval']['start']['timestamp'] if account_birthdays else None,
        )
        result = _calculate_factors_for_birthdays(
            self.form_values['birthday'],
            account_birthdays,
            is_birthday_set_at_registration,
            self.user_env,
            self.registration_env,
        )

        result['entered'] = str(self.form_values['birthday'])
        result['account'] = [dict(value=str(info['value']), interval=info['interval']) for info in account_birthdays]

        self.statbox.bind(factor_dict_to_statbox_dict(BIRTHDAYS_ENTITY_NAME, result['factor']))
        return {BIRTHDAYS_ENTITY_NAME: result}


class RegistrationDateHandler(UserDataHandler):
    """
    Сравнение дат регистрации.
    """
    def handle(self):
        result = {}
        user_tz = safe_detect_timezone(self.user_env.ip) or MOSCOW_TIMEZONE
        entered_date = self.form_values['registration_date']

        entered_date_user_tz = user_tz.localize(entered_date)
        result['entered'] = datetime_to_string(entered_date_user_tz, DATE_WITH_TZ_FORMAT)

        registration_datetime = self.account.registration_datetime
        if registration_datetime is Undefined:
            # Иногда у пользователя нет даты регистрации. Например, из-за неконсистентной записи в БД
            result.update(
                account=None,
                factor=FACTOR_NOT_SET,
            )
        else:
            # TODO: ниже передаем текущий пояс пользователя. Если пользователь недавно
            # зарегистрировался и переехал в другой часовой пояс, можем ошибиться.
            # Как вариант, можно взять IP регистрации и по нему найти часовой пояс.
            registration_date_localized = safe_local_datetime_to_date(registration_datetime, user_tz)
            result['account'] = datetime_to_string(registration_datetime)
            result['factor'] = compare_dates_loose(
                registration_date_localized,
                entered_date,
                user_tz,
            )

        self.statbox.bind(registration_date_factor=result['factor'])
        return dict(registration_date=result)


class RegistrationCountryCityHandler(HistoryDBEventsDataHandler):
    """
    Сравнение страны и города регистрации с введенными данными.
    """
    def _compare_with_region(self, region_key, region, entered_name, entered_id, result):
        # TODO: унести в core или сделать проверку через API карт
        language = self.account.person.language
        result['history_id'] = region['id']
        result['history'] = region['name'] if not isinstance(region['name'], str) else region['name'].decode('utf-8')
        result['factor']['id'] = int(region['id'] == entered_id)
        if not result['factor']['id']:
            # по ID не сматчились - попробуем сравнить неточно по именам
            compare_results = []
            for field in ['name', 'en_name', 'short_en_name']:
                name_of_region = region[field]
                if isinstance(name_of_region, str):
                    name_of_region = name_of_region.decode('utf-8')
                if name_of_region:
                    compare_result = compare_strings(name_of_region, entered_name, language)
                    compare_results.append(compare_result)
                    if compare_result.factors.initial_equal == FACTOR_BOOL_MATCH:
                        break
            index = find_best_string_factor_index([compare_res.factors for compare_res in compare_results])
            result['factor']['text'] = string_result_to_simple_factor(compare_results[index])

    def handle(self):
        country = self.form_values['registration_country']
        country_id = self.form_values['registration_country_id']
        city = self.form_values['registration_city']
        city_id = self.form_values['registration_city_id']
        results = {}
        region = None

        registration_env = self.registration_env
        if registration_env and registration_env.ip:
            region = geobase.Region(ip=registration_env.ip)
        else:
            log.debug('Registration IP not found for UID %s', self.account.uid)

        for region_type, region_name, region_id in zip(
                ['country', 'city'],
                [country, city],
                [country_id, city_id],
        ):
            result = {
                'factor': {
                    'text': FACTOR_NOT_SET,
                    'id': FACTOR_NOT_SET,
                },
                'history_id': None,
                'history': None,
                'entered': region_name,
                'entered_id': region_id,
            }
            region_key = 'registration_%s' % region_type
            if region and getattr(region, region_type):
                self._compare_with_region(region_key, getattr(region, region_type), region_name, region_id, result)
            self.statbox.bind({
                '%s_factor_text' % region_key: result['factor']['text'],
                '%s_factor_id' % region_key: result['factor']['id'],
            })
            results[region_type] = result

        return dict(
            registration_ip=registration_env.ip if registration_env else None,
            registration_country=results['country'],
            registration_city=results['city'],
        )


def _format_password_ranges(ranges):
    formatted_ranges = []
    for start_ts, end_ts in ranges:
        range_info = {'start': {'timestamp': start_ts}, 'end': None}
        if end_ts is not None:
            range_info['end'] = {'timestamp': end_ts}
        formatted_ranges.append(range_info)
    return formatted_ranges


def _handle_password_auth_date(account, password, password_auth_date, user_tz):
    password_auth_date_factor = FACTOR_NOT_SET
    password_auth_found_factor = FACTOR_NOT_SET
    api_status, is_password_found, active_ranges = find_password_in_history(account, password)
    best_range_index = None
    if api_status:
        password_auth_found_factor = int(is_password_found)
        if is_password_found:
            active_ranges.reverse()  # нужны сначала старые записи, потом новые
            for range_index, (start_ts, end_ts) in enumerate(active_ranges):
                if end_ts is None:
                    log.debug('Entered password is current valid password (UID %s)', account.uid)
                # TODO: если изменения происходили недавно (пару дней назад) и
                # пользователь переехал, можем ошибиться с часовым поясом. Можно
                # отдельно искать соответствующие события смены пароля.
                start_date = safe_local_datetime_to_date(
                    datetime.fromtimestamp(start_ts),
                    user_tz,
                )
                end_date = safe_local_datetime_to_date(
                    datetime.fromtimestamp(end_ts),
                    user_tz,
                ) if end_ts else None
                if password_auth_date >= start_date and (not end_date or password_auth_date <= end_date):
                    # Точное попадание в интервал
                    password_auth_date_factor = FACTOR_FLOAT_MATCH
                    best_range_index = range_index
                    break
                new_password_auth_date_factor = max(
                    compare_dates_loose(start_date, password_auth_date, user_tz),
                    compare_dates_loose(end_date, password_auth_date, user_tz) if end_date else FACTOR_NOT_SET,
                )
                if new_password_auth_date_factor > password_auth_date_factor:
                    password_auth_date_factor = new_password_auth_date_factor
                    best_range_index = range_index
    return dict(
        factor=dict(auth_date=password_auth_date_factor, auth_found=password_auth_found_factor),
        intervals=_format_password_ranges(active_ranges),
        best_range_index=best_range_index,
        api_status=api_status,
    )


class PasswordsAuthDateHandler(HistoryDBEventsDataHandler):
    """
    Поиск пароля и даты авторизации - несколько вариантов пароля.
    Анализ смен пароля.
    """
    def _handle_provided_passwords(self):
        """
        Считаем факторы по введенным вариантам пароля
        """
        passwords = list(self.form_values['passwords'])
        passwords.extend([None] * (MAX_PASSWORD_FIELDS - len(passwords)))
        password_auth_date = self.form_values['password_auth_date']
        user_tz = safe_detect_timezone(self.user_env.ip) or MOSCOW_TIMEZONE
        result = {
            'auth_date_entered': datetime_to_string(
                user_tz.localize(password_auth_date),
                DATE_WITH_TZ_FORMAT,
            ),
            'factor': {
                'entered_count': len(self.form_values['passwords']),
                'auth_found': [FACTOR_NOT_SET] * MAX_PASSWORD_FIELDS,
                'auth_date': [FACTOR_NOT_SET] * MAX_PASSWORD_FIELDS,
                'first_auth_depth': [FACTOR_NOT_SET] * MAX_PASSWORD_FIELDS,
                'equals_current': [FACTOR_NOT_SET] * MAX_PASSWORD_FIELDS,
            },
            'intervals': [],
            'indices': [],
            'api_statuses': [],
        }

        for index, password in enumerate(passwords):
            if password:
                password_info = _handle_password_auth_date(self.account, password, password_auth_date, user_tz)
                result['factor']['auth_found'][index] = password_info['factor']['auth_found']
                result['factor']['auth_date'][index] = password_info['factor']['auth_date']
                result['factor']['first_auth_depth'][index] = make_depth_factor(
                    self.user_env.timestamp,
                    self.registration_env.timestamp,
                    password_info['intervals'][0]['start']['timestamp'],
                ) if password_info['intervals'] else FACTOR_NOT_SET
                result['factor']['equals_current'][index] = int(
                    result['factor']['auth_found'][index] == FACTOR_BOOL_MATCH and
                    bool(password_info['intervals']) and
                    password_info['intervals'][-1]['end'] is None
                ) if password_info['api_status'] else FACTOR_NOT_SET
                result['intervals'].append(password_info['intervals'])
                result['indices'].append(password_info['best_range_index'])
                result['api_statuses'].append(password_info['api_status'])
            self.statbox.bind({
                'passwords_factor_auth_found_%s' % index: result['factor']['auth_found'][index],
                'passwords_factor_auth_date_%s' % index: result['factor']['auth_date'][index],
                'passwords_factor_first_auth_depth_%d' % index: result['factor']['first_auth_depth'][index],
                'passwords_factor_equals_current_%d' % index: result['factor']['equals_current'][index],
            })

        self.statbox.bind(
            passwords_api_status=all(result['api_statuses']),
            passwords_factor_entered_count=len(self.form_values['passwords']),
        )
        return result

    def _handle_password_changes(self, result):
        """
        Считаем факторы по последним сменам пароля:
        - глубина последней смены
        - сравнение окружений смены и текущего пользователя
        - требование принудительной смены пароля и его выполнение
        Отложенно ищем окружение смены в истории авторизаций (см. AggregatedFactorsHandler)
        """
        changes = self.events_info.password_changes if self.events_info else []
        result['factor'].update(
            _calculate_factors_for_values_changes(
                [dict(interval={'start': change['origin_info']}) for change in changes],
                [-1],
                self.user_env,
                self.registration_env,
                compare_with_reg=False,
            ),
        )

        last_change_is_forced_change = FACTOR_NOT_SET
        if changes:
            last_change_is_forced_change = int(changes[-1]['change_type'] == PASSWORD_CHANGE_TYPE_FORCED)

        password_change_requests = self.events_info.password_change_requests if self.events_info else []
        is_forced_change_pending = FACTOR_BOOL_NO_MATCH
        if password_change_requests:
            last_change_timestamp = changes[-1]['origin_info']['timestamp'] if changes else None
            last_request = password_change_requests[-1]
            request_timestamp = last_request['origin_info']['timestamp']
            is_forced_change_pending = int(
                last_request['change_required'] and
                (last_change_timestamp is None or request_timestamp > last_change_timestamp),
            )

        result['factor'].update(
            change_count=len(changes),  # задание пароля при регистрации и так не учитывается в events_info
            last_change_is_forced_change=last_change_is_forced_change,
            forced_change_pending=is_forced_change_pending,
        )
        self.statbox.bind(factor_dict_to_statbox_dict(PASSWORDS_ENTITY_NAME, result['factor']))
        result['actual'] = {
            'last_change': changes[-1] if changes else None,
            'last_change_request': password_change_requests[-1] if password_change_requests else None,
        }
        return result

    def handle(self):
        passwords_result = self._handle_provided_passwords()
        return {PASSWORDS_ENTITY_NAME: self._handle_password_changes(passwords_result)}


PASSWORD_HASH_WITH_VERSION_RE = re.compile(r'^\d+:.*$')
PASSWORD_MATCH_DEPTH = 5


class PasswordMatchesHandler(HistoryDBEventsDataHandler):
    """
    Обработчик ищет совпадения указанного пароля с последними N паролями в истории.
    """
    def handle(self):
        """
        Подготавливаем хеши для проверки, сравниваем пароль с хешами через ЧЯ.
        """
        history_hashes = self.events_info.password_hashes if self.events_info else []
        hashes_to_check = []
        for history_hash_value in reversed(history_hashes):
            if history_hash_value == '*':
                # Хеши не всегда писались в историю
                break
            if not PASSWORD_HASH_WITH_VERSION_RE.match(history_hash_value):
                history_hash_value = '1:' + history_hash_value
            hashes_to_check.append(base64.b64encode(history_hash_value))
            if len(hashes_to_check) >= PASSWORD_MATCH_DEPTH:
                break

        match_statuses = {}
        if hashes_to_check:
            match_statuses = get_blackbox().test_pwd_hashes(
                self.form_values['password'],
                set(hashes_to_check),
                uid=self.account.uid,
            )

        # Вычисляем факторы совпадения для последних N паролей (сначала более свежие пароли)
        factor = [int(match_statuses[hash_value]) for hash_value in hashes_to_check]
        factor = factor + [FACTOR_NOT_SET] * (PASSWORD_MATCH_DEPTH - len(factor))
        self.statbox.bind(factor_list_to_statbox_dict('password_matches_factor', factor))
        return dict(password_matches=dict(factor=factor))


class MobileDeviceIdHandler(HistoryDBEventsDataHandler):
    """
    Обработчик проверяет, что ID мобильного устройства найден в истории.
    ID записывается в историю при включении 2ФА.
    """
    def handle(self):
        incoming_device_id = self.user_env.device_info.device_id if self.user_env.device_info else None
        history_app_key_info = self.events_info.app_key_info if self.events_info else []

        history_device_ids = []
        for info in history_app_key_info:
            device_info = MobileDeviceInfo.from_app_key_info(info)
            if device_info:
                history_device_ids.append(device_info.device_id)

        factor = FACTOR_NOT_SET
        if incoming_device_id and self.events_info:
            factor = int(incoming_device_id in history_device_ids)
        result = {
            'actual': incoming_device_id,
            'history': history_device_ids,
            'factor': factor,
        }
        self.statbox.bind(device_id_factor=result['factor'])
        return dict(device_id=result)


class RestorePassedAttemptsHandler(HistoryDBEventsDataHandler):
    """
    Сохраняет информацию о последних успешных попытках восстановления пароля через анкету или через саппорт.
    Используется как ручной фактор для предотвращения перетягиваний аккаунта.
    """
    def handle(self):
        attempts = self.events_info.restore_passed_attempts if self.events_info else []
        user_ts = self.user_env.timestamp
        from_ts = user_ts - settings.RESTORE_SEMI_AUTO_POSITIVE_DECISION_RETRY_IMPOSSIBLE_INTERVAL
        factor = dict(has_recent_positive_decision=False)
        for attempt in attempts:
            if attempt['timestamp'] < from_ts:
                continue
            if attempt['method'] == 'semi_auto' or 'link_type' in attempt:
                factor['has_recent_positive_decision'] = True
                break

        self.statbox.bind(restore_attempts_has_recent_positive_decision=factor['has_recent_positive_decision'])
        return dict(restore_attempts=dict(attempts=attempts, factor=factor))


def _compare_envs(env, other_env):
    # вычислить факторы совпадения для окружений
    return {
        'ip': compare_ips(env.ip, other_env.ip),
        'subnet': compare_ip_with_subnet(env.ip, other_env.subnet),
        'ua': compare_uas(env.ua, other_env.ua),
    }


def _match_envs(env, other_env):
    # вычислить точные совпадения для окружений
    return {
        'ip': compare_ips(env.ip, other_env.ip) == FACTOR_BOOL_MATCH,
        'subnet': compare_ip_with_subnet(env.ip, other_env.subnet) == FACTOR_BOOL_MATCH,
        'ua': compare_uas(env.ua, other_env.ua) == UA_FACTOR_FULL_MATCH,
    }


def _match_env_with_auth_item(auth_item, env):
    """
    Найти совпадения окружения для элемента истории авторизаций ручки auths_aggregated_runtime.
    """
    history_ip = auth_item['ip'].get('ip')
    history_ua = {
        'os.name': auth_item['os'].get('name'),
        'yandexuid': auth_item['browser'].get('yandexuid'),
        'browser.name': auth_item['browser'].get('name'),
    }
    auth_env = UserEnv(ip=history_ip, ua=history_ua)
    return _match_envs(auth_env, env)


def _format_auth_info(timestamp, auth_item):
    return {
        'timestamp': timestamp,
        'authtype': auth_item['authtype'],
        'status': auth_item['status'],
    }


class AggregatedAuthsDataHandler(HistoryDBEventsDataHandler):
    """
    Обработчик пользовательских данных, использующий данные аггрегированной истории авторизаций.
    """
    def __init__(self, *args, **kwargs):
        """
        @param aggregated_auths_info: объект типа AuthsAggregatedRuntimeInfo
        """
        self.aggregated_auths_info = kwargs.pop('aggregated_auths_info')
        super(AggregatedAuthsDataHandler, self).__init__(*args, **kwargs)


def _serialize_env(env, with_timestamp=False):
    data = {
        'ip': str(env.ip) if env.ip is not None else None,
        'subnet': str(env.subnet) if env.subnet is not None else None,
        'ua': env.ua,
    }
    if env.entity is not None:
        data['entity'] = env.entity
    if with_timestamp:
        data['timestamp'] = env.timestamp
    return data


class UserEnvAuthsHandler(AggregatedAuthsDataHandler):
    """
    Обработчик выполняет поиск компонент окружения (IP/подсеть/UA) в истории авторизации. Также сравниваются
    окружения текущего пользователя и при регистрации.
    """
    def _build_registration_factors(self):
        factors = _compare_envs(self.user_env, self.registration_env)
        return {'%s_eq_reg' % field: value for (field, value) in factors.items()}

    def handle(self):
        result = {
            'factor': self._build_registration_factors(),
        }
        for type_, env in zip(('actual', 'registration'), (self.user_env, self.registration_env)):
            result[type_] = _serialize_env(env)
        for field in ('ip', 'subnet', 'ua'):
            result['factor']['%s_first_auth_depth' % field] = FACTOR_NOT_SET
            result['factor']['%s_auth_interval' % field] = FACTOR_NOT_SET
            result['actual']['%s_first_auth' % field] = None
            result['actual']['%s_last_auth' % field] = None

        total_auths_count = 0
        for timestamp, auth_item, count in self.aggregated_auths_info:
            # Проходим по всем авторизациям, чтобы посчитать общее число
            total_auths_count += count
            matches = _match_env_with_auth_item(auth_item, self.user_env)
            for field, is_match in matches.items():
                if is_match:
                    auth_info = _format_auth_info(timestamp, auth_item)
                    last_field, first_field = '%s_last_auth' % field, '%s_first_auth' % field
                    if result['actual'][last_field] is None:
                        result['actual'][last_field] = auth_info
                    result['actual'][first_field] = auth_info

        for field in ('ip', 'subnet', 'ua'):
            last_field, first_field = '%s_last_auth' % field, '%s_first_auth' % field
            if result['actual'][first_field] is not None:
                timestamp = result['actual'][first_field]['timestamp']
                depth_factor = make_depth_factor(self.user_env.timestamp, self.registration_env.timestamp, timestamp)
                result['factor']['%s_first_auth_depth' % field] = depth_factor
                interval_field = '%s_auth_interval' % field
                result['factor'][interval_field] = calculate_timestamp_interval_factor(
                    self.registration_env.timestamp,
                    self.user_env.timestamp,
                    timestamp,
                    result['actual'][last_field]['timestamp'],
                    max_depth_threshold=timedelta(days=settings.RESTORE_SEMI_AUTO_IP_CHECK_DEPTH).total_seconds(),
                )

        # Определяем, смогли ли мы получить все данные за интервал
        result['factor']['auths_limit_reached'] = int(total_auths_count >= self.aggregated_auths_info.auths_limit)
        result['actual']['gathered_auths_count'] = total_auths_count

        self.statbox.bind(factor_dict_to_statbox_dict('user_env_auths', result['factor']))
        return dict(user_env_auths=result)


def _make_key_from_parsed_user_agent(ua):
    """
    Строим строковое представление UA по расширенному представлению.
    """
    return '\t'.join(
        [str(ua[field]) for field in ('os.name', 'browser.name', 'yandexuid')],
    )


def _make_not_empty_key_from_auth_item_user_agent(auth_item):
    """
    Специальный случай - строим строковое представление UA по элементу истории авторизаций, причем только в том
    случае, если все компоненты присутствуют (ОС/браузер/yandexuid).
    """
    values = (
        auth_item['os'].get('name'),
        auth_item['browser'].get('name'),
        auth_item['browser'].get('yandexuid'),
    )
    if not all(values):
        return None
    return '\t'.join(values).lower()


# максимальное число изменений, по которым строятся аггрегированные факторы
PASSWORD_AND_PERSONAL_MAX_ANALYZED_CHANGES = 1 + 2
PASSWORD_AND_RECOVERY_MAX_ANALYZED_CHANGES = 1 + 2
PERSONAL_AND_RECOVERY_MAX_ANALYZED_CHANGES = 2 + 2


class AggregatedFactorsHandler(AggregatedAuthsDataHandler):

    def handle(self):
        # собираем информацию о сменах в один день и отдельных сменах
        change_pairs_info, changes_info = self._gather_change_pairs_and_changes()
        # вычисляем часть факторов
        result = self._calculate_factors_for_one_day_and_env_matches(change_pairs_info)
        # подготавливаем данные для поиска в истории авторизаций
        search_task = self._prepare_auths_search_task(change_pairs_info, changes_info)
        # выполняем поиск первых авторизаций с IP/подсетей/UA в истории авторизаций
        search_result = self._find_first_auths_in_history_by_envs(search_task)
        # теперь можно вычислить факторы глубины для найденных совпадений
        return self._calculate_factors_for_auth_history_matches(result, search_result, change_pairs_info, changes_info)

    def _gather_change_pairs_and_changes(self):
        """
        Собираем изменения пароля, личных данных, средств восстановления из данных,
        вычисленных другими обработчиками
        """
        changes_info = []
        passwords = self.calculated_factors[PASSWORDS_ENTITY_NAME]
        password_changes = []
        last_password_change = passwords['actual']['last_change']
        if last_password_change:
            password_changes.append(build_change_env(last_password_change['origin_info'], entity=PASSWORDS_ENTITY_NAME))
        changes_info.append(
            {
                'entity': PASSWORDS_ENTITY_NAME,
                'type': 'change',
                'envs': [password_changes[0] if password_changes else None],
            },
        )

        personal_changes = []
        for entity in (NAMES_ENTITY_NAME, BIRTHDAYS_ENTITY_NAME):
            items = self.calculated_factors[entity]['account']
            if len(items) > 1:  # сменой считается задание нового значения взамен предыдущего
                personal_changes.append(build_change_env(items[-1]['interval']['start'], entity=entity))

        recovery_method_changes = []
        for entity in (ANSWERS_ENTITY_NAME, PHONE_NUMBERS_ENTITY_NAME):
            entity_data = self.calculated_factors[entity]
            history_items = entity_data['history']
            if entity == ANSWERS_ENTITY_NAME:
                match_items = _get_intervals_for_matches_from_answer_data(entity_data)
                items = flatten_question_answer_mapping(history_items)
            else:
                match_items = _get_intervals_for_matches_from_phone_numbers_data(entity_data)
                items = flatten_phones_mapping(history_items)
            if len(items) > 1:  # сменой считается задание нового значения взамен предыдущего
                recovery_method_changes.append(build_change_env(items[-1]['interval']['start'], entity=entity))
            match_items = _extract_values_by_indices(match_items, MATCHED_RESTORE_METHODS_CHANGE_INDICES)
            changes_info.append(
                {
                    'entity': entity,
                    'type': 'match',
                    'envs': [build_change_env(item['interval']['start']) if item else None for item in match_items],
                },
            )
        return (
            (
                {
                    'pair_name': 'password_and_personal_change_one_day',
                    'changes_pairs': (password_changes, personal_changes),
                    'max_different_changes': PASSWORD_AND_PERSONAL_MAX_ANALYZED_CHANGES,
                },
                {
                    'pair_name': 'password_and_recovery_change_one_day',
                    'changes_pairs': (password_changes, recovery_method_changes),
                    'max_different_changes': PASSWORD_AND_RECOVERY_MAX_ANALYZED_CHANGES,
                },
                {
                    'pair_name': 'personal_and_recovery_change_one_day',
                    'changes_pairs': (personal_changes, recovery_method_changes),
                    'max_different_changes': PERSONAL_AND_RECOVERY_MAX_ANALYZED_CHANGES,
                },
            ),
            changes_info,
        )

    def _make_key_from_env_field(self, env, field):
        """
        Вычисляем значение для ключа по компоненту окружения
        """
        value = getattr(env, field)
        if value is None:
            # сразу отсекаем пустые значения
            return
        elif field == 'ua':
            return _make_key_from_parsed_user_agent(value)
        elif field == 'subnet':
            return value
        else:  # IP
            return str(value)

    def _find_same_day_and_same_env_matches(self, changes, other_changes):
        """
        Проверить условие смены в один день и с одного IP/подсети/UA,
        подготовить данные для поиска в истории авторизаций.
        @return список данных смен, для которых выполнено условие; отображение значений компонентов
        в данные смены для дальнейшего поиска в истории авторизаций.
        """
        change_map = {'ip': OrderedDict(), 'subnet': OrderedDict(), 'ua': OrderedDict()}
        match_infos = []
        for change in changes:
            for other_change in other_changes:
                if abs(change.timestamp - other_change.timestamp) < timedelta(days=1).total_seconds():
                    env_matches = _match_envs(change, other_change)
                    if not any(env_matches.values()):
                        continue
                    # сохраним информацию о найденном совпадении - данные окружений, совпавшие компоненты окружений
                    match_info = {
                        'envs': [change, other_change],
                        'fields': [field for field, is_match in env_matches.items() if is_match],
                    }
                    match_infos.append(match_info)
                    # нужно собрать все компоненты для поиска в истории
                    for change_to_process in (change, other_change):
                        # обновим отображение значений IP/UA/subnet в данные смены
                        for field in ('ip', 'subnet', 'ua'):
                            key = self._make_key_from_env_field(change_to_process, field)
                            if key is None:
                                continue
                            prev_change = change_map[field].get(key)
                            if prev_change:
                                # если мы уже добавляли данный IP/подсеть/UA в отображение - нужно выбрать
                                # более раннюю смену для учета при поиске
                                if change_to_process.timestamp < prev_change.timestamp:
                                    change_map[field][key] = change_to_process
                            else:
                                change_map[field][key] = change_to_process
        return match_infos, change_map

    def _find_first_auths_in_history_by_envs(self, search_task):
        """
        Найти последние авторизации с заданных IP/подсетей/UA.
        """
        ip_task, subnet_task, ua_task = search_task['ip'], search_task['subnet'], search_task['ua']
        ip_result = {}
        subnet_result = {}
        ua_result = {}
        for timestamp, auth_item, count in reversed(list(self.aggregated_auths_info)):
            auth_ip = auth_item['ip'].get('ip')
            if auth_ip:
                for ip in ip_task:
                    match = compare_ips(auth_ip, ip) == FACTOR_BOOL_MATCH
                    if match:
                        ip_result[ip] = _format_auth_info(timestamp, auth_item)
                        ip_task.remove(ip)
                        break
                for subnet in subnet_task:
                    match = compare_ip_with_subnet(auth_ip, subnet) == FACTOR_BOOL_MATCH
                    if match:
                        subnet_result[subnet] = _format_auth_info(timestamp, auth_item)
                        # здесь безопасно выйти сразу - подсети, полученные из as_lookup, не пересекаются
                        subnet_task.remove(subnet)
                        break
            # сравниваем UA точно, поэтому используем строковое представление для быстрого сравнения
            auth_ua_key = _make_not_empty_key_from_auth_item_user_agent(auth_item)
            if auth_ua_key:
                for ua_key in ua_task:
                    match = auth_ua_key == ua_key
                    if match:
                        ua_result[ua_key] = _format_auth_info(timestamp, auth_item)
                        ua_task.remove(ua_key)
                        break
            if not ip_task and not subnet_task and not ua_task:
                # удача - всё сматчили и можем выйти из цикла
                break
        return {'ip': ip_result, 'subnet': subnet_result, 'ua': ua_result}

    def _calculate_factors_for_one_day_and_env_matches(self, change_pairs_info):
        """
        Вычислить факторы-признаки смены в один день, факторы совпадения с окружениями пользователя и регистрации.
        Подготовить информацию о найденных сменах в один день.
        """
        factors = {}
        matches = {}
        for pair_info in change_pairs_info:
            pair_name = pair_info['pair_name']
            changes, other_changes = pair_info['changes_pairs']
            factors[pair_name] = {}

            change_matches, change_map = self._find_same_day_and_same_env_matches(changes, other_changes)
            matches[pair_name] = change_matches
            all_matches = set([field for match_info in change_matches for field in match_info['fields']])
            pair_info['change_map'] = change_map
            for field in ('ip', 'subnet', 'ua'):
                # признаки того, что найдено совпадение по условию: один день, совпадение IP/UA/subnet
                factors[pair_name]['%s_match' % field] = int(bool(field in all_matches))

                # предустанавливаем значения по умолчанию для факторов совпадения с окружениями
                factors[pair_name]['%s_eq_reg' % field] = FACTOR_NOT_SET
                factors[pair_name]['%s_eq_user' % field] = FACTOR_NOT_SET

                # вычислим факторы совпадения с окружением пользователя и регистрации
                for value_key, change in change_map[field].items():
                    for env_name, env in zip(('user', 'reg'), (self.user_env, self.registration_env)):
                        factor = _compare_envs(env, change)[field]
                        prev_factor = factors[pair_name]['%s_eq_%s' % (field, env_name)]
                        factors[pair_name]['%s_eq_%s' % (field, env_name)] = max(factor, prev_factor)
        return dict(factor=factors, matches=matches)

    def _prepare_auths_search_task(self, change_pairs_info, changes_info):
        """
        Подготовить данные (IP/подсети/UA) для поиска в истории авторизаций
        """
        # для смен в один день
        search_task = {'ip': set(), 'subnet': set(), 'ua': set()}
        for pair_info in change_pairs_info:
            for field in ('ip', 'subnet', 'ua'):
                # подготовим данные для поиска в истории авторизаций
                for value_key, change in pair_info['change_map'][field].items():
                    if field == 'ua' and any(comp is None for comp in change.ua.values()):
                        # для поиска используем только ua с полным набором данных
                        continue
                    search_task[field].add(value_key)

        # для отдельных смен
        for change_info in changes_info:
            for change_env in change_info['envs']:
                for field in ('ip', 'subnet', 'ua'):
                    value = getattr(change_env, field, None)
                    if value is not None:
                        if field == 'ua':
                            if any(comp is None for comp in change_env.ua.values()):
                                continue
                            value = _make_key_from_parsed_user_agent(value)
                        search_task[field].add(value)
        return search_task

    def _serialize_one_day_matches_with_auth_info(self, matches, search_result):
        """
        Дополнить информацию о совпадениях (смены в один день и с одного окружения) результатами поиска в
        истории авторизаций, подготовить к сериализации в JSON
        """
        for match in matches:
            for index, env in enumerate(match['envs']):
                serialized_env = _serialize_env(env, with_timestamp=True)
                for field in ('ip', 'subnet', 'ua'):
                    key = self._make_key_from_env_field(env, field)
                    auth_info = search_result[field].get(key)
                    serialized_env['%s_first_auth_info' % field] = auth_info
                match['envs'][index] = serialized_env

    def _calculate_factors_for_auth_history_matches(self, result, search_result, change_pairs_info, changes_info):
        """
        Вычислить факторы по результатам поиска в истории авторизаций
        """
        factors = result['factor']

        # для факторов смен в один день
        for pair_info in change_pairs_info:
            pair_name = pair_info['pair_name']
            max_different_changes = pair_info['max_different_changes']
            change_map = pair_info['change_map']
            for field in ('ip', 'subnet', 'ua'):
                auth_matches = search_result[field]
                env_changes = change_map[field]
                cur_factors = []
                cur_values = []
                for value, auth_info in auth_matches.items():
                    first_auth_ts = auth_info['timestamp']
                    if value in env_changes:
                        change = env_changes[value]
                        # вычисляем фактор для проверки утверждения:
                        # IP/subnet/UA последней смены - присутствовал до этого в истории авторизаций в последний
                        # месяц до смены/не присутствовал/присутствовал до этого в истории дольше, чем в последний месяц до смены
                        new_factor = make_depth_factor(
                            change.timestamp,  # время смены выступает в роли "времени сейчас"
                            self.registration_env.timestamp,
                            first_auth_ts,
                            fixed_threshold=settings.RESTORE_SEMI_AUTO_FIXED_THRESHOLD_FOR_AGGREGATED_FACTORS,
                        )
                        cur_factors.append(new_factor)
                        cur_values.append(value)
                cur_factors = cur_factors + [FACTOR_NOT_SET] * (max_different_changes - len(cur_factors))
                factors[pair_name]['%s_first_auth_depth' % field] = sorted(cur_factors, reverse=True)
            # сохранить информацию о найденных авторизациях вместе с данными по сменам в один день
            self._serialize_one_day_matches_with_auth_info(result['matches'][pair_name], search_result)

        # для факторов по отдельным сменам
        for change_info in changes_info:
            entity = change_info['entity']
            change_type = change_info['type']
            entity_data = self.calculated_factors[entity]
            for field in ('ip', 'subnet', 'ua'):
                # инициализируем поля данных (либо очищаем, в случае повторного вызова)
                factor_field = '%s_%s_first_auth_depth' % (change_type, field)
                actual_field = '%s_%s_first_auth' % (change_type, field)
                entity_data['factor'][factor_field] = []
                entity_data.setdefault('actual', {})[actual_field] = []

            for index, change_env in enumerate(change_info['envs']):
                for field in ('ip', 'subnet', 'ua'):
                    auth_matches = search_result[field]
                    factor_field = '%s_%s_first_auth_depth' % (change_type, field)
                    actual_field = '%s_%s_first_auth' % (change_type, field)

                    factor = FACTOR_NOT_SET
                    auth_info = None
                    if change_env:
                        env_value = getattr(change_env, field)
                        if field == 'ua':
                            env_value = _make_key_from_parsed_user_agent(env_value)
                        if env_value in auth_matches:
                            # значение найдено в истории авторизаций
                            auth_info = auth_matches[env_value]
                            first_auth_ts = auth_info['timestamp']
                            factor = make_depth_factor(
                                change_env.timestamp,  # время смены выступает в роли "времени сейчас"
                                self.registration_env.timestamp,
                                first_auth_ts,
                                fixed_threshold=settings.RESTORE_SEMI_AUTO_FIXED_THRESHOLD_FOR_AGGREGATED_FACTORS,
                            )
                    entity_data['factor'][factor_field].append(factor)
                    entity_data['actual'][actual_field].append(auth_info)
                    self.statbox.bind(
                        {'%s_factor_%s_%s_first_auth_depth_%d' % (entity, change_type, field, index): factor},
                    )

        self.statbox.bind(factor_dict_to_statbox_dict('aggregated', factors))
        return dict(aggregated=result)


def _compare_phone_lists(entered_phones, history_phones):
    matches = []
    match_indices = []
    not_matched_history_phones = dict((phone, index) for index, phone in enumerate(history_phones))
    # нужно найти совпадения номеров из двух множеств, при этом нельзя допускать
    # повторных совпадений различных номеров из-за дублей и опечаток во введенных
    # номерах, а также возможной похожести номеров в истории
    for entered_index, phone in enumerate(entered_phones):
        for history_phone in not_matched_history_phones:
            compare_result = compare_phones(history_phone, phone)
            if compare_result.status:
                history_index = not_matched_history_phones.pop(history_phone)
                matches.append(history_phone)
                match_indices.append((entered_index, history_index))
                break
    return matches, match_indices


def _get_intervals_for_matches_from_phone_numbers_data(phone_numbers_data):
    """
    Получить информацию об интервалах актуальности телефонов в истории для найденных совпадений
    с телефонами, которые ввел пользователь.
    """
    history_phones_info = phone_numbers_data['history']
    match_indices = phone_numbers_data['match_indices']
    history_match_indices = [history_index for (_, history_index) in match_indices]
    matched_phones_info = [history_phones_info[index] for index in history_match_indices]
    # работаем с телефонами в плоском виде. совпадение значений может быть одно, но интервалов
    # актуальности - несколько, в этом случае посмотрим на два различных окружения
    flat_phones = flatten_phones_mapping(matched_phones_info)
    return flat_phones


class PhoneNumbersHandler(HistoryDBEventsDataHandler):
    """
    Сравнение провалидированных номеров телефонов из истории и введенных.
    """

    def _prepare_entered_phones(self, phone_numbers):
        """
        Необходимо оставить во введённых строках только цифры для корректного сравнения
        """
        return [filter(lambda char: char.isdigit(), phone) for phone in phone_numbers]

    def _prepare_history_phones(self):
        """
        История телефонов зачастую не содержит корректной информации о всех номерах пользователя -
        дополним данные из истории недостающими подтвержденными телефонами из YaSMS.
        """
        phones_raw = Yasms(None, get_yasms(), None).userphones(self.account)
        # FIXME: после переезда в новый ЯСМС все телефоны в базе будут валидны, удалить этот код
        phones_raw = [phone for phone in phones_raw if parse_phone_number(phone['number']) is not None]
        current_confirmed_phones = sorted(
            # allow_impossible, потому что мы верим телефонам, которые записаны в истории. Они все проходили валидацию
            build_phones(phones_raw, allow_impossible=True).confirmed,
            key=lambda phone: phone.validation_datetime,
        )
        history_phones_info = self.events_info.confirmed_phones if self.events_info else []
        numbers_found_in_history = {phone_info['value'] for phone_info in history_phones_info}
        for phone in current_confirmed_phones:
            if phone.number.digital in numbers_found_in_history:
                continue
            phone_info = {
                'value': phone.number.digital,
                'intervals': [
                    {
                        'start': _make_interval_point(timestamp=datetime_to_unixtime(phone.validation_datetime)),
                        'end': None,
                    },
                ],
            }
            history_phones_info.append(phone_info)
        return history_phones_info

    def _calculate_match_change_factors(self, phone_numbers_data):
        """
        При нахождении совпадений введенных телефонов и телефонов в истории, нужно вычислить факторы смены
        для первого (по времени) совпадения и для последнего.
        Отдельно для совпадений вычисляются факторы поиска окружения в истории авторизаций.
        """
        flat_phones = _get_intervals_for_matches_from_phone_numbers_data(phone_numbers_data)
        match_change_factors = _calculate_factors_for_values_changes(
            flat_phones,
            MATCHED_RESTORE_METHODS_CHANGE_INDICES,
            self.user_env,
            self.registration_env,
            prefix='match',
        )
        return match_change_factors

    def handle(self):
        raw_entered_phones = self.form_values['phone_numbers'] or []
        filtered_entered_phones = self._prepare_entered_phones(raw_entered_phones)
        history_phones_info = self._prepare_history_phones()

        result = {}
        matches, match_indices = _compare_phone_lists(
            filtered_entered_phones,
            [info['value'] for info in history_phones_info],
        )
        flat_phones = flatten_phones_mapping(history_phones_info)

        result.update(
            entered=raw_entered_phones,
            history=history_phones_info,
            matches=matches,
            match_indices=match_indices,
            factor=dict(
                entered_count=len(raw_entered_phones),
                history_count=len(history_phones_info),
                matches_count=len(matches),
                change_count=len(flat_phones) - 1 if flat_phones else 0,
            ),
        )

        # вычисляем факторы для конкретных смен значений
        change_factors = _calculate_factors_for_values_changes(
            flat_phones[1:],  # первое значение не считаем сменой
            RESTORE_METHODS_CHANGE_INDICES,
            self.user_env,
            self.registration_env,
            compare_with_reg=False,
        )
        result['factor'].update(change_factors)

        # вычисляем факторы для значений в истории, совпавших с введенными пользователем значениями
        match_change_factors = self._calculate_match_change_factors(result)
        result['factor'].update(match_change_factors)

        self.statbox.bind(factor_dict_to_statbox_dict(PHONE_NUMBERS_ENTITY_NAME, result['factor']))
        return {PHONE_NUMBERS_ENTITY_NAME: result}


class ConfirmedEmailsHandler(HistoryDBEventsDataHandler):
    """
    Сравнение почтовых адресов из истории и введенных.
    """
    def handle(self):
        entered_emails = unique_preserve_order(encode_emails(self.form_values['emails'] or []))
        emails_info = self.events_info.confirmed_emails if self.events_info else []
        history_emails = [info['value'] for info in emails_info]
        matches = []
        match_indices = []
        result = {}

        not_matched_history_emails = dict((email, index) for index, email in enumerate(history_emails))
        for entered_index, email in enumerate(entered_emails):
            for history_email in not_matched_history_emails:
                compare_result = compare_emails(history_email, email)
                if compare_result.status:
                    matches.append(history_email)
                    history_index = not_matched_history_emails.pop(history_email)
                    match_indices.append((entered_index, history_index))
                    break

        result.update(
            entered=entered_emails,
            history=emails_info,
            matches=matches,
            match_indices=match_indices,
            factor=dict(
                entered_count=len(entered_emails),
                history_count=len(history_emails),
                matches_count=len(matches),
            ),
        )
        self.statbox.bind(factor_dict_to_statbox_dict('emails', result['factor']))
        return dict(emails=result)


class MailApiDataHandler(UserDataHandler):
    """
    Обработчик пользовательских данных, использующий почтовые данные.
    """
    @property
    def has_mail(self):
        return self.account.is_subscribed(Service.by_slug('mail'))


class EmailFoldersHandler(MailApiDataHandler):
    """
    Сравнение почтовых папок.
    """
    def handle(self):
        entered_folders = self.form_values['email_folders']
        api_status, actual_folders = True, []
        if self.has_mail:
            api_status, actual_folders = get_user_email_folders(
                self.account.uid,
                self.account.subscriptions[MAIL_SID].suid,
                self.account.mail_db_id,
            )
        matches = []

        not_matched_folders = set(actual_folders)
        for folder in entered_folders or []:
            for email_folder in not_matched_folders:
                compare_result = compare_strings(email_folder, folder)
                if compare_result.status:
                    matches.append(email_folder)
                    not_matched_folders.remove(email_folder)
                    break

        result = {}
        result['factor'] = {
            'entered_count': len(entered_folders) if entered_folders else 0,
            'actual_count': len(actual_folders),
            'matches_count': len(matches),
        }
        result['entered'] = entered_folders
        result['matches'] = matches
        result['actual'] = actual_folders
        result['api_status'] = api_status
        self.statbox.bind(
            factor_dict_to_statbox_dict('email_folders', result['factor']),
            email_folders_api_status=api_status,
        )
        return dict(email_folders=result)


class CommonEmailListHandlerMixin(object):
    def _compare_email_lists(self, entered_list, actual_list, api_status):
        """
        Сравнение списков email'ов.
        Перед сравнением - кодировка в punycode, удаление дубликатов.
        @param entered_list: список введенных email'ов.
        @param actual_list: список актуальных email'ов, полученный из API почты.
        @param api_status: признак успешности работы API почты.
        @return словарь с результатами сравнения.
        """
        matches = []
        result = {}
        filtered_entered_list = unique_preserve_order(encode_emails(entered_list or []))
        actual_list = unique_preserve_order(encode_emails(actual_list))
        not_matched_actual_emails = set(actual_list)
        for email in filtered_entered_list:
            for actual_email in not_matched_actual_emails:
                compare_result = compare_emails(actual_email, email)
                if compare_result.status:
                    matches.append(actual_email)
                    not_matched_actual_emails.remove(actual_email)
                    break

        result['factor'] = {
            'entered_count': len(entered_list) if entered_list else 0,
            'actual_count': len(actual_list),
            'matches_count': len(matches),
        }
        result['entered'] = entered_list
        result['matches'] = matches
        result['actual'] = actual_list
        result['api_status'] = api_status
        return result

    def _bind_factors_to_statbox(self, list_name, result):
        statbox_dict = {'%s_api_status' % list_name: result['api_status']}
        statbox_dict.update(factor_dict_to_statbox_dict(list_name, result['factor']))
        self.statbox.bind(statbox_dict)


class EmailBlackWhiteHandler(CommonEmailListHandlerMixin, MailApiDataHandler):
    """
    Сравнение ЧБ списков.
    """
    def handle(self):
        api_status, blacklist, whitelist = True, [], []
        if self.has_mail:
            api_status, blacklist, whitelist = get_blackwhite_lists(self.account.uid)
        result = {}
        for list_name, actual_list in zip(['email_blacklist', 'email_whitelist'], [blacklist, whitelist]):
            entered_list = self.form_values[list_name]
            list_result = self._compare_email_lists(entered_list, actual_list, api_status)
            self._bind_factors_to_statbox(list_name, list_result)
            result.update({list_name: list_result})
        return result


class EmailCollectorsHandler(CommonEmailListHandlerMixin, MailApiDataHandler):
    """
    Сравнение сборщиков.
    """
    def handle(self):
        api_status, collectors = True, []
        if self.has_mail:
            api_status, collectors = get_collectors(
                self.account.subscriptions[MAIL_SID].suid,
                self.account.mail_db_id,
            )
        entered_collectors = self.form_values['email_collectors']
        result = self._compare_email_lists(entered_collectors, collectors, api_status)
        self._bind_factors_to_statbox('email_collectors', result)
        return dict(email_collectors=result)


class OutboundEmailsHandler(CommonEmailListHandlerMixin, MailApiDataHandler):
    """
    Сравнение исходящих адресов.
    """
    def handle(self):
        api_status, outbound_emails = True, []
        if self.has_mail:
            api_status, outbound_emails = get_outbound_emails(self.account.uid)
        entered_emails = self.form_values['outbound_emails']
        result = self._compare_email_lists(entered_emails, outbound_emails, api_status)
        self._bind_factors_to_statbox('outbound_emails', result)
        return dict(outbound_emails=result)


def _get_intervals_for_matches_from_answer_data(answer_data):
    """
    Получить информацию об интервалах актуальности КВ/КО в истории для найденного совпадения
    с КО, который ввел пользователь.
    """
    best_factor = answer_data['factor']['best']
    intervals = []
    if best_factor >= STRING_FACTOR_INEXACT_MATCH:
        question_index, answer_index = answer_data['indices']['best']
        question_info = answer_data['history'][question_index]
        answer_info = question_info['answers'][answer_index]
        intervals = [dict(interval=interval) for interval in answer_info['intervals']]
    return intervals


class ControlAnswerHandler(HistoryDBEventsDataHandler):
    """
    Поиск КО в истории.
    """
    def _compare_answers(self, history_answers, entered_answer):
        language = self.account.person.language
        compare_results = []
        factors = []
        for history_answer in history_answers:
            compare_result = compare_answers(entered_answer, history_answer, language)
            compare_results.append(compare_result)
            factors.append(compare_result.factors)
        best_index = find_best_string_factor_index(factors)
        return best_index, compare_results

    def _calculate_match_change_factors(self, answer_data):
        """
        При нахождении совпадения, нужно вычислить факторы смены для первого (по времени) задания
        значения и для последнего.
        Отдельно для совпадения вычисляются факторы поиска окружения в истории авторизаций.
        """
        flat_values = _get_intervals_for_matches_from_answer_data(answer_data)
        match_change_factors = _calculate_factors_for_values_changes(
            flat_values,
            MATCHED_RESTORE_METHODS_CHANGE_INDICES,
            self.user_env,
            self.registration_env,
            prefix='match',
        )
        return match_change_factors

    def handle(self):
        question_key = None
        entered_answer = None
        question_answer = self.form_values['question_answer']
        if question_answer:
            # КВ/КО заданы в форме
            entered_answer = question_answer['answer']
            question_id, question_text = question_answer['question_id'], question_answer['question']
            question_key = smart_text(Question(question_text, question_id=question_id))

        question_answer_mapping = self.events_info.question_answer_mapping if self.events_info else {}
        serialized_question_answer_mapping = serialize_question_answer_mapping(question_answer_mapping)
        flat_question_answers = flatten_question_answer_mapping(serialized_question_answer_mapping)
        history_answers = question_answer_mapping.get(question_key, {}).keys()
        result = {}

        result['entered'] = {'question': question_key, 'answer': entered_answer}
        result['history'] = serialized_question_answer_mapping

        best_factor = current_factor = FACTOR_NOT_SET
        best_index = None

        if history_answers and entered_answer:
            best_index, compare_results = self._compare_answers(history_answers, entered_answer)
            best_factor = string_result_to_simple_factor(compare_results[best_index])
            # КВ, на который отвечает пользователь, является текущим, если какой-либо ответ на этот КВ
            # актуален в настоящее время (не установлен конец интервала)
            current_answer_index = None
            for index, answer in enumerate(question_answer_mapping[question_key].values()):
                if answer['intervals'][-1]['end'] is None:
                    current_answer_index = index
                    break
            if current_answer_index is not None:
                # В случае ответа на текущий КВ устанавливаем фактор для текущего КО
                current_factor = string_result_to_simple_factor(compare_results[current_answer_index])
            best_index = (question_answer_mapping.keys().index(question_key), best_index)

        result['indices'] = {'best': best_index}

        result['factor'] = factor = {
            'best': best_factor,
            'current': current_factor,
            'change_count': len(flat_question_answers) - 1 if flat_question_answers else 0,
        }

        change_factors = _calculate_factors_for_values_changes(
            flat_question_answers[1:],  # первое значение не считаем сменой,
            RESTORE_METHODS_CHANGE_INDICES,
            self.user_env,
            self.registration_env,
            compare_with_reg=False,
        )
        factor.update(change_factors)

        match_change_factors = self._calculate_match_change_factors(result)
        factor.update(match_change_factors)

        self.statbox.bind(factor_dict_to_statbox_dict(ANSWERS_ENTITY_NAME, factor))

        return {ANSWERS_ENTITY_NAME: result}


class ServicesHandler(UserDataHandler):
    """
    Сравнение наборов используемых сервисов.
    """
    def _get_account_services(self):
        services = []
        for sid, subscription in self.account.subscriptions.items():
            if subscription.service and subscription.service in PROCESSED_SERVICES:
                services.append(subscription.service)
        return services

    def _get_entered_service_names(self):
        """
        Данный обработчик работает с сервисами в виде списка объектов Service, а также в виде списка строк.
        TODO: Удалить поддержку списка объектов Service после переезда на многошаговую анкету.
        """
        entered_services = self.form_values['services'] or []
        return [service.slug if isinstance(service, Service) else service for service in entered_services]

    def handle(self):
        entered_service_names = self._get_entered_service_names()
        account_services = self._get_account_services()
        matched_services = []
        result = {}
        # пока умеем определять факт использования сервиса только по наличию подписки
        for service_name in entered_service_names:
            service = SERVICE_NAME_TO_SUBSCRIBED_SERVICE.get(service_name)
            if service in account_services:
                matched_services.append(service_name)

        result['entered'] = entered_service_names
        result['account'] = [SUBSCRIBED_SERVICE_TO_SERVICE_NAME[acc_service] for acc_service in account_services]
        result['matches'] = matched_services
        result['factor'] = {
            'entered_count': len(entered_service_names),
            'account_count': len(account_services),
            'matches_count': len(matched_services),
        }
        self.statbox.bind(factor_dict_to_statbox_dict('services', result['factor']))
        return dict(services=result)


class SocialAccountsHandler(UserDataHandler):
    """
    Проверка привязанных соц. аккаунтов.
    """
    def handle(self):
        task_ids = self.form_values['social_accounts'] or []
        result = {}

        api_status, account_profiles = get_profiles_with_person_by_uid(self.account.uid)
        if not api_status:
            log.warning('Failed to get social profiles for uid %s', self.account.uid)

        api_calls_succeeded, entered_accounts, entered_profiles = get_unique_social_profiles_by_task_ids(task_ids)
        matches_count = len(filter(
            lambda profile: profile['uid'] == self.account.uid,
            entered_profiles,
        ))

        result.update(
            entered_accounts=entered_accounts,
            entered_profiles=entered_profiles,
            account_profiles=account_profiles,
            factor={
                'matches_count': matches_count,
                'entered_accounts_count': len(task_ids),
                'entered_profiles_count': len(entered_profiles),
                'account_profiles_count': len(account_profiles),
            },
            api_status=api_calls_succeeded and api_status,
        )
        self.statbox.bind(
            social_accounts_factor_matches_count=matches_count,
            social_accounts_factor_entered_accounts_count=len(task_ids),
            social_accounts_factor_entered_profiles_count=len(entered_profiles),
            social_accounts_factor_account_profiles_count=len(account_profiles),
            social_accounts_api_status=api_calls_succeeded and api_status,
        )
        return dict(social_accounts=result)


class DeliveryAddressesHandler(UserDataHandler):
    """
    Сравнение адресов доставки.
    """
    def handle(self):
        # Мы больше не храним адреса в своей БД, поэтому результат сверки всегда отрицательный
        # TODO: удалить эту логику и этот фактор из рестора
        account_addresses = []
        entered_addresses = self.form_values['delivery_addresses']
        result = {}
        matches = []

        result['entered'] = entered_addresses
        result['account'] = account_addresses
        result['matches'] = matches
        result['factor'] = {
            'entered_count': len(entered_addresses) if entered_addresses else 0,
            'account_count': len(account_addresses),
            'matches_count': len(matches),
        }

        self.statbox.bind(factor_dict_to_statbox_dict('delivery_addresses', result['factor']))
        return dict(delivery_addresses=result)


class CalculateFactorsMixin(object):
    FACTORS_TO_HANDLERS_MAPPING = {
        # Используют данные аккаунта
        'registration_date': RegistrationDateHandler,
        'services': ServicesHandler,
        'delivery_addresses': DeliveryAddressesHandler,
        'names': NamesHandler,  # Также использует события HistoryDB
        'multiple_names': MultipleNamesHandler,  # Также использует события HistoryDB
        'simple_birthday': SimpleBirthdayHandler,  # Также использует события HistoryDB
        'birthday': BirthdayHandler,  # Также использует события HistoryDB

        # Используют API Почты
        'email_folders': EmailFoldersHandler,  # /folders
        'email_blackwhite': EmailBlackWhiteHandler,  # /blackwhite
        'email_collectors': EmailCollectorsHandler,  # /rpop
        'outbound_emails': OutboundEmailsHandler,   # /collie

        # Используют HistoryDB API
        'passwords': PasswordsAuthDateHandler,  # /events_passwords, /events
        'password_matches': PasswordMatchesHandler,  # /events
        'user_env_auths': UserEnvAuthsHandler,  # /auths_aggregated_runtime
        'device_id': MobileDeviceIdHandler,  # /events
        'restore_attempts': RestorePassedAttemptsHandler,  # /events

        # Используют события HistoryDB
        'reg_country_city': RegistrationCountryCityHandler,
        'phone_numbers': PhoneNumbersHandler,
        'confirmed_emails': ConfirmedEmailsHandler,
        'question_answer': ControlAnswerHandler,

        # Используют социальное API
        'social_accounts': SocialAccountsHandler,

        # Использует результат работы нескольких обработчиков для получения аггрегированного результата
        'aggregated_factors': AggregatedFactorsHandler,
    }

    FACTORS_TO_EVENTS_INFO_PARAMS_MAPPING = {
        'names': 'names',
        'multiple_names': ('grouped_names', 'registration_env'),
        'simple_birthday': 'birthdays',
        'birthday': ('birthdays', 'registration_env'),
        'confirmed_emails': 'confirmed_emails',
        'question_answer': 'question_answer_mapping',
        'phone_numbers': 'confirmed_phones',
        'reg_country_city': 'registration_env',
        'user_env_auths': 'registration_env',
        'passwords': 'password_changes',
        'password_matches': 'password_hashes',
        'aggregated_factors': 'registration_env',
        'device_id': 'app_key_info',
        'restore_attempts': 'restore_passed_attempts',
    }

    # Следующие данные, полученные из истории, сохраняются для использования на последующих шагах
    CACHED_EVENTS_INFO_ITEMS = (
        'registration_env',
    )

    FACTORS_USING_EVENTS = {
        'names',
        'multiple_names',
        'simple_birthday',
        'birthday',
        'reg_country_city',
        'phone_numbers',
        'confirmed_emails',
        'question_answer',
        'user_env_auths',
        'passwords',
        'password_matches',
        'aggregated_factors',
        'device_id',
        'restore_attempts',
    }

    FACTORS_USING_AGGREGATED_AUTHS = {
        'user_env_auths',
        'aggregated_factors',
    }

    FACTORS_USING_MAIL_API = {
        'email_folders',
        'email_blackwhite',
        'email_collectors',
        'outbound_emails',
    }

    def _get_events(self, factor_names, factors, events_info_cache):
        events_info_params = {}
        analyzer_params = {}
        if 'question_answer' in factor_names:
            analyzer_params['language'] = get_preferred_language(self.account)

        for name, param in self.FACTORS_TO_EVENTS_INFO_PARAMS_MAPPING.items():
            if name in factor_names:
                if isinstance(param, (list, tuple)):
                    for sub_param in param:
                        if sub_param not in events_info_cache:
                            events_info_params[sub_param] = True
                elif param not in events_info_cache:
                    events_info_params[param] = True

        if not events_info_params:
            # в случае, если в параметрах ничего нет - мы всё нашли в кеше и события не нужно получать
            return
        historydb_status, historydb_events_info = get_historydb_events_info(
            self.account,
            analyzer_params=analyzer_params,
            **events_info_params
        )
        factors['historydb_api_events_status'] = historydb_status
        self.statbox.bind(historydb_status=historydb_status)
        return historydb_events_info

    def _get_aggregated_auths(self, factors):
        auths_info = AuthsAggregatedRuntimeInfo(
            self.account,
            check_depth=settings.RESTORE_SEMI_AUTO_IP_CHECK_DEPTH,
            limit=settings.RESTORE_SEMI_AUTO_AUTHS_HISTORY_LIMIT,
        )
        factors['auths_aggregated_runtime_api_status'] = auths_info.api_status
        self.statbox.bind(auths_aggregated_runtime_api_status=auths_info.api_status)
        return auths_info

    def calculate_factors(self, *factor_names, **options):
        handler_classes = set()
        factors = options.get('calculated_factors', {})
        events_info_cache = options.get('events_info_cache', {})
        factor_names = set(factor_names)

        are_events_required = not factor_names.isdisjoint(self.FACTORS_USING_EVENTS)
        are_auths_required = not factor_names.isdisjoint(self.FACTORS_USING_AGGREGATED_AUTHS)

        historydb_events_info = self._get_events(factor_names, factors, events_info_cache) if are_events_required else None
        aggregated_auths_info = self._get_aggregated_auths(factors) if are_auths_required else None

        for factor in factor_names:
            handler_classes.add(self.FACTORS_TO_HANDLERS_MAPPING[factor])

        user_env = build_user_env(self.headers, self.cookies.get('yandexuid'), self.client_ip, track=self.track)
        registration_env = build_registration_env(historydb_events_info, events_info_cache, self.account)

        handlers = [
            cls(
                self.form_values,
                self.account,
                user_env,
                self.statbox,
                factors,
                events_info=historydb_events_info,
                registration_env=registration_env,
                aggregated_auths_info=aggregated_auths_info,
            ) for cls in handler_classes
        ]

        for handler in handlers:
            factors.update(handler.handle())

        return factors
