# -*- coding: utf-8 -*-

from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.core import validators
from passport.backend.core.support_link_types import ALL_SUPPORT_LINK_TYPES

from .base import (
    RESTORE_METHOD_HINT,
    SELECTABLE_RESTORE_METHODS,
)


DEFAULT_STRING_UPPER_LIMIT = 1024


class RestoreSubmitForm(validators.Schema):
    login = validators.LooseEmailValidator()
    retpath = validators.RetPath(ignore_invalid=True, if_missing=None, not_empty=True, strip=True)

    gps_package_name = validators.GpsPackageName()


class RestoreKeySubmitForm(validators.Schema):
    secret_key = validators.PersistentTrackKey()


class RestoreSelectMethodForm(validators.Schema):
    method = validators.OneOf(SELECTABLE_RESTORE_METHODS, not_empty=True, strip=True)


class RestoreCheckPhoneForm(validators.Schema):
    phone_number = validators.String(not_empty=True, strip=True)
    country = validators.CountryCode(if_missing=None, strip=True)
    display_language = validators.DisplayLanguage()
    confirm_method = validators.OneOf(
        [CONFIRM_METHOD_BY_SMS, CONFIRM_METHOD_BY_CALL],
        if_missing=CONFIRM_METHOD_BY_SMS,
        not_empty=True,
    )
    code_format = validators.OneOf(['by_3_dash'], if_missing=None, not_empty=True)

    chained_validators = [
        validators.PhoneNumber(),
    ]


class RestoreConfirmPhoneForm(validators.Schema):
    code = validators.String(not_empty=True, strip=True)


class RestoreCheckPinForm(validators.Schema):
    pin = validators.String(not_empty=True, strip=True)


class RestoreCheckEmailForm(validators.Schema):
    email = validators.SimpleEmailValidator(not_empty=True, strip=True)
    display_language = validators.DisplayLanguage()

    is_simple_format = validators.StringBoolean(not_empty=True, strip=True, if_missing=False)


class RestoreCheckAnswerForm(validators.Schema):
    answer = validators.String(not_empty=True, strip=True, max=DEFAULT_STRING_UPPER_LIMIT)


class RestoreCheck2FAFormForm(validators.Schema):
    firstname = validators.FirstName(not_empty=True, strip=True)
    lastname = validators.LastName(not_empty=True, strip=True)

    password = validators.String(not_empty=True, strip=True)
    answer = validators.String(strip=True, if_missing=None, max=DEFAULT_STRING_UPPER_LIMIT)


class RestoreCommitForm(validators.Schema):
    password = validators.String(not_empty=True, strip=True)
    display_language = validators.DisplayLanguage()

    revoke_web_sessions = validators.StringBool(if_missing=True, strip=True)
    revoke_tokens = validators.StringBool(if_missing=True, strip=True)
    revoke_app_passwords = validators.StringBool(if_missing=True, strip=True)


class RestoreCommitNewMethodForm(validators.Schema):
    def __init__(self, allowed_methods, is_new_method_required, *args, **kwargs):
        super(RestoreCommitNewMethodForm, self).__init__(*args, **kwargs)
        one_of_kwargs = dict(
            not_empty=is_new_method_required,
        )
        if not is_new_method_required:
            one_of_kwargs['if_missing'] = None
        self.add_field(
            'new_method',
            validators.OneOf(allowed_methods, **one_of_kwargs),
        )

    question_id = validators.HintQuestionId()
    question = validators.HintQuestion()
    answer = validators.HintAnswer(if_missing=None)

    chained_validators = [
        validators.RequireSet(
            allowed_sets=[
                ['question_id', 'answer'],
                ['question', 'answer'],
            ],
            allow_empty=True,
        ),
        validators.RequireIfEquals(['answer'], 'new_method', RESTORE_METHOD_HINT),
    ]


class RestoreCreateLinkForm(validators.Schema):
    uid = validators.Uid()

    link_type = validators.OneOf(ALL_SUPPORT_LINK_TYPES, not_empty=True)

    # Имя админа и его комментарий для лога
    admin_name = validators.String(not_empty=True, strip=True)
    comment = validators.String(strip=True, if_missing='')
