# -*- coding: utf-8 -*-

from passport.backend.api.common.common import extract_tld
from passport.backend.api.views.bundle.mixins import TrackRestricter
from passport.backend.api.views.bundle.phone.exceptions import SendingLimitExceededError
from passport.backend.api.views.bundle.register.helpers import get_auth_by_key_link
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    MailInfo,
    make_email_context,
)
from passport.backend.core.support_link_types import (
    BLOCKING_SUPPORT_LINK_TYPES,
    SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD,
    SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION,
    SUPPORT_LINK_TYPE_FORCE_PHONE_RESTORATION,
    SUPPORT_LINK_TYPE_REDIRECT_TO_COMPLETION,
)
from passport.backend.core.types.email.email import mask_email_for_challenge
from passport.backend.core.types.login.login import masked_login
from passport.backend.core.types.phone_number.phone_number import (
    mask_phone_number,
    parse_phone_number,
)
from passport.backend.core.utils.decorators import cached_property

from .base import (
    RESTORE_METHOD_EMAIL,
    RESTORE_METHOD_HINT,
    RESTORE_METHOD_PHONE,
    RESTORE_METHOD_SEMI_AUTO_FORM,
)


OTP_DISABLED_ON_RESTORE_NOTIFICATION_TEMPLATE = 'mail/otp_disabled_on_restore_notification.html'

RESTORATION_KEY_MESSAGE_TEMPLATE = 'mail/email_restoration_message_body.html'
RESTORATION_KEY_MESSAGE_SIMPLE_TEMPLATE = 'mail/email_restoration_message_body_simple.txt'

RESTORE_PASSED_BY_HINT_MESSAGE_TEMPLATE = 'mail/restore_passed_by_hint_notification.html'
RESTORE_PASSED_BY_EMAIL_MESSAGE_TEMPLATE = 'mail/restore_passed_by_email_notification.html'
RESTORE_PASSED_BY_PHONE_MESSAGE_TEMPLATE = 'mail/restore_passed_by_phone_notification.html'
RESTORE_PASSED_BY_SEMI_AUTO_FORM_MESSAGE_TEMPLATE = 'mail/restore_passed_by_semi_auto_form_notification.html'


def get_restoration_link(restoration_key, tld):
    return settings.RESTORE_BY_LINK_TEMPLATE_URL % dict(key=restoration_key, tld=tld)


def get_restoration_key_email_notification_data(host, login, account, language, restoration_key, is_simple_format):
    """
    Подготовить информацию для отправки сообщения со ссылкой и кодом восстановления.
    Возвращает template_name, info, context.
    """
    translations = settings.translations.NOTIFICATIONS[language]
    template_name = (
        RESTORATION_KEY_MESSAGE_TEMPLATE if not is_simple_format else RESTORATION_KEY_MESSAGE_SIMPLE_TEMPLATE
    )
    tld = extract_tld(host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD
    info = MailInfo(
        subject=translations['restore.auto.email_restoration_message_subject'],
        from_=translations['email_sender_display_name'],
        tld=tld,
    )
    restoration_link = get_restoration_link(restoration_key, tld)
    context = make_email_context(
        language=language,
        account=account,
        context={
            'LOGIN': masked_login(login),
            'RESTORATION_KEY': restoration_key,
            'RESTORATION_LINK': restoration_link,
        },
    )
    return template_name, info, context


def get_2fa_email_notification_data(host, account, language):
    """
    Подготовить информацию для отправки сообщения о выключении 2ФА при восстановлении.
    Возвращает template_name, info, context.
    """
    translations = settings.translations.NOTIFICATIONS[language]
    template_name = OTP_DISABLED_ON_RESTORE_NOTIFICATION_TEMPLATE
    info = MailInfo(
        subject=translations['2fa_disabled_on_restore.subject'],
        from_=translations['email_sender_display_name'],
        tld=extract_tld(host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD,
    )
    context = make_email_context(
        language=language,
        account=account,
    )
    return template_name, info, context


def restore_passed_message_context_shadower(context):
    context = dict(context)
    for field in ['login', 'LOGIN', 'MASKED_LOGIN']:
        context[field] = masked_login(context[field])
    if 'EMAIL' in context:
        context['EMAIL'] = mask_email_for_challenge(context['EMAIL'])
    return context


def get_restore_passed_email_notification_data(host, account, language, restore_method, phone=None, email=None):
    """
    Подготовить информацию для отправки сообщения о пройденном восстановлении (случаи
    восстановления по КВ/КО, email-у, телефону и анкете).
    Возвращает template_name, info, context.
    """
    translations = settings.translations.NOTIFICATIONS[language]
    template_name = {
        RESTORE_METHOD_HINT: RESTORE_PASSED_BY_HINT_MESSAGE_TEMPLATE,
        RESTORE_METHOD_EMAIL: RESTORE_PASSED_BY_EMAIL_MESSAGE_TEMPLATE,
        RESTORE_METHOD_PHONE: RESTORE_PASSED_BY_PHONE_MESSAGE_TEMPLATE,
        RESTORE_METHOD_SEMI_AUTO_FORM: RESTORE_PASSED_BY_SEMI_AUTO_FORM_MESSAGE_TEMPLATE,
    }[restore_method]
    info = MailInfo(
        subject=translations['restore.auto.passed.subject'],
        from_=translations['email_sender_display_name'],
        tld=extract_tld(host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD,
    )
    context = make_email_context(
        language=language,
        account=account,
        context={
            'LOGIN': account.login,
            # На аккаунте может быть защищенный телефон (уже был или привязали в процессе) -
            # в этом случае текст письма не содержит рекомендаций привязки телефона
            'is_secure_phone_bound': bool(account.phones.secure),
            'restore_url': settings.RESTORE_DEFAULT_URL_TEMPLATE % dict(tld=info.tld),
            'profile_url': settings.PROFILE_URL_TEMPLATE % dict(tld=info.tld),
            'profile_emails_url': settings.PROFILE_EMAILS_URL_TEMPLATE % dict(tld=info.tld),
            'profile_phones_url': settings.PROFILE_PHONES_URL_TEMPLATE % dict(tld=info.tld),
            'profile_access_url': settings.PROFILE_ACCESS_URL_TEMPLATE % dict(tld=info.tld),
            'profile_journal_url': settings.PROFILE_JOURNAL_URL_TEMPLATE % dict(tld=info.tld),
            'mail_journal_url': settings.MAIL_JOURNAL_URL_TEMPLATE % dict(tld=info.tld),
            'disk_journal_url': settings.DISK_JOURNAL_URL_TEMPLATE % dict(tld=info.tld),
            'change_hint_url': settings.PROFILE_CHANGE_HINT_URL_TEMPLATE % dict(tld=info.tld),
            'account_security_help_url': translations['account_security_help_url'],
        },
    )
    if restore_method == RESTORE_METHOD_EMAIL:
        context['EMAIL'] = email
    elif restore_method == RESTORE_METHOD_PHONE:
        # Телефон маскируем всегда
        context['PHONE'] = mask_phone_number(parse_phone_number(phone).international)
    return template_name, info, context


class SupportLinkOptions(object):

    def __init__(self, link_type):
        self.link_type = link_type

    @cached_property
    def is_disabled_account_allowed(self):
        """
        Признак допустимости работы с заблокированным аккаунтом
        """
        return self.link_type in BLOCKING_SUPPORT_LINK_TYPES

    @cached_property
    def force_hint_restoration(self):
        """
        Разрешить восстановление по КВ/КО, даже если есть блокирующие способы восстановления
        """
        return self.link_type == SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION

    @cached_property
    def force_2fa_phone_restoration(self):
        """
        Разрешить восстановление по телефону для 2ФА-пользователей
        """
        return self.link_type == SUPPORT_LINK_TYPE_FORCE_PHONE_RESTORATION

    @cached_property
    def entities_to_flush(self):
        """
        Сущности, которые необходимо сбросить (инвалидировать) на аккаунте
        """
        entities = {'emails', 'social_profiles'}
        if self.link_type != SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION:
            entities.add('hint')
        if self.link_type != SUPPORT_LINK_TYPE_FORCE_PHONE_RESTORATION:
            entities.add('phones')

        return entities

    @cached_property
    def is_method_binding_required(self):
        """
        Требуется ли привязка нового средства восстановления
        """
        return self.link_type in BLOCKING_SUPPORT_LINK_TYPES

    @cached_property
    def allowed_methods_to_bind(self):
        """
        Допустимые варианты средств восстановления, одно из которых требуется привязать на аккаунте
        """
        return {
            # При вводе нового пароля и сбросе всех данных нужно привязать телефон или КВ/КО
            SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD: [RESTORE_METHOD_PHONE, RESTORE_METHOD_HINT],
            # При восстановлении по КВ/КО остальные данные сбрасываются, рекомендуется привязать телефон
            SUPPORT_LINK_TYPE_FORCE_HINT_RESTORATION: [RESTORE_METHOD_PHONE],
        }.get(self.link_type, [])

    @cached_property
    def is_missing_password_with_portal_alias_allowed(self):
        """
        Разрешено ли восстановление пользователю с портальным алиасом и без пароля
        """
        return self.link_type in {
            SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD,
            SUPPORT_LINK_TYPE_REDIRECT_TO_COMPLETION,
        }

    @cached_property
    def key_link_getter(self):
        """
        Функция для получения ссылки для отображения пользователю
        """
        if self.link_type != SUPPORT_LINK_TYPE_REDIRECT_TO_COMPLETION:
            return get_restoration_link
        else:
            return get_auth_by_key_link


class RestoreRestricter(TrackRestricter):
    def __init__(self, *args):
        super(RestoreRestricter, self).__init__(*args)
        self.restrict_ip()
        self.restrict_rate()

    def check(self):
        super(RestoreRestricter, self).check()
        if self._confirmation_info.code_send_count >= settings.SMS_VALIDATION_MAX_SMS_COUNT:
            self._statbox.stash(error='sms_limit.exceeded', reason='track_limit')
            raise SendingLimitExceededError()
