# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.account import is_auth_by_sms_secure_enough_for_account
from passport.backend.api.common.account_manager import (
    AmPlatform,
    get_am_platform,
)
from passport.backend.api.common.common import should_ignore_per_ip_counters
from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_FLASH_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.api.common.processes import (
    PROCESS_LOGIN_RESTORE,
    PROCESS_WEB_REGISTRATION,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    BaseBundleError,
    CallsShutDownError,
    CompareNotMatchedError,
    CreateCallFailed,
    InvalidTrackStateError,
    RateLimitExceedError,
    SecurePhoneNotFoundError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundlePhoneMixin,
    BundleTrackedPhoneConfirmationMixin,
    OctopusAvailabilityMixin,
)
from passport.backend.api.views.bundle.phone.helpers import (
    dump_number,
    normalize_code,
    VerifyingPhoneCallConfirmator,
)
from passport.backend.api.views.bundle.restore.login.forms import (
    LoginRestoreCheckNamesForm,
    LoginRestoreCheckNamesSimpleForm,
    LoginRestoreCheckPhoneForm,
    LoginRestoreConfirmPhoneForm,
    LoginRestoreSubmitForm,
)
from passport.backend.api.views.bundle.restore.login.helpers import LoginRestoreRestricter
from passport.backend.api.yasms.utils import get_many_accounts_with_phones_by_uids
from passport.backend.core.builders.octopus import OctopusPermanentError
from passport.backend.core.compare.compare import (
    compare_names,
    STRING_FACTOR_INEXACT_MATCH,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import login_restore_counter
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    StatboxLogger,
)
from passport.backend.core.models.account import ACCOUNT_DISABLED_ON_DELETION
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_SOCIAL,
)
from passport.backend.core.types.mobile_device_info import get_app_id_from_track
from passport.backend.core.types.phone_number.phone_number import (
    get_alt_phone_numbers_of_phone_number,
    parse_phone_number,
    PhoneNumber,
)
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport.api.views.bundle.restore.login.controllers')


RESTORE_BASE_GRANT = 'login_restore.base'
RESTORE_SIMPLE_GRANT = 'login_restore.simple'

RESTORE_TRACK_TYPE = 'restore'

RESTORE_STATE_SUBMITTED = 'submitted'

RESTORE_STATE_PHONE_CONFIRMED = 'phone_confirmed'

RESTORE_STATE_ACCOUNTS_FOUND = 'accounts_found'


AUTH_FLOW_INSTANT = 'instant'
AUTH_FLOW_FULL = 'full'
REGISTER_FLOW_PORTAL = 'portal'
REGISTER_FLOW_NEOPHONISH = 'neophonish'


class BaseLoginRestoreView(BaseBundleView, BundleAssertCaptchaMixin, BundleAccountPropertiesMixin,
                           BundleAccountResponseRendererMixin, BundlePhoneMixin):
    process_name = PROCESS_LOGIN_RESTORE

    track_type = RESTORE_TRACK_TYPE

    require_process = True

    require_track = True

    required_grants = [RESTORE_BASE_GRANT]

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
    )

    restore_step = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='login_restore',
            track_id=self.track_id,
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
            step=self.restore_step,
        )

    @cached_property
    def antifraud_logger(self):
        # Аналогично PASSP-36049
        scenario = None
        if self.track.scenario:
            scenario = self.track.scenario
        elif self.track.track_type:
            scenario = self.track.track_type

        return AntifraudLogger(
            channel='pharma',
            sub_channel=self.consumer,
            status='OK',
            external_id='track-{}'.format(self.track_id),
            phone_confirmation_method=self.track.phone_confirmation_method,
            request_path=self.request.env.request_path,
            scenario=self.track.scenario,
        )

    def raise_error_with_logging(self, exception_type, **statbox_kwargs):
        self.statbox.log(action='finished_with_error', error=exception_type.error, **statbox_kwargs)
        raise exception_type()

    def get_suitable_accounts_by_phone(self, phone_number, allow_disabled=True, allow_social=False,
                                       allow_neophonish=False):
        allowed_account_types = {ACCOUNT_TYPE_NORMAL, ACCOUNT_TYPE_PDD, ACCOUNT_TYPE_LITE}
        if allow_social:
            allowed_account_types.add(ACCOUNT_TYPE_SOCIAL)
        if allow_neophonish:
            allowed_account_types.add(ACCOUNT_TYPE_NEOPHONISH)

        all_phone_numbers = [phone_number] + get_alt_phone_numbers_of_phone_number(phone_number)

        bindings = self.blackbox.phone_bindings(
            phone_numbers=[p.e164 for p in all_phone_numbers],
            need_history=False,
            need_unbound=False,
        )

        uids = {b[u'uid'] for b in bindings}

        accounts, _ = get_many_accounts_with_phones_by_uids(uids, self.blackbox)

        suitable_accounts = []
        for phone_number in all_phone_numbers:
            for account in accounts:
                if account.type not in allowed_account_types:
                    continue
                if not account.person.firstname or not account.person.lastname:
                    continue
                if not account.is_enabled and not allow_disabled:
                    continue
                if (
                    account.disabled_status == ACCOUNT_DISABLED_ON_DELETION and
                    not self.can_restore_disabled_account(account)
                ):
                    continue
                if not (account.phones.secure and account.phones.secure.number == phone_number):
                    continue
                suitable_accounts.append(account)

        return suitable_accounts

    def filter_accounts_with_matching_names(self, accounts, entered_names):
        filtered_accounts = []
        for account in accounts:
            # Убираем детские аккаунт из результатирующей выборки
            if account.is_child:
                continue
            names = account.person.firstname, account.person.lastname
            firstname_factor, lastname_factor = compare_names(names, entered_names)
            if firstname_factor >= STRING_FACTOR_INEXACT_MATCH and lastname_factor >= STRING_FACTOR_INEXACT_MATCH:
                filtered_accounts.append(account)
        return filtered_accounts

    def check_global_counters(self, phone):
        if login_restore_counter.get_per_phone_buckets().hit_limit(phone.digital):
            self.raise_error_with_logging(RateLimitExceedError, rate_limited_by='phone')

        app_id = get_app_id_from_track(self.track)
        if should_ignore_per_ip_counters(app_id, phone.e164):
            log.debug('login_restore_per_ip_limit_counter ignored due to YANGO trusted phone code')
            return
        if login_restore_counter.get_per_ip_buckets().hit_limit_by_ip(self.client_ip):
            self.raise_error_with_logging(RateLimitExceedError, rate_limited_by='ip')

    def increase_global_counters(self, phone=None):
        if phone is not None:
            login_restore_counter.get_per_phone_buckets().incr(phone.digital)
        login_restore_counter.get_per_ip_buckets().incr(self.client_ip)

    def accounts_to_response(self, accounts, need_auth_info=False):
        response = []
        for account in accounts:
            login = account.human_readable_login
            if account.is_neophonish:
                login = ''  # по неофонишному алиасу ЧЯ акк не найдёт
            account_dict = {
                'uid': account.uid,
                'login': login,
                'primary_alias_type': self.get_alias_type(account),
            }
            if account.plus.has_plus:
                account_dict['has_plus'] = True
            if need_auth_info:
                auth_flows = []
                if (
                    account.is_enabled and
                    is_auth_by_sms_secure_enough_for_account(account) and
                    (account.is_neophonish or settings.ALLOW_AUTH_AFTER_LOGIN_RESTORE_FOR_ALL)
                ):
                    account_dict.update(
                        phone_number=dump_number(account.phones.secure.number, only_masked=True),
                    )
                    auth_flows.append(AUTH_FLOW_INSTANT)
                if login:
                    auth_flows.append(AUTH_FLOW_FULL)
                account_dict.update(
                    display_name=account.person.display_name.as_dict(),
                    allowed_auth_flows=auth_flows,
                )
            if account.person and account.person.default_avatar:
                account_dict.update(
                    default_avatar=account.person.default_avatar,
                    avatar_url=settings.GET_AVATAR_URL % (
                        account.person.default_avatar,
                        self.track.avatar_size or 'normal',
                    ),
                )
            response.append(account_dict)
        response = sorted(
            response,
            key=lambda d: d.get('has_plus', False),
            reverse=True,
        )
        self.response_values['accounts'] = response
        self.statbox.bind_context(uids=','.join([str(item['uid']) for item in response]))


class LoginRestoreSubmitView(BaseLoginRestoreView):
    """
    Создать трек для процесса восстановления логина.
    """
    require_track = False

    restore_step = 'submit'

    basic_form = LoginRestoreSubmitForm

    def process_request(self):
        self.process_basic_form()
        self.create_track(self.track_type, process_name=PROCESS_LOGIN_RESTORE)

        with self.track_transaction.rollback_on_error():
            # Retpath не патчим, вернем как есть при успешном восстановлении логина - далее он будет
            # проброшен на авторизацию, где его при необходимости пропатчат
            self.track.retpath = self.form_values['retpath']
            self.track.is_captcha_required = True
            self.track.restore_state = RESTORE_STATE_SUBMITTED
            self.track.gps_package_name = self.form_values['gps_package_name']

        self.response_values['track_id'] = self.track_id
        self.statbox.log(action='passed')


class LoginRestoreGetStateView(BaseLoginRestoreView):
    """
    Получить состояние процесса.
    """
    restore_step = 'get_state'

    def process_request(self):
        self.read_track()

        self.response_values['track_id'] = self.track_id
        self.response_values['retpath'] = self.track.retpath
        self.response_values['restore_state'] = self.track.restore_state

        if self.track.phone_confirmation_phone_number_original:
            # Номер, который был подходящим для восстановления логина на момент вызова CheckPhoneView
            phone = parse_phone_number(self.track.phone_confirmation_phone_number_original, self.track.country)
            self.statbox.bind_context(number=phone.masked_format_for_statbox)

            self.response_values['number'] = dump_number(phone)
            self.response_values['is_code_sent'] = bool(self.track.phone_confirmation_code)
            self.response_values['is_phone_confirmed'] = self.is_phone_confirmed_in_track(allow_by_flash_call=True)

            if self.track.restore_state in (RESTORE_STATE_PHONE_CONFIRMED, RESTORE_STATE_ACCOUNTS_FOUND):
                # Проверим, что номер всё еще подходит для восстановления логина только в случае,
                # если номер уже подтвержден - в данном случае при изменении условий
                # пользователю придется начинать процесс заново
                suitable_accounts = self.get_suitable_accounts_by_phone(phone)
                if not suitable_accounts:
                    self.raise_error_with_logging(SecurePhoneNotFoundError)

                if self.track.restore_state == RESTORE_STATE_ACCOUNTS_FOUND:
                    # Подходящие имя и фамилия введены, проверим актуальность и вернем логины
                    entered_names = self.track.user_entered_firstname, self.track.user_entered_lastname
                    filtered_accounts = self.filter_accounts_with_matching_names(suitable_accounts, entered_names)
                    if not filtered_accounts:
                        self.raise_error_with_logging(CompareNotMatchedError)
                    self.accounts_to_response(filtered_accounts)

        self.statbox.log(action='passed')


class LoginRestoreCheckPhoneView(
    BaseLoginRestoreView,
    BundleTrackedPhoneConfirmationMixin,
    BundlePhoneMixin,
    OctopusAvailabilityMixin,
):
    """
    Поиск аккаунтов по номеру телефона, отправка СМС с кодом подтверждения.
    Возможно изменение номера, повторная отправка СМС. Число попыток ограничено:
     1) поиск аккаунтов по номеру - глобальный счетчик на IP
     2) отправка СМС - глобальные счетчики по номеру и по IP (общие для разных процессов)
    """
    restore_step = 'check_phone'

    basic_form = LoginRestoreCheckPhoneForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        if self.track.restore_state != RESTORE_STATE_SUBMITTED:
            raise InvalidTrackStateError()

        if self.form_values['confirm_method'] == CONFIRM_METHOD_BY_CALL and not self.track.phone_valid_for_call:
            raise InvalidTrackStateError()
        elif self.form_values['confirm_method'] == CONFIRM_METHOD_BY_FLASH_CALL and not self.track.phone_valid_for_flash_call:
            raise InvalidTrackStateError()

        previous_confirmation_method = self.track.phone_confirmation_method or CONFIRM_METHOD_BY_SMS
        # Сейчас нельзя переходить между звонком с диктовкой и флеш-колом
        if previous_confirmation_method == CONFIRM_METHOD_BY_CALL and self.form_values['confirm_method'] == CONFIRM_METHOD_BY_FLASH_CALL:
            raise InvalidTrackStateError()
        elif previous_confirmation_method == CONFIRM_METHOD_BY_FLASH_CALL and self.form_values['confirm_method'] == CONFIRM_METHOD_BY_CALL:
            raise InvalidTrackStateError()

        if self.form_values['confirm_method'] in [CONFIRM_METHOD_BY_CALL, CONFIRM_METHOD_BY_FLASH_CALL]:
            if not self.are_calls_available():
                raise CallsShutDownError()

        phone = self.form_values['phone_number']
        self.statbox.bind_context(number=phone.masked_format_for_statbox)
        self.check_global_counters(phone=phone)

        is_sms_resend = self.old_phone and self.old_phone.e164 == phone.e164
        self.statbox.bind_context(is_sms_resend=is_sms_resend)

        if not is_sms_resend and not self.is_captcha_passed:
            # Капчу проверяем, если это не случай перепосылки смс на тот же номер
            self.raise_error_with_logging(UserNotVerifiedError)

        with self.track_transaction.commit_on_error():
            self.invalidate_captcha()

            suitable_accounts = self.get_suitable_accounts_by_phone(phone)
            if not suitable_accounts:
                # Неуспешные поиски аккаунтов по номеру ограничены по IP и капчей
                self.increase_global_counters(phone=None)
                self.raise_error_with_logging(SecurePhoneNotFoundError)

            if self.old_phone and self.old_phone.e164 != phone.e164:
                self.confirmation_info.reset_phone()
                self.confirmation_info.save()
            self.track.phone_confirmation_phone_number = phone.e164
            self.track.phone_confirmation_phone_number_original = phone.original
            self.track.country = self.form_values['country']
            self.track.display_language = self.form_values['display_language']
            self.track.phone_confirmation_method = self.form_values['confirm_method']

            try:
                self.send_code(
                    phone,
                    confirmation_method_changed=previous_confirmation_method != self.track.phone_confirmation_method,
                )
            except OctopusPermanentError:
                raise CreateCallFailed()

            if self.form_values['confirm_method'] == CONFIRM_METHOD_BY_SMS:
                self.response_values.update(
                    resend_timeout=self.confirmation_code_resend_timeout,
                    code_length=len(str(self.confirmation_info.code_value)),
                )
            elif self.form_values['confirm_method'] == CONFIRM_METHOD_BY_CALL:
                self.response_values.update(
                    code_length=self.call_confirmator.code_length,
                )
            elif self.form_values['confirm_method'] == CONFIRM_METHOD_BY_FLASH_CALL:
                calling_number = self.flash_call_confirmator.code_to_number[normalize_code(self.track.phone_confirmation_code)]
                template = PhoneNumber.parse(calling_number, allow_impossible=True).masked_for_flash_call
                self.response_values.update(
                    calling_number_template=template,
                    code_length=len(normalize_code(self.track.phone_confirmation_code)),
                )
            else:
                raise NotImplementedError()  # noqa

        self.statbox.log(action='passed')

    @cached_property
    def old_phone(self):
        # Важно вызвать эту пропертю до изменения трека, потому что перед подтверждением телефона
        # эти поля в треке перезаписываются телефоном из формы
        if not self.track.phone_confirmation_phone_number_original:
            return
        return parse_phone_number(
            self.track.phone_confirmation_phone_number_original,
            self.track.country,
        )

    @cached_property
    def call_confirmator(self):
        return self.build_calling_phone_confirmator(
            code_format=self.form_values['code_format'] or (
                'by_3_dash' if get_am_platform(self.track) == AmPlatform.IOS else 'by_3'
            ),
            phone_confirmation_language=self.form_values['display_language'],
            previous_number=self.old_phone,
        )

    @cached_property
    def flash_call_confirmator(self):
        return self.build_flash_calling_phone_confirmator(
            previous_number=self.old_phone,
            statbox=self.statbox,
        )

    def send_code(self, phone, confirmation_method_changed=False):
        try:
            if self.form_values['confirm_method'] == CONFIRM_METHOD_BY_SMS:
                restricter = LoginRestoreRestricter(
                    self.confirmation_info,
                    self.request.env,
                    self.statbox,
                    self.consumer,
                    phone,
                    self.track,
                )

                # NOTE Здесь нигде не участвует code_format, возможно потенциально его надо прокинуть вглубь,
                # но пока запроса не было.
                super(LoginRestoreCheckPhoneView, self).send_code(
                    phone,
                    restricter=restricter,
                    language=self.form_values['display_language'],
                    gps_package_name=self.track.gps_package_name,
                    force_new_code=confirmation_method_changed,
                )
            elif self.form_values['confirm_method'] == CONFIRM_METHOD_BY_CALL:
                self.call_confirmator.make_call(phone)
            elif self.form_values['confirm_method'] == CONFIRM_METHOD_BY_FLASH_CALL:
                self.flash_call_confirmator.make_call(phone)
            self.statbox.dump_stashes()
        except Exception:
            self.statbox.dump_stashes(action='finished_with_error')
            raise


class LoginRestoreConfirmPhoneView(BaseLoginRestoreView, BundleTrackedPhoneConfirmationMixin, BundlePhoneMixin):
    """
    Подтверждение телефона.
    В рамках процесса телефон можно подтвердить только один раз, для подтверждения
    другого нужно начать заново. Число попыток подтверждения ограничено счетчиком в треке
    (сбрасывается при смене номера).
    """
    restore_step = 'confirm_phone'

    basic_form = LoginRestoreConfirmPhoneForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        if self.track.restore_state == RESTORE_STATE_PHONE_CONFIRMED:
            raise ActionNotRequiredError()

        original_number = self.track.phone_confirmation_phone_number_original
        if not original_number or self.track.restore_state != RESTORE_STATE_SUBMITTED:
            raise InvalidTrackStateError()

        phone = parse_phone_number(original_number, self.track.country)
        self.statbox.bind_context(number=phone.masked_format_for_statbox)
        self.antifraud_logger.bind_context(user_phone=phone.e164)
        self.check_global_counters(phone=phone)
        suitable_accounts = self.get_suitable_accounts_by_phone(phone)
        if not suitable_accounts:
            self.raise_error_with_logging(SecurePhoneNotFoundError)

        with self.track_transaction.commit_on_error():
            self.confirm_code(self.form_values['code'])
            self.track.restore_state = RESTORE_STATE_PHONE_CONFIRMED

        self.statbox.log(
            action='passed',
            code_checks_count=self.track.phone_confirmation_confirms_count.get(default=0),
        )

    @cached_property
    def call_confirmator(self):
        confirmator = VerifyingPhoneCallConfirmator(
            track=self.track,
            confirmations_limit=settings.PHONE_VALIDATION_MAX_CALLS_CHECKS_COUNT,
            statbox=self.statbox,
            antifraud_logger=self.antifraud_logger,
            client=self.octopus_api,
        )
        return confirmator

    def confirm_code(self, code):
        try:
            if self.track.phone_confirmation_method in [CONFIRM_METHOD_BY_CALL, CONFIRM_METHOD_BY_FLASH_CALL]:
                self.call_confirmator.confirm_code(code)
                self.update_phone_confirmation_time()
            else:
                super(LoginRestoreConfirmPhoneView, self).confirm_code(code)
        except BaseBundleError as e:
            self.raise_error_with_logging(e.__class__)


class LoginRestoreCheckNamesView(BaseLoginRestoreView):
    """
    Ввод имени и фамилии и поиск подходящего аккаунта. При нахождении аккаунтов
    повторный вызов запрещен. Число попыток поиска ограничено глобальными счетчиками по номеру и IP.
    """
    restore_step = 'check_names'

    basic_form = LoginRestoreCheckNamesForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        if self.track.restore_state == RESTORE_STATE_ACCOUNTS_FOUND:
            raise ActionNotRequiredError()

        if self.track.restore_state != RESTORE_STATE_PHONE_CONFIRMED:
            raise InvalidTrackStateError()

        phone = parse_phone_number(self.track.phone_confirmation_phone_number_original, self.track.country)
        self.statbox.bind_context(number=phone.masked_format_for_statbox)
        self.check_global_counters(phone=phone)
        suitable_accounts = self.get_suitable_accounts_by_phone(phone)
        if not suitable_accounts:
            self.raise_error_with_logging(SecurePhoneNotFoundError)

        entered_names = self.form_values['firstname'], self.form_values['lastname']
        filtered_accounts = self.filter_accounts_with_matching_names(suitable_accounts, entered_names)
        if not filtered_accounts:
            # Увеличиваем счетчики по IP и по номеру
            self.increase_global_counters(phone=phone)
            self.raise_error_with_logging(CompareNotMatchedError)

        with self.track_transaction.rollback_on_error():
            self.track.user_entered_firstname, self.track.user_entered_lastname = entered_names
            self.track.restore_state = RESTORE_STATE_ACCOUNTS_FOUND

        self.accounts_to_response(filtered_accounts)
        self.statbox.log(action='passed')


class LoginRestoreCheckNamesSimpleView(LoginRestoreCheckNamesView):
    """
    В отличие от родителя, более гибкая (может вызываться вне рамок процесса или с треком иного типа)
    """
    process_name = None
    require_process = False
    allowed_processes = [PROCESS_LOGIN_RESTORE, PROCESS_WEB_REGISTRATION]
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = [RESTORE_SIMPLE_GRANT]
    basic_form = LoginRestoreCheckNamesSimpleForm

    def check_track_state(self):
        if not (
            self.track.phone_confirmation_phone_number and
            self.is_phone_confirmed_in_track(allow_by_flash_call=True)
        ):
            raise InvalidTrackStateError()

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.check_track_state()

        with self.track_transaction.commit_on_error():
            phone = parse_phone_number(self.track.phone_confirmation_phone_number)
            self.check_global_counters(phone=phone)
            suitable_accounts = self.get_suitable_accounts_by_phone(
                phone,
                allow_disabled=self.form_values['allow_disabled'],
                allow_social=self.form_values['allow_social'],
                allow_neophonish=self.form_values['allow_neophonish'],
            )

            entered_names = self.form_values['firstname'], self.form_values['lastname']
            filtered_accounts = self.filter_accounts_with_matching_names(suitable_accounts, entered_names)
            if not filtered_accounts:
                self.increase_global_counters(phone=phone)
            self.accounts_to_response(filtered_accounts, need_auth_info=True)

            allowed_registration_flows = []
            if settings.ALLOW_REGISTRATION:
                # Нового неофониша предлагаем зарегистрировать, только если старых не нашлось
                if (
                    settings.ALLOW_NEOPHONISH_REGISTRATION and
                    not any([acc.is_neophonish for acc in filtered_accounts])
                ):
                    allowed_registration_flows.append(REGISTER_FLOW_NEOPHONISH)
                allowed_registration_flows.append(REGISTER_FLOW_PORTAL)
            self.response_values.update(
                allowed_registration_flows=allowed_registration_flows,
            )
