# -*- coding: utf-8 -*-

from datetime import datetime

from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.api.views.bundle.exceptions import (
    BaseBundleError,
    CallsShutDownError,
    CompareNotMatchedError,
    CreateCallFailed,
    InvalidTrackStateError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAssertCaptchaMixin,
    BundlePersistentTrackMixin,
    BundlePhoneMixin,
    BundleTrackedPhoneConfirmationMixin,
    OctopusAvailabilityMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleHintAnswerCheckMixin
from passport.backend.api.views.bundle.phone.helpers import VerifyingPhoneCallConfirmator
from passport.backend.core.builders.messenger_api import (
    BaseMessengerApiError,
    get_mssngr_fanout_api,
)
from passport.backend.core.builders.octopus import OctopusPermanentError
from passport.backend.core.compare import FACTOR_BOOL_MATCH
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import send_mail_for_account
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.persistent_track import TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.email.email import punycode_email
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import datetime_to_unixtime

from ..exceptions import EmailSendLimitExceededError
from .base import (
    PHONE_BASED_RESTORE_METHODS,
    RESTORE_METHOD_EMAIL,
    RESTORE_METHOD_HINT,
    RESTORE_METHOD_PHONE,
    RESTORE_METHOD_PHONE_AND_2FA_FACTOR,
    RESTORE_STATE_METHOD_SELECTED,
    RESTORE_STEP_CHECK_2FA_FORM,
    RESTORE_STEP_CHECK_PIN,
)
from .controllers import BaseRestoreView
from .exceptions import (
    AnswerNotMatchedError,
    EmailCheckLimitExceededError,
    EmailNotMatchedError,
    PhoneChangedError,
    PhoneCheckLimitExceededError,
    PhoneNotMatchedError,
    PinCheckLimitExceededError,
    PinNotMatchedError,
)
from .factors import (
    CalculateFactorsMixin,
    get_names_check_status,
    get_user_env_check_status,
)
from .forms import (
    RestoreCheck2FAFormForm,
    RestoreCheckAnswerForm,
    RestoreCheckEmailForm,
    RestoreCheckPhoneForm,
    RestoreCheckPinForm,
    RestoreConfirmPhoneForm,
)
from .helpers import (
    get_restoration_key_email_notification_data,
    RestoreRestricter,
)


class BaseRestoreWithPhoneView(BaseRestoreView, BundleTrackedPhoneConfirmationMixin, BundlePhoneMixin):
    """
    Базовый View методов восстановления, проверяющих телефон.
    """
    def assert_phone_not_changed(self):
        """
        Проверим, что сохраненный в треке телефон актуален.
        """
        suitable_phones = [phone.number.e164 for phone in self.account_phones.suitable_for_restore]
        if (not suitable_phones or
                self.track.secure_phone_number not in suitable_phones):
            self.raise_error_with_logging(PhoneChangedError)


class RestoreCheckPhoneView(BaseRestoreWithPhoneView, OctopusAvailabilityMixin):
    """
    Проверяет, что введенный телефон пригоден для восстановления, и, в случае успеха, отправляет на него
    СМС с кодом подтверждения. Также выполняет повторную отправку СМС.
    """
    basic_form = RestoreCheckPhoneForm
    restore_step = 'check_phone'

    SEND_CODE_DESCRIBING_PROPERTIES = [
        'code_length',
        'resend_timeout',
    ]

    def get_phone_checks_left(self):
        return max(
            settings.SECURE_PHONE_CHECK_ERRORS_COUNT_LIMIT - self.track.secure_phone_checks_count.get(default=0),
            0,
        )

    def check_counters(self):
        if not self.get_phone_checks_left():
            self.raise_error_with_logging(PhoneCheckLimitExceededError)
        self.check_global_counters()

    def increase_phone_checks_counter(self):
        self.track.secure_phone_checks_count.incr()

    def increase_counters(self):
        self.increase_global_counters()
        self.increase_phone_checks_counter()

    def assert_phone_suitable_for_restore(self, phone_number):
        account_phone = self.account_phones.find(phone_number)
        self.statbox.bind_context(
            is_phone_found=bool(account_phone),
        )
        if account_phone:
            is_suitable = account_phone.is_suitable_for_restore
            self.statbox.bind_context(
                is_phone_suitable=is_suitable,
                is_phone_confirmed=account_phone.is_confirmed,
            )
            if is_suitable:
                return

        self.increase_counters()
        self.raise_error_with_logging(PhoneNotMatchedError)

    def build_call_confirmator(self, previous_number):
        return self.build_calling_phone_confirmator(
            client=self.octopus_api,
            code_format=self.form_values['code_format'] or 'by_3',
            phone_confirmation_language=self.form_values['display_language'],
            previous_number=previous_number,
        )

    def assert_phone_validated_for_confirm_method(self):
        method = self.form_values['confirm_method']
        if method == CONFIRM_METHOD_BY_SMS:
            pass
        elif method == CONFIRM_METHOD_BY_CALL:
            if not self.track.phone_valid_for_call:
                raise InvalidTrackStateError()
            if not self.track.phone_validated_for_call:
                raise InvalidTrackStateError()
            phone = self.form_values['phone_number']
            if self.track.phone_validated_for_call != phone.e164:
                raise InvalidTrackStateError()
        else:
            raise NotImplementedError()  # pragma: no cover

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_SELECTED],
            allowed_methods=PHONE_BASED_RESTORE_METHODS,
        )
        if self.is_phone_confirmed_in_track():
            raise InvalidTrackStateError()

        if self.form_values['confirm_method'] == CONFIRM_METHOD_BY_CALL:
            if not self.are_calls_available():
                raise CallsShutDownError()

        self.fill_basic_response()

        self.check_counters()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        self.assert_phone_validated_for_confirm_method()

        with self.track_transaction.commit_on_error():
            phone = self.form_values['phone_number']
            previous_phone = self.track.secure_phone_number
            is_phone_changed = previous_phone and previous_phone != phone.e164
            self.statbox.bind_context(is_phone_changed=is_phone_changed)
            confirmation_method = self.form_values['confirm_method']
            previous_confirmation_method = self.track.phone_confirmation_method or CONFIRM_METHOD_BY_SMS
            is_confirmation_method_changed = previous_confirmation_method != confirmation_method
            # сохраняем номер в двух форматах: так, как его ввёл пользователь - для показа пользователю в форме,
            # и в формате e164 для проверок и валидации пароля
            self.track.user_entered_phone_number = phone.original
            self.track.secure_phone_number = phone.e164
            if is_phone_changed or is_confirmation_method_changed:
                # сбрасываем значение, чтобы не было неконсистентности в случае задания неподходящего
                # номера вместо подходящего
                self.confirmation_info.reset_phone()
                self.confirmation_info.save()
            self.assert_phone_suitable_for_restore(phone)
            with self.statbox.make_context(action='restore.check_phone'):
                send_code_result = self.send_code(phone, previous_phone)

        for prop in self.SEND_CODE_DESCRIBING_PROPERTIES:
            if send_code_result.get(prop):
                self.response_values[prop] = send_code_result[prop]

    def send_code(self, phone, previous_number):
        retval = dict(
            code_length=None,
            resend_timeout=None,
        )
        try:
            if self.form_values['confirm_method'] == CONFIRM_METHOD_BY_SMS:
                super(RestoreCheckPhoneView, self).send_code(
                    phone,
                    restricter=RestoreRestricter(
                        self.confirmation_info,
                        self.request.env,
                        self.statbox,
                        self.consumer,
                        phone,
                        self.track,
                    ),
                    language=self.form_values['display_language'],
                    gps_package_name=self.track.gps_package_name,
                    code_format=self.form_values['code_format'],
                )
                self.statbox.dump_stashes(action='passed')
                retval['resend_timeout'] = self.confirmation_code_resend_timeout
            elif self.form_values['confirm_method'] == CONFIRM_METHOD_BY_CALL:
                call_confirmator = self.build_call_confirmator(previous_number)
                try:
                    call_confirmator.make_call(phone)
                except OctopusPermanentError:
                    raise CreateCallFailed()
                retval['code_length'] = call_confirmator.code_length
        except Exception:
            self.statbox.dump_stashes(action='finished_with_error')
            raise
        self.track.phone_confirmation_method = self.form_values['confirm_method']
        return retval


class RestoreConfirmPhoneView(BaseRestoreWithPhoneView):
    """
    Выполняет проверку кода подтверждения телефона.
    """
    basic_form = RestoreConfirmPhoneForm
    restore_step = 'confirm_phone'

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_SELECTED],
            allowed_methods=PHONE_BASED_RESTORE_METHODS,
        )
        if not self.track.phone_confirmation_code or self.is_phone_confirmed_in_track():
            raise InvalidTrackStateError()
        self.fill_basic_response()

        self.check_global_counters()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        self.assert_phone_not_changed()

        with self.track_transaction.commit_on_error():
            with UPDATE(
                self.account,
                self.request.env,
                {'action': 'confirm_phone_on_restore'},
            ):
                self.confirm_code(self.form_values['code'])
            self.track.phone_confirmation_is_confirmed = True
            if self.track.current_restore_method == RESTORE_METHOD_PHONE:
                # сообщаем, что метод пройден, если метод восстановления - телефон
                self.mark_method_passed()

        self.statbox.log(action='passed')

    def confirm_code(self, code):
        try:
            if self.track.phone_confirmation_method == CONFIRM_METHOD_BY_CALL:
                self.call_confirmator.confirm_code(code)
                phone = self.account.phones.by_number(self.call_confirmator.number)
                phone.confirm()
                self.statbox.log(
                    action='phone_confirmed',
                    number=phone.number.masked_format_for_statbox,
                    confirmation_time=datetime.now(),
                    code_checks_count=self.track.phone_confirmation_confirms_count.get(default=0),
                    uid=self.account.uid,
                    login=self.account.login,
                    phone_id=phone.id,
                )
            else:
                super(RestoreConfirmPhoneView, self).confirm_code(code)
        except BaseBundleError as e:
            self.raise_error_with_logging(e.__class__)

    def _get_number(self):
        # Номер для логгирования в BundleTrackedPhoneConfirmationMixin
        return PhoneNumber.parse(self.track.secure_phone_number)

    @cached_property
    def call_confirmator(self):
        return VerifyingPhoneCallConfirmator(
            track=self.track,
            confirmations_limit=settings.PHONE_VALIDATION_MAX_CALLS_CHECKS_COUNT,
            statbox=self.statbox,
            antifraud_logger=self.antifraud_logger,
            client=self.octopus_api,
        )


class RestoreCheckPinView(BaseRestoreWithPhoneView):
    """
    Выполняет проверку пин-кода 2ФА.
    """
    basic_form = RestoreCheckPinForm
    restore_step = RESTORE_STEP_CHECK_PIN

    def assert_pin_checks_track_limit_not_exceeded(self):
        if self.track.pin_check_errors_count.get() >= settings.ALLOWED_PIN_CHECK_FAILS_COUNT:
            self.raise_error_with_logging(PinCheckLimitExceededError)

    def increase_counters(self, pin, is_pin_valid):
        """
        Увеличить счетчик проверок пина (в случае неудачной проверки). Сбросить счетчик в случае успеха.
        Не увеличиваем глобальный счетчик восстановления, так как это делается на первом шаге
        двухфакторного восстановления.
        """
        new_failed_checks_count = self.account.totp_secret.failed_pin_checks_count
        failed_pins = self.track.failed_pins or []
        if not is_pin_valid and pin not in failed_pins:
            new_failed_checks_count += 1
            failed_pins.append(pin)
            with self.track_transaction.commit_on_error():
                self.track.failed_pins = failed_pins
                self.track.pin_check_errors_count.incr()
            self.statbox.bind_context(pin_checks_count=new_failed_checks_count)
        with UPDATE(
                self.account,
                self.request.env,
                {'action': 'pin_check_update'},
                force_history_db_external_events=True,
        ):
            self.account.totp_secret.failed_pin_checks_count = 0 if is_pin_valid else new_failed_checks_count

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_SELECTED],
            allowed_methods=[RESTORE_METHOD_PHONE_AND_2FA_FACTOR],
        )
        # Проверка того, что первый шаг 2ФА-восстановления пройден - телефон должен быть подтвержден
        if not self.is_phone_confirmed_in_track():
            raise InvalidTrackStateError()
        self.fill_basic_response()

        self.check_global_counters()
        # Счетчик проверок PIN у нас живет в атрибутах. Но он не защищен от
        # большого количества одновременных запросов.
        # Поэтому дублируем счетчик в треке, он должен работать существенно быстрее.
        self.assert_pin_checks_track_limit_not_exceeded()

        pin = self.form_values['pin']
        redirect_required = self.get_and_validate_account_from_track(pin_to_test=pin)
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        self.assert_phone_not_changed()

        with self.track_transaction.commit_on_error():
            self.track.last_restore_method_step = self.restore_step
        if self.account.totp_secret.failed_pin_checks_count >= settings.ALLOWED_PIN_CHECK_FAILS_COUNT:
            self.raise_error_with_logging(PinCheckLimitExceededError)

        is_pin_valid = self.userinfo_response['pin_status']
        self.increase_counters(pin, is_pin_valid)

        if not is_pin_valid:
            self.response_values['pin_checks_left'] = self.get_pin_checks_left()
            self.raise_error_with_logging(PinNotMatchedError)

        with self.track_transaction.commit_on_error():
            self.mark_method_passed()

        self.statbox.log(action='passed')


class RestoreCheckEmailView(BaseRestoreView, BundlePersistentTrackMixin):
    """
    Проверяет, что введенный email пригоден для восстановления, и, в случае успеха, отправляет на него
    письмо с кодом подтверждения.
    """
    basic_form = RestoreCheckEmailForm
    restore_step = 'check_email'

    def get_email_checks_left(self):
        return max(
            settings.EMAIL_CHECK_ERRORS_COUNT_LIMIT - self.track.email_checks_count.get(default=0),
            0,
        )

    def get_restoration_emails_count_left(self):
        return max(
            settings.RESTORATION_EMAILS_COUNT_LIMIT - self.track.restoration_emails_count.get(default=0),
            0,
        )

    def check_counters(self):
        if not self.get_email_checks_left():
            self.raise_error_with_logging(EmailCheckLimitExceededError)
        if not self.get_restoration_emails_count_left():
            self.raise_error_with_logging(EmailSendLimitExceededError)
        self.check_global_counters()

    def increase_email_checks_counter(self):
        self.track.email_checks_count.incr()
        self.statbox.bind_context(email_checks_count=self.track.email_checks_count.get())

    def increase_restoration_emails_counter(self):
        self.track.restoration_emails_count.incr()
        self.statbox.bind_context(restoration_emails_count=self.track.restoration_emails_count.get())

    def assert_email_suitable_for_restore(self, email):
        if not self.is_email_suitable_for_restore(email):
            self.increase_global_counters()
            self.increase_email_checks_counter()
            self.raise_error_with_logging(EmailNotMatchedError)

    def create_restoration_persistent_track(self, **track_params):
        track = self.create_persistent_track(
            self.account.uid,
            type_=TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK,
            expires_after=settings.RESTORATION_AUTO_LINK_LIFETIME_SECONDS,
            **track_params
        )
        return track

    def send_restoration_key_email(self, email, restoration_key, language, is_simple_format):
        template_name, info, context = get_restoration_key_email_notification_data(
            host=self.host,
            login=self.track.user_entered_login,
            account=self.account,
            language=language,
            restoration_key=restoration_key,
            is_simple_format=is_simple_format,
        )

        send_mail_for_account(
            template_name,
            info,
            context,
            self.account,
            specific_email=email,
            is_plain_text=is_simple_format,
            send_to_external=False,
            send_to_native=False,
        )
        self.increase_restoration_emails_counter()

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_SELECTED],
            allowed_methods=[RESTORE_METHOD_EMAIL],
        )
        self.fill_basic_response()

        self.check_counters()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()

        with self.track_transaction.commit_on_error():
            email = self.form_values['email']
            previous_email = self.track.user_entered_email
            self.statbox.bind_context(
                is_email_changed=previous_email and email != previous_email,
                is_simple_format=self.form_values['is_simple_format'],
            )
            self.track.user_entered_email = email
            self.track.is_email_check_passed = False
            self.track.restoration_key_created_at = None

            encoded_email = punycode_email(email)
            self.assert_email_suitable_for_restore(encoded_email)
            restoration_track = self.create_restoration_persistent_track(
                # Данные для сохранения в БД-трек
                retpath=self.track.retpath,
                user_entered_login=self.track.user_entered_login,
                user_entered_email=self.track.user_entered_email,
                # Это поле пишем для того, чтобы можно было связать по Статбокс-логам разорванный процесс
                initiator_track_id=self.track_id,
            )
            self.send_restoration_key_email(
                encoded_email,
                restoration_track.track_key,
                language=get_preferred_language(self.account),
                is_simple_format=self.form_values['is_simple_format'],
            )
            self.track.is_email_check_passed = True
            self.track.restoration_key_created_at = int(datetime_to_unixtime(restoration_track.created))

        self.statbox.bind_context(answer_checks_count=self.track.answer_checks_count.get())
        if settings.MESSENGER_FANOUT_API_ENABLED:
            try:
                was_online_sec_ago = get_mssngr_fanout_api().check_user_lastseen(uid=self.account.uid)
            except BaseMessengerApiError:
                self.statbox.log(
                    status='error',
                    error='messenger_api.request_failed',
                )
            else:
                self.statbox.bind(was_online_sec_ago=was_online_sec_ago)
        self.statbox.log(action='passed')


class RestoreCheckAnswerView(BaseRestoreView, BundleAssertCaptchaMixin, BundleHintAnswerCheckMixin):
    """
    Выполняет проверку КО.
    """
    basic_form = RestoreCheckAnswerForm
    restore_step = 'check_answer'

    def check_counters(self):
        self.check_global_counters()

    def increase_counters(self):
        self.track.answer_checks_count.incr()
        self.increase_global_counters()
        self.statbox.bind_context(answer_checks_count=self.track.answer_checks_count.get())

    def assert_captcha_passed_if_required(self):
        is_captcha_required = self.is_captcha_required_for_hint()
        self.statbox.bind_context(is_captcha_required=is_captcha_required)
        if is_captcha_required and not self.is_captcha_passed:
            self.raise_error_with_logging(UserNotVerifiedError)

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_SELECTED],
            allowed_methods=[RESTORE_METHOD_HINT],
        )
        self.fill_basic_response()

        self.check_counters()
        self.assert_captcha_passed_if_required()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()

        compare_status = self.compare_answers()
        with self.track_transaction.commit_on_error():
            if not compare_status:
                self.increase_counters()
                if self.is_captcha_required_for_hint():
                    self.invalidate_captcha()
                self.raise_error_with_logging(AnswerNotMatchedError)
            self.mark_method_passed()

        if settings.MESSENGER_FANOUT_API_ENABLED:
            try:
                was_online_sec_ago = get_mssngr_fanout_api().check_user_lastseen(uid=self.account.uid)
            except BaseMessengerApiError:
                self.statbox.log(
                    status='error',
                    error='messenger_api.request_failed',
                )
            else:
                self.statbox.bind(was_online_sec_ago=was_online_sec_ago)
        self.statbox.log(action='passed')


class RestoreCheck2FAFormView(RestoreCheckAnswerView, BaseRestoreWithPhoneView, CalculateFactorsMixin):
    """
    Проверяем короткую анкету 2ФА-пользователя.
    """
    basic_form = RestoreCheck2FAFormForm
    restore_step = RESTORE_STEP_CHECK_2FA_FORM

    def assert_captcha_passed_if_required(self):
        is_captcha_required = self.is_captcha_required_for_2fa_form()
        self.statbox.bind_context(is_captcha_required=is_captcha_required)
        if is_captcha_required and not self.is_captcha_passed:
            self.raise_error_with_logging(UserNotVerifiedError)

    def increase_counters(self):
        self.track.restore_2fa_form_checks_count.incr()
        self.increase_global_counters()
        self.statbox.bind_context({'2fa_form_checks_count': self.track.restore_2fa_form_checks_count.get()})

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_SELECTED],
            allowed_methods=[RESTORE_METHOD_PHONE_AND_2FA_FACTOR],
        )
        # Проверка того, что первый шаг 2ФА-восстановления пройден - телефон должен быть подтвержден
        if not self.is_phone_confirmed_in_track():
            raise InvalidTrackStateError()
        self.fill_basic_response()

        self.check_counters()
        self.assert_captcha_passed_if_required()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        self.assert_phone_not_changed()

        check_passed = self.check_conditions_passed()
        with self.track_transaction.commit_on_error():
            self.track.last_restore_method_step = self.restore_step
            if not check_passed:
                self.increase_counters()
                if self.is_captcha_required_for_2fa_form():
                    self.invalidate_captcha()
                self.raise_error_with_logging(CompareNotMatchedError)
            self.mark_method_passed()

        self.statbox.log(action='passed')

    def check_conditions_passed(self):
        """
        Проверяем критерии успешного заполнения анкеты:
         - Последний пароль + Фактор окружения
         - КВ + ФИО + Фактор окружения
        """
        answer_check_passed = False
        if self.form_values['answer'] and self.account.hint.is_set:
            answer_check_passed = self.compare_answers()

        # Обработчик multiple_names работает со списками имен и фамилий
        self.form_values['firstnames'] = [self.form_values['firstname']]
        self.form_values['lastnames'] = [self.form_values['lastname']]
        factors = self.calculate_factors('multiple_names', 'user_env_auths', 'password_matches', 'device_id')

        names_check_passed = get_names_check_status(factors)
        user_env_check_passed = get_user_env_check_status(factors) or factors['device_id']['factor'] == FACTOR_BOOL_MATCH
        password_check_passed = factors['password_matches']['factor'][0] == FACTOR_BOOL_MATCH
        self.statbox.bind_context(
            names_check_passed=names_check_passed,
            answer_check_passed=answer_check_passed,
            user_env_check_passed=user_env_check_passed,
            password_check_passed=password_check_passed,
        )
        return user_env_check_passed and (
            password_check_passed or
            answer_check_passed and names_check_passed
        )
