# -*- coding: utf-8 -*-
from passport.backend.api.common.phone import CONFIRM_METHOD_BY_SMS
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.api.views.bundle.restore.base import (
    NON_PHONE_BASED_RESTORE_METHODS,
    RESTORE_METHOD_PHONE,
    RESTORE_STATE_METHOD_PASSED,
)
from passport.backend.api.views.bundle.restore.method_controllers import (
    RestoreCheckPhoneView,
    RestoreConfirmPhoneView,
)
from passport.backend.core.types.phone_number.phone_number import (
    parse_phone_number,
    PhoneNumber,
)


class RestoreCheckNewPhoneView(RestoreCheckPhoneView):
    """
    Отправляет на новый номер СМС с кодом подтверждения. Также выполняет повторную отправку СМС.
    Ручка использует тот же набор флагов в треке, что и CheckPhone, но это безопасно, т.к. должно
    выполняться условие: в рамках одного процесса нельзя восстановиться по телефону и ввести новый номер.
    """
    restore_step = 'check_new_phone'

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_PASSED],
            allowed_methods=NON_PHONE_BASED_RESTORE_METHODS,
        )
        if self.is_phone_confirmed_in_track():
            raise InvalidTrackStateError()
        self.fill_basic_response()

        self.check_global_counters()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        if RESTORE_METHOD_PHONE not in self.allowed_methods_to_bind:
            raise InvalidTrackStateError()
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        phone = self.form_values['phone_number']
        self.response_values['number'] = dump_number(phone)
        self.assert_phone_not_compromised(phone.e164)

        with self.track_transaction.commit_on_error():
            self.track.country = self.form_values['country']
            previous_phone = self.track.phone_confirmation_phone_number
            is_phone_changed = previous_phone and previous_phone != phone.e164
            self.statbox.bind_context(is_phone_changed=is_phone_changed)
            confirmation_method = self.form_values['confirm_method']
            previous_confirmation_method = self.track.phone_confirmation_method or CONFIRM_METHOD_BY_SMS
            is_confirmation_method_changed = previous_confirmation_method != confirmation_method
            # сохраняем номер в двух форматах: так, как его ввёл пользователь - для показа пользователю в форме,
            # и в формате e164
            self.track.user_entered_phone_number = phone.original
            self.track.phone_confirmation_phone_number = phone.e164
            if is_phone_changed or is_confirmation_method_changed:
                self.confirmation_info.reset_phone()
                self.confirmation_info.save()
            with self.statbox.make_context(action='restore.check_new_phone'):
                send_code_result = self.send_code(phone, previous_phone)

        for prop in self.SEND_CODE_DESCRIBING_PROPERTIES:
            if send_code_result.get(prop):
                self.response_values[prop] = send_code_result[prop]


class RestoreConfirmNewPhoneView(RestoreConfirmPhoneView):
    """
    Выполняет проверку кода подтверждения для привязываемого телефона.
    """
    restore_step = 'confirm_new_phone'

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.assert_track_valid(
            allowed_states=[RESTORE_STATE_METHOD_PASSED],
            allowed_methods=NON_PHONE_BASED_RESTORE_METHODS,
        )
        if (not self.track.user_entered_phone_number or
            not self.track.phone_confirmation_phone_number or
            not self.track.phone_confirmation_code or
            self.is_phone_confirmed_in_track()
        ):
            raise InvalidTrackStateError()
        self.fill_basic_response()

        self.check_global_counters()

        redirect_required = self.get_and_validate_account_from_track()
        if redirect_required:
            return
        if RESTORE_METHOD_PHONE not in self.allowed_methods_to_bind:
            raise InvalidTrackStateError()
        self.assert_track_and_restoration_key_valid_for_account()
        self.assert_restore_method_suitable()
        phone = parse_phone_number(self.track.user_entered_phone_number, country=self.track.country)
        self.response_values['number'] = dump_number(phone)
        self.assert_phone_not_compromised(self.track.phone_confirmation_phone_number)

        with self.track_transaction.commit_on_error():
            self.confirm_code(self.form_values['code'])
            self.track.phone_confirmation_is_confirmed = True

        self.statbox.log(action='passed')

    def _get_number(self):
        # Номер для логгирования в BundleTrackedPhoneConfirmationMixin
        return PhoneNumber.parse(self.track.phone_confirmation_phone_number)
