# -*- coding: utf-8 -*-
import logging
import os
import time

from passport.backend.api.common.processes import PROCESS_RESTORE
from passport.backend.api.common.restore import restore_log
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    AccountNotFoundError,
    CompareNotMatchedError,
    EulaIsNotAcceptedError,
    HistoryDBApiUnavailableError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAssertCaptchaMixin
from passport.backend.api.views.bundle.states import RateLimitExceeded
from passport.backend.core.conf import settings
from passport.backend.core.counters import restore_semi_auto_compare_counter
from passport.backend.core.historydb.converter import RestoreEntryConverter
from passport.backend.core.historydb.entry import RestoreEntry
from passport.backend.core.historydb.events import (
    ACTION_RESTORE_SEMI_AUTO_REQUEST,
    EVENT_ACTION,
    EVENT_INFO_RESTORE_ID,
    EVENT_INFO_RESTORE_REQUEST_SOURCE,
    EVENT_INFO_RESTORE_STATUS,
    RESTORE_STATUS_PASSED,
    RESTORE_STATUS_REJECTED,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.serializers.logs.historydb.runner import HistorydbActionRunner
from passport.backend.core.types.email.email import (
    is_yandex_email,
    normalize_email,
    normalize_email_with_phonenumber,
    punycode_email,
)
from passport.backend.core.types.login.login import (
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.restore_id import RestoreId
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_str

from ..base import (
    GetAccountForRestoreMixin,
    RestoreSemiAutoBaseMixin,
)
from ..exceptions import ContactEmailFromSameAccountError
from ..factors import (
    CalculateFactorsMixin,
    get_names_birthday_matches,
    get_user_env_check_status,
)
from .forms import (
    RestoreSemiAutoCommitShortForm,
    RestoreSemiAutoSubmitForm,
    RestoreSemiAutoSubmitWithCaptchaForm,
    RestoreSemiAutoValidateForm,
)
from .helpers import (
    get_question_from_account_as_list,
    prepare_questions_for_response,
)


log = logging.getLogger('passport.api.views.bundle.restore.semi_auto.controllers')

USER_ENTERED_FIELDS = (
    'login',
    'email',
    'question_id',
    'question',
    'answer',
    'phone_number',
)


class RestoreSemiAutoViewBase(BaseBundleView, RestoreSemiAutoBaseMixin, GetAccountForRestoreMixin):
    """
    Базовый класс для форм расширенного восстановления
    """
    process_name = PROCESS_RESTORE

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
    )

    required_grants = ['restore.semi_auto']
    statbox_mode = 'restore_semi_auto'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.statbox_mode,
            track_id=self.track.track_id,
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            login=self.track.user_entered_login,
            yandexuid=self.cookies.get('yandexuid'),
            request_source=self.request_source,
            version=self.version,
            is_for_learning=self.track.is_for_learning,
        )

    @cached_property
    def request_source(self):
        return self.track.request_source

    @cached_property
    def restore_id(self):
        return RestoreId.from_args(
            ('%x' % get_current_host().get_id()).upper(),
            os.getpid(),
            time.time(),
            self.account.uid,
            self.track_id,
        ).to_string()

    @cached_property
    def version(self):
        return self.track.version

    def check_eula_accepted(self):
        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

    def write_events_to_event_log(self, restore_status):
        runner = HistorydbActionRunner(
            {},
            uid=self.account.uid,
            user_ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            host_id=get_current_host().get_id(),
        )
        runner.execute((EVENT_ACTION, ACTION_RESTORE_SEMI_AUTO_REQUEST))
        runner.execute((EVENT_INFO_RESTORE_ID, self.restore_id))
        runner.execute((EVENT_INFO_RESTORE_REQUEST_SOURCE, self.request_source))
        runner.execute((EVENT_INFO_RESTORE_STATUS, restore_status))

    def write_factors_to_restore_log(self, factors):
        entry = RestoreEntry(
            action=ACTION_RESTORE_SEMI_AUTO_REQUEST,
            uid=self.account.uid,
            restore_id=self.restore_id,
            data=factors,
        )
        restore_log.debug(RestoreEntryConverter().convert(entry))

    def is_ip_limit_exceeded(self, only_check=False):
        if restore_semi_auto_compare_counter.is_ip_limit_exceeded(self.client_ip, only_check):
            self.state = RateLimitExceeded()
            self.statbox.log(action='finished_with_state', state=self.state.state)
            return True
        return False

    def is_uid_limit_exceeded(self, only_check=False):
        uid = self.track.uid or (getattr(self, 'account') and self.account.uid)
        if uid and restore_semi_auto_compare_counter.is_uid_limit_exceeded(uid, only_check):
            self.state = RateLimitExceeded()
            self.statbox.log(action='finished_with_state', state=self.state.state)
            return True
        return False

    def raise_if_retry_required(self, factors):
        """Проверим, получен ли минимально необходимый набор данных от API Яндекса"""
        exception = None
        any_find_password_call_failed = 'passwords' in factors and not all(factors['passwords']['api_statuses'])
        auths_aggregated_status = factors.get('auths_aggregated_runtime_api_status', True)
        events_status = factors['historydb_api_events_status']
        if not events_status or not auths_aggregated_status or any_find_password_call_failed:
            # Не получены события, либо не удалось проверить введенный пароль в HistoryDB
            exception = HistoryDBApiUnavailableError()
        if exception is not None:
            self.statbox.log(
                action='finished_with_error',
                error=exception.error,
            )
            raise exception

    def validate_contact_email(self, check_personal_data=False):
        if (not self.form_values['contact_email'] or
                self.request_source == settings.RESTORE_REQUEST_SOURCE_FOR_CHANGE_HINT):
            # Если в треке или на форме нет контактного email-адреса, либо анкета предназначена
            # для смены КВ/КО (у пользователя есть доступ к аккаунту), не делаем проверку
            return

        contact_email = normalize_email_with_phonenumber(
            # ЧЯ отдает email-ы в punycode, приведем для сравнения входной email в punycode
            punycode_email(self.form_values['contact_email']),
            country=self.track.country,
        )
        login_part = raw_login_from_email(contact_email)
        if login_part.isdigit() and not check_personal_data:
            # Не будем раскрывать информацию о цифровом алиасе аккаунта
            return

        try:
            # PASSP-21736: Не давать удаленным пользователям указывать в анкете почтовый адрес,
            # совпадающий с логином
            # Проверяем, если смогли добраться до аккаунта
            if self.account is None:
                self.get_and_validate_account(
                    self.track.user_entered_login,
                    skip_validation=True,
                    skip_statbox_log=True,
                )

            if (self.account.is_normal and is_yandex_email(contact_email) and
                    self.account.normalized_login == normalize_login(login_part)):
                raise ContactEmailFromSameAccountError()

            if (self.account.is_pdd and
                    normalize_email(self.account.login) == normalize_email(contact_email)):
                raise ContactEmailFromSameAccountError()
        except AccountNotFoundError:
            pass

        if self.track.emails is None:
            # В треке не сохранены email'ы аккаунта - пропускаем для обратной совместимости
            return
        if contact_email in self.track.emails.split():
            raise ContactEmailFromSameAccountError()

    def increment_uid_counter(self):
        restore_semi_auto_compare_counter.is_uid_limit_exceeded(self.account.uid)

    def increment_ip_counter(self):
        restore_semi_auto_compare_counter.is_ip_limit_exceeded(self.client_ip)


class RestoreSemiAutoSubmitView(RestoreSemiAutoViewBase):
    """
    Получаем логин и данные о неудачном автовосстановлении, создаем и возвращаем трек
    """
    basic_form = RestoreSemiAutoSubmitForm

    def fill_track_with_form_data(self):
        for field in USER_ENTERED_FIELDS:
            if self.form_values[field]:
                setattr(self.track, 'user_entered_%s' % field, smart_str(self.form_values[field]))
        self.track.request_source = self.form_values['request_source']
        self.track.is_for_learning = self.is_form_suitable_for_learning(self.form_values['request_source'])
        self.track.device_application = self.form_values['app_id']

    def process_request(self):
        self.process_basic_form()
        self.create_track('restore')

        with self.track_transaction.commit_on_error():
            self.fill_track_with_form_data()
            self.setup_multistep_process()
            self.statbox.log(action='track_created')
            self.response_values['track_id'] = self.track_id


class RestoreSemiAutoSubmitWithCaptchaView(RestoreSemiAutoViewBase, GetAccountForRestoreMixin, BundleAssertCaptchaMixin):
    """
    Проверяем капчу и применимость аккаунта
    """
    basic_form = RestoreSemiAutoSubmitWithCaptchaForm

    require_track = True

    @cached_property
    def request_source(self):
        return self.form_values['request_source']

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.statbox.bind_context(
            login=self.form_values['login'],
            step='submit_with_captcha',
            is_unconditional_pass=self.form_values['is_unconditional_pass'],
        )

        if not self.is_captcha_passed:
            self.statbox.log(
                action='finished_with_error',
                error=UserNotVerifiedError.error,
            )
            raise UserNotVerifiedError()

        processing_finished = self.get_and_validate_account(
            self.form_values['login'],
            skip_validation=self.form_values['is_unconditional_pass'],
            check_domain_support=True,
        )
        if processing_finished:
            return

        with self.track_transaction.commit_on_error():
            self.setup_track_for_semi_auto_form(
                self.form_values['login'],
                self.form_values['request_source'],
                is_unconditional_pass=self.form_values['is_unconditional_pass'],
            )
            self.statbox.log(
                is_for_learning=self.track.is_for_learning,
                action='submit_with_captcha_passed',
            )


class RestoreSemiAutoFormDataView(RestoreSemiAutoViewBase, GetAccountForRestoreMixin):
    """
    Отдаем данные, необходимые для показа формы, по треку
    """
    require_track = True

    def process_request(self):
        self.read_track()
        self.statbox.log(action='form_data_requested')

        processing_finished = self.get_and_validate_account(
            self.track.user_entered_login,
            check_domain_support=True,
            emails=True,
        )
        if processing_finished:
            return

        if self.is_ip_limit_exceeded(only_check=True) or self.is_uid_limit_exceeded(only_check=True):
            return

        with self.track_transaction.commit_on_error():
            questions = get_question_from_account_as_list(self.account)
            self.fill_track_with_account_data_for_semi_auto_form()
            self.track.questions = questions
            self.response_values['questions'] = prepare_questions_for_response(questions)


class RestoreSemiAutoValidateView(RestoreSemiAutoViewBase):
    """
    Ручка AJAX-валидации данных формы
    """
    basic_form = RestoreSemiAutoValidateForm

    require_track = True

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.validate_contact_email()


class RestoreShortFormCommitView(RestoreSemiAutoViewBase, GetAccountForRestoreMixin, CalculateFactorsMixin):
    """
    Проверка укороченной версии анкеты (PASSP-9775) восстановления пароля, которая
    будет использоваться в случае отсутствия прикрепленного защищенного номера
    фронтендом. В отличие от полной версии не высылает письма в саппорт, в остальном
    функционально почти идентична.
    """
    basic_form = RestoreSemiAutoCommitShortForm
    statbox_mode = 'restore_short_form'
    require_track = True

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.statbox.log(action='submitted_short_form')
        self.check_eula_accepted()

        if self.is_ip_limit_exceeded():
            return

        processing_finished = self.get_and_validate_account(self.track.user_entered_login, need_phones=True)
        if self.account and self.account.phones.secure:
            raise AccountInvalidTypeError()
        if processing_finished:
            return

        if self.is_uid_limit_exceeded(only_check=True):
            # Проверим, что число успешных проверок для этого уида не превышает ограничение
            return

        with self.track_transaction.commit_on_error():
            self.fill_track_with_account_data_for_semi_auto_form()
            factors = self.calculate_factors(
                'names',
                'simple_birthday',
                'user_env_auths',
                'reg_country_city',
                'registration_date',
            )
            names_birthday_check_passed = get_names_birthday_matches(factors).check_passed
            whole_check_passed = names_birthday_check_passed and get_user_env_check_status(factors)

            self.statbox.log(action='compared', check_passed=whole_check_passed)

            restore_status = RESTORE_STATUS_PASSED if whole_check_passed else RESTORE_STATUS_REJECTED
            factors['request_source'] = self.request_source
            factors['restore_status'] = restore_status

            self.track.factors = factors
            self.raise_if_retry_required(factors)

            # Разрешаем сменить пароль в случаях, когда пройдена проверка по IP + ФИО/ДР
            if not whole_check_passed:
                raise CompareNotMatchedError()

            # Увеличим счетчик успешных проверок по UID
            self.increment_uid_counter()
            self.track.is_short_form_factors_checked = True
