# -*- coding: utf-8 -*-
import logging

from passport.backend.core.compare import FACTOR_NOT_SET
from passport.backend.core.conf import settings
from passport.backend.utils import file
from six.moves.urllib.parse import quote


log = logging.getLogger('passport.api.views.bundle.restore.semi_auto.helpers')

MULTIPLE_SELECTION_ENTITIES = (
    'phone_numbers',
    'emails',
    'email_blacklist',
    'email_collectors',
    'email_whitelist',
    'outbound_emails',
    'email_folders',
    'services',
    'delivery_addresses',
    'social_accounts',
)

NON_DATA_ENTITIES = (
    'oauth_api_status',
    'request_info',
    'is_for_learning',
    'historydb_api_events_status',
    'version',
    'registration_ip',
    'auths_aggregated_runtime_api_status',
    'restore_attempts',
    # Эти поля попадают в словарь с данными уже после вычисления результата, но в случае фатальной ошибки
    # при повторном вызове ручки эти поля будут заданы.
    'restore_status',
    'tensornet_estimate',
    'tensornet_status',
    'decision_source',
)

# Значение, которое требуется передавать в TensorNet в случае, если фактор не вычислялся (имеет значение -1)
FEATURE_NOT_CALCULATED_VALUE = 10000


def _format_delivery_addresses(addresses, language):
    notifications = settings.translations.NOTIFICATIONS[language]
    formatted_addresses = []
    for address in addresses or []:
        formatted_address = notifications['restore.semi_auto.delivery_address'] % address
        if address.get('suite'):
            formatted_address = ', '.join([
                formatted_address,
                notifications['restore.semi_auto.delivery_address_suite'] % address,
            ])
        formatted_addresses.append(formatted_address)
    if addresses:
        return '; '.join(formatted_addresses)
    return addresses


def _make_subject(login, language, request_source=None):
    notifications = settings.translations.NOTIFICATIONS[language]
    tanker_key = 'restore.semi_auto.otrs_message_subject'
    if request_source == settings.RESTORE_REQUEST_SOURCE_FOR_CHANGE_HINT:
        tanker_key = 'restore.semi_auto.otrs_message_subject_for_change_hint'
    return notifications[tanker_key] % login


def _make_adm_form_data_url(restore_id):
    quoted_restore_id = quote(restore_id)
    return settings.RESTORE_ADM_FORM_DATA_URL % dict(restore_id=quoted_restore_id)


def _format_social_accounts(entered_accounts_data):
    """
    Нужно подготовить строчку с описанием соц. аккаунтов, в которых авторизовался пользователь на анкете.
    В лучшем случае выводим строку в виде 'Имя Фамилия (ссылка на аккаунт), ...'. Части данных может не
    быть, в худшем случае известно только имя соц. сети: 'Google, ...'.
    """
    account_descriptions = []
    for account_info in entered_accounts_data:
        names = []
        for field in ('firstname', 'lastname'):
            if account_info.get(field):
                names.append(account_info[field])
        link = account_info['links'][0] if account_info['links'] else account_info['provider']['name'].capitalize()
        if names:
            account_description = ' '.join(names + ['(%s)' % link])
        else:
            account_description = link
        account_descriptions.append(account_description)
    return ', '.join(account_descriptions)


def get_message_context_from_factors(login, factors, check_passed, restore_id, request_source=None):
    """
    Получить контекст для подстановки в шаблон письма анкеты. Контекст строится на основе факторов, вычисленных
    на шагах анкеты.
    """
    language = factors['request_info']['language']
    notifications = settings.translations.NOTIFICATIONS[language]
    context = {
        'subject': _make_subject(login, language, request_source=factors['request_info']['request_source']),
        'adm_form_data_url': _make_adm_form_data_url(restore_id),
        'real_reason': None,
    }
    if request_source == settings.RESTORE_REQUEST_SOURCE_FOR_CHANGE_HINT and factors['request_info'].get('real_reason'):
        reason_key = 'restore.otrs.message_type.%s' % factors['request_info']['real_reason']
        context['real_reason'] = notifications[reason_key]

    for field in (
        'user_enabled',
        'contact_reason',
        'contact_email',
        'language',
    ):
        context[field] = factors['request_info'][field]

    context['firstname'] = u', '.join(factors['names']['entered']['firstnames'])
    context['lastname'] = u', '.join(factors['names']['entered']['lastnames'])

    for field in (
        'birthday',
        'registration_date',
        'registration_country',
        'registration_city',
    ):
        context[field] = factors[field]['entered']
        if field == 'registration_city':
            context[field] = context[field] or notifications['restore.not_set.masculine']

    for field in (
        'emails',
        'phone_numbers',
        'outbound_emails',
        'delivery_addresses',
        'email_folders',
        'email_collectors',
        'email_whitelist',
        'email_blacklist',
        'social_accounts',
    ):
        if field == 'delivery_addresses':
            value = _format_delivery_addresses(factors[field]['entered'], language)
        elif field == 'social_accounts':
            value = _format_social_accounts(factors[field]['entered_accounts'])
        else:
            entered_value = factors[field]['entered']
            value = ', '.join(entered_value) if isinstance(entered_value, list) else entered_value
        context[field] = value or notifications['restore.not_specified.neuter']

    context['question'] = factors['answer']['entered']['question'] or notifications['restore.not_specified.neuter']
    context['login'] = login
    context['user_ip'] = factors['user_env_auths']['actual']['ip']
    context['factors'] = factors
    context['check_passed'] = check_passed
    return context


def get_question_from_account_as_list(account):
    if not account.hint.is_set:
        return []
    return [
        dict(
            id=account.hint.question.id,
            text=account.hint.question.text,
        ),
    ]


def prepare_questions_for_response(questions):
    """
    Подменить id в списке вопросов порядковым номером для однозначной идентификации.
    Оригинальный список сохранен в треке для восстановления исходного вопроса в ручке commit.
    """
    return [dict(id=i, text=question['text']) for (i, question) in enumerate(questions)]


def flatten_factors_list(factors, key):
    flat_factors = {}
    for index, factor_value in enumerate(factors):
        nested_key = '%s_%s' % (key, index)
        flat_factors.update(flatten_factors(factor_value, nested_key))
    return flat_factors


def flatten_factors_dict(factors, key):
    flat_factors = {}
    for factor_name, factor_value in factors.items():
        nested_key = '%s_%s' % (key, factor_name)
        flat_factors.update(flatten_factors(factor_value, nested_key))
    return flat_factors


def flatten_factors(factors, key):
    if isinstance(factors, dict):
        return flatten_factors_dict(factors, key)
    elif isinstance(factors, list):
        return flatten_factors_list(factors, key)
    elif isinstance(factors, float):
        return {key: round(factors, 3)}
    return {key: factors}


def transform_entity_factors(factors, entity):
    if not isinstance(factors, dict):
        return factors

    factors = dict(factors)
    factors.pop('change_count', None)
    if entity == 'passwords':
        factors.pop('entered_count')
    if entity in MULTIPLE_SELECTION_ENTITIES:
        if entity == 'social_accounts':
            factors.pop('entered_accounts_count')
            factors.pop('entered_profiles_count')
        else:
            factors.pop('entered_count')
        for field in ('account_count', 'actual_count', 'history_count', 'account_profiles_count'):
            factors.pop(field, None)
        matches_count = factors.pop('matches_count')
        factors['has_matches'] = int(matches_count >= 1)
    return factors


def get_features_from_factors(factors):
    """
    На основании данных анкеты (factors) подготавливает список факторов для передачи в TensorNet.
    В случае ошибки пишет в лог и отдает None.
    """
    all_features = {}
    for entity, data in factors.items():
        if entity in NON_DATA_ENTITIES:
            continue
        entity_factors = data['factor']
        entity_factors = transform_entity_factors(entity_factors, entity)
        all_features.update(flatten_factors(entity_factors, entity))

    allowed_features = file.read_file(settings.RESTORE_SEMI_AUTO_FEATURES_FILE).strip().split()

    if not allowed_features:
        log.error('Features file is empty')
        return
    filtered_features = []
    for feature_name in allowed_features:
        if feature_name not in all_features:
            log.error('Unknown feature "%s"', feature_name)
            return
        feature_value = all_features[feature_name]
        if feature_value == FACTOR_NOT_SET:
            feature_value = FEATURE_NOT_CALCULATED_VALUE
        filtered_features.append(feature_value)
    return filtered_features
