# -*- coding: utf-8 -*-
import logging
import os

from passport.backend.api.common.mail import create_semi_auto_message
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    BaseBundleError,
    CompareNotMatchedError,
    InvalidTrackStateError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.restore.base import GetAccountForRestoreMixin
from passport.backend.api.views.bundle.restore.exceptions import SendmailFailedError
from passport.backend.api.views.bundle.restore.factors import (
    CalculateFactorsMixin,
    get_names_birthday_check_status,
    get_user_env_check_status,
)
from passport.backend.api.views.bundle.restore.semi_auto.base import (
    DECISION_SOURCE_BASIC_FORMULA,
    DECISION_SOURCE_TENSORNET,
    DECISION_SOURCE_UNCONDITIONAL,
    get_next_step,
    get_step_fields,
    MULTISTEP_FORM_VERSION,
    RESTORE_MESSAGE_REQUEST_SOURCE_FOR_LEARNING,
    RESTORE_MESSAGE_REQUEST_SOURCE_FOR_POSITIVE_DECISION_RETRY,
    STEP_2_RECOVERY_TOOLS,
    STEP_5_SERVICES_DATA,
    STEP_6_FINAL_INFO,
    STEP_FINISHED,
    STEP_TO_FACTORS_MAPPING,
    STEP_TO_SCHEMA_MAPPING,
    STEP_TO_STRICT_SCHEMA_MAPPING,
)
from passport.backend.api.views.bundle.restore.semi_auto.controllers import RestoreSemiAutoViewBase
from passport.backend.api.views.bundle.restore.semi_auto.helpers import (
    get_features_from_factors,
    get_message_context_from_factors,
    get_question_from_account_as_list,
    prepare_questions_for_response,
)
from passport.backend.api.views.bundle.restore.semi_auto.step_forms import RestoreSemiAutoMultiStepForm
from passport.backend.api.views.bundle.states import (
    ProcessRestartRequired,
    RedirectToRestorationPassed,
)
from passport.backend.core.conf import settings
from passport.backend.core.historydb.events import (
    RESTORE_STATUS_PASSED,
    RESTORE_STATUS_PENDING,
    RESTORE_STATUS_REJECTED,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import send_message
from passport.backend.core.types.phone_number.phone_number import parse_phone_number
from passport.backend.core.types.question import USER_QUESTION_ID
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.validators.jsonschema_adapter import validate_jsonschema_form
from tensornet import (
    TensorNet,
    TensorNetError,
)


log = logging.getLogger('passport.api.views.bundle.restore.semi_auto.step_controllers')


class RestoreSemiAutoMultiStepViewBase(RestoreSemiAutoViewBase):

    basic_form = RestoreSemiAutoMultiStepForm

    require_track = True

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.statbox_mode,
            track_id=self.track.track_id,
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            login=self.track.user_entered_login,
            uid=self.track.uid,
            yandexuid=self.cookies.get('yandexuid'),
            request_source=self.request_source,
            step=self.track.semi_auto_step,
            version=self.version,
            is_unconditional_pass=self.track.is_unconditional_pass,
            is_for_learning=self.track.is_for_learning,
        )

    def is_process_consistent(self):
        # Проверим, что в процессе заполнения форм анкета не была обновлена
        if self.version != MULTISTEP_FORM_VERSION:
            log.warning(
                u'Inconsistent process: current version %s, user version %s',
                MULTISTEP_FORM_VERSION,
                self.version,
            )
            self.state = ProcessRestartRequired()
            self.statbox.log(action='finished_with_state', state=self.state.state)
            return False
        return True

    def check_step_applicability(self, current_step):
        # Проверим корректность текущего шага
        if current_step == STEP_FINISHED:
            raise ActionNotRequiredError()
        if current_step not in STEP_TO_SCHEMA_MAPPING:
            raise InvalidTrackStateError()

    def any_counter_exceeded(self):
        need_check_ip = True
        factors = self.track.factors
        if factors and 'restore_status' in factors:
            # Если решение уже приняли, проверять счетчик по IP не нужно, т.к. при принятии решения
            # счетчик всегда увеличивается, но после принятия решения могут быть дополнительные шаги
            need_check_ip = False
        return (
            need_check_ip and self.is_ip_limit_exceeded(only_check=True) or
            self.is_uid_limit_exceeded(only_check=True)
        )

    def translate_question_id(self):
        """
        Заменить ID КВ на настоящий ID, сохраненный в треке.
        """
        if self.form_values.get('question_answer'):
            incoming_id = self.form_values['question_answer']['question_id']
            # по умолчанию, любой вариант ID, кроме сохраненных в треке, считаем пользовательским
            original_id = USER_QUESTION_ID
            if self.track.questions:
                if incoming_id >= 0 and incoming_id < len(self.track.questions):
                    original_question = self.track.questions[incoming_id]
                    original_id = original_question['id']
                    self.form_values['question_answer']['question'] = original_question['text']
            self.form_values['question_answer']['question_id'] = original_id

    def validate_phone_numbers(self):
        """
        Проверяем, что переданные строки похожи на телефоны, а также отсутствие дубликатов.
        Сам результат парсинга не сохраняется, так как мы можем ошибиться со страной для конкретного телефона.
        В случае, если телефон передан в международном формате, никаких проблем нет, т. к. страна не учитывается.
        """
        raw_phones = self.form_values['phone_numbers'] or []
        phones = set()

        for raw_phone in raw_phones:
            # проверку допустимости и валидности пропускаем, так как страна может быть неверной.
            phone = parse_phone_number(raw_phone, self.track.country, allow_impossible=True)
            if not phone:
                raise ValidationFailedError(['phone_numbers.invalid'])
            if phone in phones:
                raise ValidationFailedError(['phone_numbers.duplicate'])
            phones.add(phone)

    def process_step_form(self, check_missing_fields=True, check_personal_data=False):
        """
        Обработать JSON-форму:
        - декодировать JSON из HTTP-формы
        - провалидировать JSON в соответствии с требованиями текущего шага анкеты
        @param check_missing_fields: нужно ли проверять отсутствие требуемых в форме полей
        @param check_personal_data: признак того, что при валидации допустимо раскрывать личные данные
         (используется для валидации контактного email-адреса)
        """
        # Валидируем базовую форму, в т.ч. декодируем json-документ
        self.process_basic_form()
        current_step = self.track.semi_auto_step
        json_data = self.form_values['json_data']
        # Провалидируем JSON-данные в соответствии с текущим шагом
        self.check_step_applicability(current_step)
        schema = STEP_TO_STRICT_SCHEMA_MAPPING if check_missing_fields else STEP_TO_SCHEMA_MAPPING
        schema = schema[current_step]
        errors = set(validate_jsonschema_form(schema, json_data))
        # Откладываем ответ в случае ошибки для полной валидации данных
        self.expected_fields = get_step_fields(current_step)
        for field in self.expected_fields:
            self.form_values[field] = json_data.get(field)
        try:
            if ('eula_accepted' in self.expected_fields and
                    (check_missing_fields or self.form_values['eula_accepted'] is not None)):
                self.check_eula_accepted()
            if 'question_answer' in self.expected_fields:
                self.translate_question_id()
            if 'phone_numbers' in self.expected_fields:
                self.validate_phone_numbers()
            # Проверяем контактный email-адрес в последнюю очередь, для осложнения перебора в ручке commit
            if 'contact_email' in self.expected_fields:
                self.validate_contact_email(check_personal_data=check_personal_data)
        except ValidationFailedError as exception:
            errors.update(exception.errors)
        except BaseBundleError as exception:
            errors.add(exception.error)
        if errors:
            raise ValidationFailedError(list(errors))


class RestoreSemiAutoMultiStepGetStateView(RestoreSemiAutoMultiStepViewBase, GetAccountForRestoreMixin):
    require_track = True

    def get_step_2_recovery_tools_form_data(self):
        questions = get_question_from_account_as_list(self.account)
        self.track.questions = questions
        self.response_values['questions'] = prepare_questions_for_response(questions)

    def process_request(self):
        self.read_track()
        if not self.is_process_consistent():
            return
        if self.any_counter_exceeded():
            return

        current_step = self.track.semi_auto_step
        if current_step != STEP_FINISHED and current_step not in STEP_TO_SCHEMA_MAPPING:
            raise InvalidTrackStateError()

        processing_finished = self.get_and_validate_account(
            self.track.user_entered_login,
            check_domain_support=True,
            skip_validation=self.track.is_unconditional_pass,
            emails=True,  # email'ы получаем для записи в трек; используются при валидации contact_email
        )
        if processing_finished:
            return

        with self.track_transaction.commit_on_error():
            self.fill_track_with_account_data_for_semi_auto_form()
            if self.track.semi_auto_step == STEP_2_RECOVERY_TOOLS:
                self.get_step_2_recovery_tools_form_data()

        self.statbox.log(action='got_state')
        self.response_values['track_state'] = current_step
        self.response_values['user_entered_login'] = self.track.user_entered_login
        self.response_values['request_source'] = self.request_source


class RestoreSemiAutoMultiStepValidateView(RestoreSemiAutoMultiStepViewBase):
    def process_request(self):
        self.read_track()
        if not self.is_process_consistent():
            return
        if self.any_counter_exceeded():
            return
        self.process_step_form(check_missing_fields=False)


class RestoreSemiAutoMultiStepCommitView(
    RestoreSemiAutoMultiStepViewBase,
    GetAccountForRestoreMixin,
    CalculateFactorsMixin,
):
    require_track = True

    def send_message(self, factors, check_passed):
        context = get_message_context_from_factors(
            self.track.user_entered_login,
            factors,
            check_passed,
            self.restore_id,
            request_source=self.request_source,
        )
        request_source = self.request_source
        if self.track.is_for_learning:
            # Анкеты на обучение идут в отдельную очередь, несмотря на исходный request_source
            request_source = RESTORE_MESSAGE_REQUEST_SOURCE_FOR_LEARNING
        if (self.has_recent_positive_decision(factors) and
                self.request_source != settings.RESTORE_REQUEST_SOURCE_FOR_CHANGE_HINT):
            # Анкеты пользователей, у которых недавно было автоматическое или саппортское
            # положительное решение, помечаются особым образом
            request_source = RESTORE_MESSAGE_REQUEST_SOURCE_FOR_POSITIVE_DECISION_RETRY
        message = create_semi_auto_message(
            context,
            self.form_values['photo_file'],
            self.host,
            request_source,
            self.track.device_application,
        )
        return send_message(message)

    def get_factor_names_for_step(self):
        return STEP_TO_FACTORS_MAPPING.get(self.track.semi_auto_step)

    def fill_factors_with_extra_info(self, factors):
        # Сохранить данные, по которым не вычисляются факторы, но которые используются при восстановлении
        factors['version'] = self.track.version
        factors['is_for_learning'] = self.track.is_for_learning or False
        request_info = factors.setdefault('request_info', {})
        request_info['request_source'] = self.request_source
        request_info['last_step'] = self.track.semi_auto_step
        request_info['language'] = self.form_values['language']
        request_info['is_unconditional_pass'] = self.track.is_unconditional_pass or False
        for field in ('contact_email', 'contact_reason', 'user_enabled', 'real_reason'):
            if field in self.expected_fields:
                request_info[field] = self.form_values[field]

    def process_request(self):
        self.read_track()
        if not self.is_process_consistent():
            return
        if self.any_counter_exceeded():
            return
        self.process_step_form(check_personal_data=True)
        self.statbox.log(action='submitted')

        processing_finished = self.get_and_validate_account(
            self.track.user_entered_login,
            check_domain_support=True,
            skip_validation=self.track.is_unconditional_pass,
            emails=True,
            need_phones=True,
        )
        if processing_finished:
            return

        with self.track_transaction.commit_on_error():
            self.fill_track_with_account_data_for_semi_auto_form()

            # Берем данные анкеты из трека, дополняем их данными, пришедшими на текущем шаге
            factors = self.track.factors or {}
            events_info_cache = self.track.events_info_cache or {}
            factors.update(self.calculate_factors(*self.get_factor_names_for_step(),
                                                  calculated_factors=factors,
                                                  events_info_cache=events_info_cache))
            self.fill_factors_with_extra_info(factors)

            self.handle_step(factors, events_info_cache)

    def handle_step(self, factors, events_info_cache):
        if self.track.semi_auto_step == STEP_5_SERVICES_DATA:
            self.try_make_final_decision(factors)
        elif self.track.semi_auto_step == STEP_6_FINAL_INFO:
            # Сюда попадаем только со статусом pending
            self.handle_pending_user(factors)
        else:
            # "Обычные" шаги
            self.track.factors = factors
            self.track.events_info_cache = events_info_cache
            self.statbox.log(action='compared')
            self.raise_if_retry_required(factors)
            self.track.semi_auto_step = get_next_step(self.track.semi_auto_step)

    def try_make_final_decision(self, factors):
        """
        При наличии автоматического решения да/нет завершаем работу анкеты, иначе
        потребуется дополнительный шаг.
        """
        should_use_tensornet = self.should_use_tensornet
        should_use_basic_formula = self.should_use_basic_formula
        restore_status = RESTORE_STATUS_PENDING
        decision_source = DECISION_SOURCE_UNCONDITIONAL

        # Всегда вычисляем результаты проверки по базовой формуле и по обученной формуле для статистики
        basic_formula_results = self.get_basic_formula_results(factors)
        tensornet_status = tensornet_estimate = None
        if self.can_use_tensornet:
            tensornet_status, tensornet_estimate = self.get_tensornet_estimate(factors)

        if should_use_tensornet and tensornet_status:
            decision_source = DECISION_SOURCE_TENSORNET
            can_make_negative_decision = settings.RESTORE_SEMI_AUTO_NEGATIVE_DECISION_ENABLED
            can_make_positive_decision = (
                # Принятие положительного решения включается в настройках, недоступно для 2ФА-пользователя,
                # запрещается в случае недавнего принятия положительного решения
                settings.RESTORE_SEMI_AUTO_POSITIVE_DECISION_ENABLED and
                not self.account.totp_secret.is_set and
                not self.has_recent_positive_decision(factors)
            )
            lower_bound, upper_bound = settings.RESTORE_SEMI_AUTO_DECISION_THRESHOLDS[self.request_source]
            if can_make_negative_decision and tensornet_estimate <= lower_bound:
                # Принимаем отрицательное решение
                restore_status = RESTORE_STATUS_REJECTED
            elif can_make_positive_decision and tensornet_estimate >= upper_bound:
                # Принимаем положительное решение автоматически
                restore_status = RESTORE_STATUS_PASSED

        elif should_use_basic_formula or should_use_tensornet and not tensornet_status:
            decision_source = DECISION_SOURCE_BASIC_FORMULA
            if not basic_formula_results['any_check_passed']:
                restore_status = RESTORE_STATUS_REJECTED

        factors.update(
            restore_status=restore_status,
            tensornet_estimate=tensornet_estimate,
            tensornet_status=tensornet_status,
            decision_source=decision_source,
        )
        self.track.factors = factors

        is_final_step = restore_status != RESTORE_STATUS_PENDING
        self.statbox.log(
            action='compared',
            restore_status=restore_status,
            tensornet_estimate=tensornet_estimate,
            tensornet_status=tensornet_status,
            decision_source=decision_source,
            restore_id=self.restore_id if is_final_step else None,
            **basic_formula_results
        )

        self.increment_ip_counter()

        if is_final_step:
            self.write_events_to_event_log(restore_status)
            self.write_factors_to_restore_log(factors)

            self.track.semi_auto_step = STEP_FINISHED

            if restore_status == RESTORE_STATUS_REJECTED:
                raise CompareNotMatchedError()
            elif restore_status == RESTORE_STATUS_PASSED:
                # Можем сразу восстановить доступ, требуется перейти на страницу ввода новых данных
                self.setup_track_after_semi_auto_form_passed()
                self.state = RedirectToRestorationPassed()
        else:
            self.track.semi_auto_step = get_next_step(self.track.semi_auto_step)

    def handle_pending_user(self, factors):
        """
        Для pending-пользователя сохраняем анкету в историю и отправляем письмо в саппорт.
        """
        decision_source = factors['decision_source']
        send_message_to_primary_maillist = True

        if decision_source == DECISION_SOURCE_BASIC_FORMULA:
            # Поддерживаем отправку в отдельную рассылку для анкет, не полностью прошедших
            # проверку базовой формулой.
            basic_formula_results = self.get_basic_formula_results(factors)
            send_message_to_primary_maillist = basic_formula_results['whole_check_passed']

        self.track.factors = factors
        self.statbox.log(
            action='compared',
            restore_id=self.restore_id,
        )
        if self.send_message(factors, send_message_to_primary_maillist) is None:
            raise SendmailFailedError()
        self.increment_uid_counter()

        self.write_events_to_event_log(RESTORE_STATUS_PENDING)
        self.write_factors_to_restore_log(factors)

        self.track.semi_auto_step = get_next_step(self.track.semi_auto_step)

        if not send_message_to_primary_maillist:
            raise CompareNotMatchedError()

    @property
    def should_use_tensornet(self):
        """
        Применяем обученную формулу, если анкета пришла в потоке на обучение, а также
        использование tensornet разрешено.
        """
        return self.track.is_for_learning and self.can_use_tensornet

    @property
    def can_use_tensornet(self):
        """
        Использование tensornet разрешено, если не существует файла-индикатора.
        """
        return not os.path.exists(settings.RESTORE_SEMI_AUTO_DISABLE_AUTO_DECISION_FILE)

    @property
    def should_use_basic_formula(self):
        """
        Применяем базовую (ручную) формулу, если анкета не используется для безусловного пропуска в саппорт,
        и обученная формула не применяется.
        """
        return not self.track.is_unconditional_pass and not self.should_use_tensornet

    @cached_property
    def tensornet(self):
        return TensorNet(
            bin=settings.RESTORE_SEMI_AUTO_TENSORNET_BINARY,
            model=settings.RESTORE_SEMI_AUTO_TENSORNET_MODEL,
            fdfile=settings.RESTORE_SEMI_AUTO_TENSORNET_FDFILE,
        )

    def get_basic_formula_results(self, factors):
        names_birthday_check_passed = get_names_birthday_check_status(factors)
        whole_check_passed = names_birthday_check_passed and get_user_env_check_status(factors)
        any_check_passed = names_birthday_check_passed or get_user_env_check_status(factors)
        return {
            'any_check_passed': any_check_passed,
            'whole_check_passed': whole_check_passed,
        }

    def get_tensornet_estimate(self, factors):
        features = get_features_from_factors(factors)
        if not features:
            return False, None
        try:
            estimate = self.tensornet.predict(features)
            return True, estimate
        except TensorNetError as e:
            log.error('TensorNet prediction failed: %s.', e)
            return False, None

    def has_recent_positive_decision(self, factors):
        return factors['restore_attempts']['factor']['has_recent_positive_decision']
