# -*- coding: utf-8 -*-

from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.validators.jsonschema_adapter import JsonSchema

from ..factors import (
    ALLOWED_INPUT_SERVICE_NAMES,
    MAX_PASSWORD_FIELDS,
)


MAX_MULTIPLE_CHOICE_FIELDS = 5


MAX_SOCIAL_ACCOUNTS_COUNT = 10


MAX_DEFAULT_STRING_LENGTH = 1024


DELIVERY_ADDRESSES_JSON_SCHEMA = {
    '$schema': 'http://json-schema.org/schema#',
    'type': 'array',
    'items': {
        'type': 'object',
        'properties': {
            'country': {'type': 'string', 'minLength': 1},
            'city': {'type': 'string', 'minLength': 1},
            'street': {'type': 'string', 'minLength': 1},
            'building': {'type': 'string', 'minLength': 1},
            'suite': {'type': 'string'},
            'metro': {'type': 'string'},
            'zip': {'type': 'string'},
            'flat': {'type': 'string'},
            'firstname': {'type': 'string'},
            'cargolift': {'type': 'string'},
            'floor': {'type': 'string'},
            'comment': {'type': 'string'},
            'fathersname': {'type': 'string'},
            'phone': {'type': 'string'},
            'lastname': {'type': 'string'},
            'intercom': {'type': 'string'},
            'phone_extra': {'type': 'string'},
            'email': {'type': 'string'},
            'entrance': {'type': 'string'},
        },
        'required': ['country', 'city', 'street', 'building'],
        'additionalProperties': False,
    },
    'minItems': 1,
    'maxItems': MAX_MULTIPLE_CHOICE_FIELDS,
    'uniqueItems': True,
}


MULTI_STEP_BASE_SCHEMA = {
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
}


class PhotoFileValidator(validators.SingleFileUploadValidator):
    # Костыль по аналогии с валидатором Password
    def _to_python(self, value, state=None):
        self.max_size = settings.RESTORE_SEMI_AUTO_MAX_PHOTO_FILE_BYTES
        return super(PhotoFileValidator, self)._to_python(value, state)


class RestoreSemiAutoMultiStepForm(validators.Schema):
    language = validators.DisplayLanguage(if_missing=settings.DEFAULT_LANGUAGE)  # язык, на котором была отображена анкета

    # Данные формы приходят в JSON'е, проверяем, что пришел JSON-объект
    json_data = validators.JSONValidator(schema=MULTI_STEP_BASE_SCHEMA, not_empty=True)

    chained_validators = (
        validators.FileUploadSchemaValidator(
            photo_file=PhotoFileValidator(if_missing=None),
        ),
    )


def _make_generic_list_schema(items, max_items=MAX_MULTIPLE_CHOICE_FIELDS):
    return {
        'type': 'array',
        'items': items,
        'maxItems': max_items,
        'uniqueItems': True,
    }


# Поддерживаем два варианта схем - с требуемыми полями и без них.
# Вариант без требуемых полей используется в ручке валидации.
STEP_1_PERSONAL_DATA_SCHEMA = JsonSchema({
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'real_reason': validators.OneOf(
            ['restore', 'accountdel', 'killkvko'],
            not_empty=True,
            strip=True,
            # TODO: убрать после того, как выкатится фронт
            if_missing=None,
        ),
        'firstnames': {
            'type': 'array',
            'items': validators.FirstName(not_empty=True, strict=True),
            'minItems': 1,
            'maxItems': 2,
        },
        'lastnames': {
            'type': 'array',
            'items': validators.LastName(not_empty=True, strict=True),
            'minItems': 1,
            'maxItems': 2,
        },
        'birthday': validators.Birthday(not_empty=True, year_offset=200),
        'contact_email': validators.SimpleEmailValidator(),
        'passwords': {
            'type': 'array',
            'items': validators.String(not_empty=True, strict=True, strip=True),
            'minItems': 1,
            'maxItems': MAX_PASSWORD_FIELDS,
            'uniqueItems': True,
        },
        'password_auth_date': validators.LooseDateValidator(
            not_empty=True,
            earliest_year=settings.YANDEX_FOUNDATION_YEAR,
            max_days_in_future=2,
        ),
        'eula_accepted': {'type': 'boolean'},
    },
})


STEP_1_PERSONAL_DATA_STRICT_SCHEMA = JsonSchema(
    dict(
        STEP_1_PERSONAL_DATA_SCHEMA.schema,
        required=[
            'firstnames',
            'lastnames',
            'birthday',
            'contact_email',
            'passwords',
            'password_auth_date',
            'eula_accepted',
        ],
    ),
)


STEP_2_RECOVERY_TOOLS_SCHEMA = STEP_2_RECOVERY_TOOLS_STRICT_SCHEMA = JsonSchema({
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'phone_numbers': _make_generic_list_schema(validators.String(strict=True, not_empty=True, strip=True)),
        'emails': _make_generic_list_schema(validators.SimpleEmailValidator()),
        'question_answer': {
            'type': 'object',
            'properties': {
                'question_id': {'type': 'integer'},
                'question': validators.String(
                    strict=True,
                    not_empty=True,
                    strip=True,
                    if_missing=validators.NoDefault,
                    max=MAX_DEFAULT_STRING_LENGTH,
                ),
                'answer': validators.String(
                    strict=True,
                    not_empty=True,
                    strip=True,
                    max=MAX_DEFAULT_STRING_LENGTH,
                ),
            },
            'required': ['question', 'question_id', 'answer'],
            'additionalProperties': False,
        },
    },
})


STEP_3_REGISTRATION_DATA_SCHEMA = JsonSchema({
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'registration_date': validators.LooseDateValidator(
            not_empty=True,
            earliest_year=settings.YANDEX_FOUNDATION_YEAR,
            max_days_in_future=2,
        ),
        'registration_country': validators.String(not_empty=True, strict=True, strip=True),
        'registration_country_id': {'type': 'integer'},
        'registration_city': validators.String(strict=True, strip=True),
        'registration_city_id': {'type': 'integer'},
    },
})


STEP_3_REGISTRATION_DATA_STRICT_SCHEMA = JsonSchema(
    dict(
        STEP_3_REGISTRATION_DATA_SCHEMA.schema,
        required=[
            'registration_date',
            'registration_country',
        ],
    ),
)


STEP_4_USED_SERVICES_SCHEMA = STEP_4_USED_SERVICES_STRICT_SCHEMA = JsonSchema({
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'social_accounts': _make_generic_list_schema(
            validators.HexString(not_empty=True, strip=True),
            max_items=MAX_SOCIAL_ACCOUNTS_COUNT,
        ),
        'services': {
            'type': 'array',
            'items': {
                'enum': ALLOWED_INPUT_SERVICE_NAMES,
            },
            'uniqueItems': True,
        },
    },
})


STEP_5_SERVICES_DATA_SCHEMA = STEP_5_SERVICES_DATA_STRICT_SCHEMA = JsonSchema({
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'delivery_addresses': dict(DELIVERY_ADDRESSES_JSON_SCHEMA, minItems=0),
        'email_folders': _make_generic_list_schema(validators.String(not_empty=True, strip=True, strict=True)),
        'outbound_emails': _make_generic_list_schema(validators.SimpleEmailValidator()),
        'email_collectors': _make_generic_list_schema(validators.SimpleEmailValidator()),
        'email_whitelist': _make_generic_list_schema(validators.SimpleEmailValidator()),
        'email_blacklist': _make_generic_list_schema(validators.SimpleEmailValidator()),
    },
})


STEP_6_FINAL_INFO_SCHEMA = JsonSchema({
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'contact_reason': validators.String(strip=True, strict=True),
        'user_enabled': {'type': 'boolean'},
    },
})


STEP_6_FINAL_INFO_STRICT_SCHEMA = JsonSchema(
    dict(
        STEP_6_FINAL_INFO_SCHEMA.schema,
        required=[
            'user_enabled',
        ],
    ),
)
