# -*- coding: utf-8 -*-
from passport.backend.api.common.processes import PROCESS_WEB_REGISTRATION
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.core.builders.bbro_api import (
    BaseBBroApiError,
    BBroApiInvalidRequestError,
    get_bbro_api,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from .helpers import process_bt_counters


class CreateSecuritySession(BaseBundleView):
    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )
    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]
    required_grants = ['security.create_session']
    require_track = True
    track_type = 'register'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='security',
            action='create_session',
            ip=self.client_ip,
            track_id=self.track_id,
            consumer=self.consumer,
            user_agent=self.user_agent,
        )

    @cached_property
    def bbro_api(self):
        return get_bbro_api()

    def process_request(self):
        yandexuid = self.cookies.get('yandexuid')
        if yandexuid is None:
            return
        self.read_track()
        try:
            bt_counters = process_bt_counters(
                self.bbro_api.get_bt_counters_by_yandexuid(yandexuid),
            )
            if len(bt_counters) > 0:
                self.statbox.log(
                    yandexuid=yandexuid,
                    **bt_counters
                )
        except BBroApiInvalidRequestError:
            self.statbox.log(
                yandexuid=yandexuid,
                error='bbro_api.invalid_yandexuid',
            )
        except BaseBBroApiError:
            self.statbox.log(
                yandexuid=yandexuid,
                error='bbro_api.unavailable',
            )
