# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import (
    AUTHORIZATION_SESSION_POLICY_PERMANENT,
    bb_response_to_session,
    build_auth_cookies,
    build_auth_cookies_and_session,
    build_cookie_mda2_beacon,
    build_cookie_yandex_login,
    get_session_policy_by_ttl,
    is_session_created,
    is_session_valid,
    is_user_session_valid,
    log_new_session,
    SessGuardContainer,
    try_update_cookie_lah,
)
from passport.backend.api.common.processes import (
    PROCESS_ACCOUNT_DELETE_V2,
    PROCESS_AUTH_BY_SMS,
    PROCESS_RESTORE,
    PROCESS_WEB_REGISTRATION,
)
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.exceptions import (
    AccountGlobalLogoutError,
    InvalidIpError,
    SessionExpiredError,
)
from passport.backend.api.views.bundle.auth.exceptions import (
    AuthAlreadyPassedError,
    AuthInvalidIpError,
    AuthNotAllowedError,
    AuthSessionExpiredError,
    AuthSessionOverflowError,
    CodeVerifierNotMatchedError,
    ContainerInvalidHostError,
    ContainerInvalidSignatureError,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountGlobalLogoutError as BundleAccountGlobalLogoutError,
    AccountNotFoundError,
    SessguardInvalidError,
    SessionidInvalidError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleEditSessionMixin,
    CookieCheckStatus,
)
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core import authtypes
from passport.backend.core.conf import settings
from passport.backend.core.encrypted_container import EncryptedContainerInvalid
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.types.account.account import ACCOUNT_TYPE_NORMAL
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.pkce import is_pkce_valid
from six.moves.urllib.parse import urlparse

from .forms import (
    SessionCreateForm,
    SessionUpdateForm,
    UseSessGuardContainerForm,
)


log = logging.getLogger('passport.api.view.bundle.session')


class SessionCreateView(BaseBundleView, BundleAccountGetterMixin):
    """
    Ручка создания сессии.
    """

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
    )

    required_grants = ['session.create']

    require_track = True

    basic_form = SessionCreateForm

    allowed_processes = [
        PROCESS_RESTORE,
        PROCESS_AUTH_BY_SMS,
        PROCESS_ACCOUNT_DELETE_V2,
        PROCESS_WEB_REGISTRATION,
    ]

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        with self.track_transaction.rollback_on_error():

            self.response_values['track_id'] = self.track_id
            self.fill_response_with_track_fields('retpath')

            if not self.track.allow_authorization:
                raise AuthNotAllowedError()

            if is_session_created(self.track):
                raise AuthAlreadyPassedError('Session already created for track: %s' % self.track.track_id)

            if self.track.oauth_code_challenge:
                if not is_pkce_valid(
                    challenge=self.track.oauth_code_challenge,
                    challenge_method=self.track.oauth_code_challenge_method,
                    verifier=self.form_values['code_verifier'],
                ):
                    raise CodeVerifierNotMatchedError()

            extend_session = False
            multi_session_users = None
            sessionid = self.request.env.cookies.get('Session_id')
            ssl_sessionid = self.request.env.cookies.get('sessionid2')
            sessguard = self.request.env.cookies.get('sessguard')
            if sessionid:
                self.track.old_session = sessionid
                self.track.old_ssl_session = ssl_sessionid
                self.track.old_sessguard = sessguard
                to_statbox({
                    'mode': 'check_cookies',
                    'host': self.request.env.host,
                    'consumer': self.consumer,
                    'have_sessguard': sessguard is not None,
                    'sessionid': mask_sessionid(sessionid),
                })
                bb_response = self.blackbox.sessionid(
                    allow_scholar=True,
                    dbfields=[],
                    host=self.host,
                    ip=self.client_ip,
                    multisession=True,
                    request_id=self.request.env.request_id,
                    sessguard=sessguard,
                    sessionid=sessionid,
                    sslsessionid=ssl_sessionid,
                )
                if is_session_valid(bb_response['cookie_status']):
                    multi_session_users = bb_response.get('users', {})
                    if not bb_response.get('allow_more_users', True) and (
                        int(self.track.uid) not in multi_session_users
                    ):
                        raise AuthSessionOverflowError()
                    extend_session = True
                    self.track.old_session_ttl = bb_response['ttl']

            policy = self.track.authorization_session_policy or AUTHORIZATION_SESSION_POLICY_PERMANENT
            auth_params = dict(
                # После 2ФА восстановления нужно выключить флаг в куке YP
                account_type=ACCOUNT_TYPE_NORMAL,
                authorization_session_policy=policy,
                display_name=DisplayName(name=self.track.login),
                env=self.request.env,
                extend_session=extend_session,
                is_2fa_enabled_yp=False if self.track.is_otp_restore_passed else None,
                multi_session_users=multi_session_users,
                need_extra_sessguard=True,
                otp_magic_passed=self.track.is_otp_magic_passed,
                track=self.track,
                x_token_magic_passed=self.track.is_x_token_magic_passed,
            )
            try:
                self.get_account_by_uid(self.track.uid, enabled_required=False)
                self.assert_allowed_to_get_cookie()
                auth_params.update(
                    account_type=self.account.type,
                    is_yandexoid=self.account.is_yandexoid,
                    is_betatester=self.account.is_betatester,
                    display_name=self.account.person.display_name,
                    logout_datetime=self.account.web_sessions_logout_datetime,
                    is_2fa_enabled=self.account.totp_secret.is_set,
                    is_child=self.account.is_child,
                )

                if self.track.auth_source == authtypes.AUTH_SOURCE_XTOKEN:
                    # При авторизации по X-токену, шаг обмена на трек может быть вызван сервером, профиль пользователя
                    # может быть неполным. Поэтому сохраняем профиль при выписывании сессии.
                    process_env_profile(self.account, track=self.track)
            except AccountNotFoundError:
                # После регистрации данные могут не успеть дойти до реплики
                pass

            try:
                cookies, session, service_guard_container = build_auth_cookies_and_session(**auth_params)
                if service_guard_container:
                    self.response_values.update({
                        'service_guard_container': service_guard_container.pack(),
                    })

                self.response_values.update({
                    'cookies': cookies,
                    'default_uid': int(self.track.uid),
                    'sensitive_fields': ['cookies'],
                })
            except AccountGlobalLogoutError:
                raise BundleAccountGlobalLogoutError()
            except SessionExpiredError:
                raise AuthSessionExpiredError()
            except InvalidIpError:
                raise AuthInvalidIpError()


class SessionUpdateView(BaseBundleView, BundleAccountGetterMixin, BundleEditSessionMixin):

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
    )

    required_grants = ['session.update']

    require_track = False

    basic_form = SessionUpdateForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            action='updated',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            referer=self.referer,
        )

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()
        if self.form_values['retpath']:
            retpath = self.form_values['retpath']
            self.response_values['retpath'] = retpath
            parse = urlparse(retpath)
            extra_sessguard_host = parse.hostname
        else:
            extra_sessguard_host = None
            retpath = None

        # Если от пользователя пришло две или более куки Session_id (https://st.yandex-team.ru/PASSP-15163),
        # значит, куки надо принудительно перевыставить
        session_id_count = len(self.request.env.cookies_all.get('Session_id', []))
        force = session_id_count > 1 or self.form_values['force_prolong']

        # TODO: если куки две, то стоит не всегда брать первую,
        # а эвристически искать самую свежую (RFC 6265 4.2.2)
        session_info = self.check_session_cookie(
            allow_scholar=True,
            extra_sessguard_host=extra_sessguard_host,
            force_prolong=force,
            prolong_cookies=True,
        )
        cookies = []
        is_session_cookie_valid = session_info.cookie_status in (
            CookieCheckStatus.Valid,
            CookieCheckStatus.NeedReset,
        )
        if not is_session_cookie_valid:
            if force:
                # сносим невалидные куки, чтоб не ломать ими другие сервисы
                empty_session = self.build_empty_session(with_ssl_session=True, with_sessguard=True)
                cookies = build_auth_cookies(self.request.env, empty_session)
            else:
                if session_info.cookie_status == CookieCheckStatus.WrongGuard:
                    raise SessguardInvalidError()
                raise SessionidInvalidError()
        else:
            bb_response = session_info.response
            session, service_guard_container = bb_response_to_session(
                bb_response,
                self.request.env,
                extra_sessguard_host=extra_sessguard_host,
                retpath=retpath,
            )
            if service_guard_container:
                self.response_values.update({
                    'service_guard_container': service_guard_container.pack(),
                })
            if session:
                log_new_session(session, service_guard_container)
                cookies = build_auth_cookies(self.request.env, session)
                if session.get('session') and is_user_session_valid(bb_response):
                    # Если сессия дефолтного пользователя валидна и мы её подновили - надо выставить куку yandex_login
                    # (срок её жизни обычно совпадает со сроком жизни session_id) и подновить куку lah
                    self.parse_account(
                        bb_response,
                        enabled_required=False,
                    )
                    cookies.append(
                        build_cookie_yandex_login(
                            self.request.env,
                            human_readable_login=self.account.human_readable_login,
                        ),
                    )

                    mda2_beacon_cookie_params = dict()
                    if session['session']['value']:
                        mda2_beacon_cookie_params['expires'] = session['session']['expires']
                    cookies.append(
                        build_cookie_mda2_beacon(
                            self.request.env,
                            **mda2_beacon_cookie_params
                        ),
                    )

                    cookie_lah = try_update_cookie_lah(
                        env=self.request.env,
                        uid=self.account.uid,
                        auth_method=None,
                        authorization_session_policy=get_session_policy_by_ttl(session_info.ttl),
                    )
                    if cookie_lah:
                        cookies.append(cookie_lah)

        if cookies:
            self.response_values.update(cookies=cookies)
            self.statbox.log(
                uid=self.account.uid if self.account else None,
                auth_id=session_info.authid,
                session_cookie_count=session_id_count,
                is_session_valid=is_session_cookie_valid,
            )


class UseSessGuardContainerView(BaseBundleView):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
    )

    required_grants = ['session.use_sessguard_container']
    basic_form = UseSessGuardContainerForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            consumer=self.consumer,
            action='use_sessguard_container',
            event='use_sessguard_container',
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
        )

    @staticmethod
    def _extract_meaningful_level3_host(hostname):
        """
        Сократить домен до третьего уровня, считая tld за один, например:
        some.thing.on.mail.yandex.com.tr -> mail.yandex.com.tr
        other.thing.on.mail.yandex.ru -> mail.yandex.ru
        """
        hostname = hostname.lower()
        for tld in settings.PASSPORT_TLDS:
            if hostname.endswith('.%s' % tld):
                without_tld = hostname[:-(len(tld) + 1)]
                parts_23 = without_tld.split('.')[-2:]
                result_parts = parts_23 + [tld]
                break
        else:
            result_parts = hostname.split('.')[-3:]

        return '.'.join(result_parts)

    def process_request(self):
        self.process_basic_form()

        try:
            container = SessGuardContainer.unpack(self.form_values['container'])
        except EncryptedContainerInvalid as e:
            log.debug('Container: check sign error %s' % e)
            raise ContainerInvalidSignatureError(e)

        parse = urlparse(container['retpath'])
        retpath_host_l3 = self._extract_meaningful_level3_host(parse.hostname)
        request_host_l3 = self._extract_meaningful_level3_host(self.headers[HEADER_CLIENT_HOST.name])

        if retpath_host_l3 != request_host_l3:
            message = 'Request l3 host doesn\'t match retpath l3 host: request %s retpath %s' % (
                request_host_l3,
                retpath_host_l3,
            )
            log.debug(message)
            raise ContainerInvalidHostError(message)

        log.debug(
            'Unpacked sessguard container retpath=%s, cookies=%s' %
            (container['retpath'], container['cookies']),
        )

        self.statbox.log(
            status='ok',
            retpath=container['retpath'],
        )

        self.response_values.update(
            retpath=container['retpath'],
            cookies=container['cookies'],
        )


__all__ = (
    'SessionCreateView',
    'SessionUpdateView',
    'UseSessGuardContainerView',
)
