# -*- coding: utf-8 -*-
import time

from passport.backend.api.common.authorization import (
    check_track_for_logout,
    is_user_default_in_multisession,
    is_user_in_multisession,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutPasswordError,
    InvalidTrackStateError,
    PasswordRequiredError,
    SessguardInvalidError,
    SessionidInvalidError,
    UidNotInSessionError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleEditSessionMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.api.views.bundle.states import (
    RedirectToSocialCompletion,
    RedirectToSocialCompletionWithLogin,
)
from passport.backend.core.builders.blackbox.constants import BLACKBOX_EDITSESSION_OP_ADD
from passport.backend.core.builders.social_api import get_social_api
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from .exceptions import (
    NotUsersProfileError,
    SingleAuthMethodError,
)


BASIC_GRANT = 'social_profiles.base'


class BaseSocialBundleView(BaseBundleView,
                           BundleAccountGetterMixin,
                           BundleAccountPropertiesMixin,
                           BundleAccountResponseRendererMixin):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
    )
    require_track = False
    required_grants = [BASIC_GRANT]
    track_type = 'authorize'

    @property
    def social(self):
        return get_social_api()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='social_profiles',
            ip=self.client_ip,
            useragent=self.user_agent,
            uid=self.account.uid,
            track_id=self.track_id,
        )


class BaseSocialTrackRequiredBundleView(BaseSocialBundleView, BundleVerifyPasswordMixin, BundleEditSessionMixin):
    require_track = True

    def __init__(self, *args, **kwargs):
        super(BaseSocialTrackRequiredBundleView, self).__init__(*args, **kwargs)
        self.account_extra_kwargs = dict()

    def update_cookies(self):
        if not getattr(self, 'session_info', None):
            return

        if not is_user_in_multisession(self.session_info, self.track_uid):
            return

        # Меняем дефолтный аккаунт в сессии или подновляем время ввода пароля в
        # сессии, только если аккаунт уже находится в сессии.

        with self.track_transaction.rollback_on_error():
            if self.track.is_password_passed:
                cookies, default_uid = self.edit_session(
                    operation=BLACKBOX_EDITSESSION_OP_ADD,
                    uid=self.track_uid,
                    new_default_uid=self.track_uid,
                    cookie_session_info=self.session_info,
                    statbox_params=dict(action='cookie_edit'),
                    password_check_time=time.time(),
                )
                self.response_values.update(cookies=cookies)
                if default_uid:
                    self.response_values.update(default_uid=default_uid)

            elif not is_user_default_in_multisession(self.session_info, self.track_uid):
                self.change_default(
                    self.session_info,
                    new_default_uid=self.track_uid,
                )

    def get_and_check_account(self):
        """
        Получает аккаунт по
          - сессии
          - треку, который можно обменять на сессию
        """
        session_cookie_body, _, _ = self.get_effective_session_and_host()

        if session_cookie_body:
            try:
                self.get_account_from_session(
                    multisession_uid=self.track_uid,
                    check_disabled_on_deletion=True,
                    **self.account_extra_kwargs
                )
            except UidNotInSessionError as e:
                no_auth_exception = InvalidTrackStateError(
                    "track.uid (%s) is not in the current user's session uids (%s)" %
                    (self.track_uid, e.known_uids),
                )
            except (SessguardInvalidError, SessionidInvalidError) as e:
                no_auth_exception = e
        else:
            no_auth_exception = SessionidInvalidError()

        if not self.account:
            if self.track_uid and self.track.allow_authorization:
                self.get_account_from_track(check_disabled_on_deletion=True)
                self.check_track_allows_to_get_session()

        if not self.account:
            raise no_auth_exception

    def check_track_allows_to_get_session(self):
        if not (self.track_uid and self.track.allow_authorization):
            raise InvalidTrackStateError()
        check_track_for_logout(self.track, self.account.web_sessions_logout_datetime)

    def check_user_profile_id(self, profile_id):
        profiles = self.social.get_profiles_by_uid(self.track_uid)
        social_profile_ids = [profile['profile_id'] for profile in profiles]
        if profile_id not in social_profile_ids:
            raise NotUsersProfileError()

    def get_completion_state(self):
        # Соц аккаунт. account.is_social означает, что логин не был установлен пользователем.
        if self.account.is_social:
            return RedirectToSocialCompletionWithLogin()
        else:
            return RedirectToSocialCompletion()

    def check_current_password(self):
        if not self.account.have_password:
            raise AccountWithoutPasswordError()

        if not self.form_values['current_password']:
            if self.is_password_verification_required():
                raise PasswordRequiredError('Password required')
        else:
            with self.track_transaction.commit_on_error():
                self.verify_password(
                    self.account.uid,
                    self.form_values['current_password'],
                    ignore_statbox_log=True,
                    useragent=self.user_agent,
                    referer=self.referer,
                    yandexuid=self.cookies.get('yandexuid'),
                    dbfields=[],  # Не выбирать из ЧЯ доп информацию о пользователе
                    attributes=[],  # Аналогично выше
                )
                self.track.is_password_passed = True

    def process_request(self):
        self.check_header(HEADER_CLIENT_HOST)
        self.read_track()
        self.response_values['track_id'] = self.track_id

        if self.basic_form is not None:
            self.process_basic_form()

        self.get_and_check_account()
        self.process()

    def process(self):
        raise NotImplementedError()  # pragma: no cover


class AuthProfileSocialBundleMixin(object):
    def check_enable_auth_profile_allowed(self, profiles):
        return self.check_action_with_auth_profile_allowed(profiles)

    def check_disable_auth_profile_allowed(self, profiles):
        return self.check_action_with_auth_profile_allowed(profiles, is_delete=True)

    def check_action_with_auth_profile_allowed(
        self,
        profiles,
        is_delete=False,
    ):
        """
        Проверяет, что на аккаунте можно включить/отключить соц. вход.

        Входные параметры

        is_delete — признак, что нужно проверить допустимость отключения соц.
            входа, иначе проверяется включение.

        profiles — соц. профили аккаунта

        self.account

        Выходные параметры

            None, когда можно продолжить включать/отключать соц. вход

            Иначе state, когда работать дальше нельзя и нужно завершить выполнение
        """

        # Важно: не может быть такой ситуации, когда при действии с профилем
        # с возможностью авторизации у нас нет альтернативного метода входа.
        # У пользователя всегда должен оставаться неудаляемый (чтобы даже гонки
        # не могли оставить пользователя без метода входа) метод входа
        # (пароль, номер телефона, E-mail).

        if self.account.have_password:
            # При изменении возможности авторизации через соц. профиль нужно
            # проверить пароль.
            self.check_current_password()

        elif (
            self.account.is_social or
            # Социальщик с логином, но без пароля
            self.account.is_normal and self.account.social_alias
        ):
            # Запрещаем управлять соц. входом социальщикам, потому что у них
            # нет альтернативных способов входа, и разрешён вход только через
            # один соц. профиль.
            allow_auth_profiles = [p for p in profiles if p['allow_auth']]
            if is_delete and len(allow_auth_profiles) == 1:
                # Возможно пользователь хочет удалить аккаунт, поэтому
                # не отправляем на дорегистрацию, а предлагаем выбор
                # пользователю:
                #
                # 1. Дорегистрироваться и стать полноценным пользователем
                # Яндекса и удалить соц. профиль.
                #
                # 2. Удалить аккаунт целиком
                raise SingleAuthMethodError()
            else:
                # У аккаунта нет альтернативного способа входа, поэтому его
                # необходимо отправить на дорегистрацию.
                return self.get_completion_state()

        elif not self.account.is_lite:
            # Запрещаем управление соц. входом в аккаунтах непредусмотренных
            # типов.
            return self.get_completion_state()
