# -*- coding: utf-8 -*-

from passport.backend.api.common.processes import PROCESS_WEB_REGISTRATION
from passport.backend.api.templatetags import l
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountStrongPasswordPolicyError,
    ActionNotRequiredError,
    TaskNotFoundError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
)
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.builders.social_api import (
    get_social_api,
    ProfileNotFoundError,
    SubscriptionAlreadyExistsError,
    SubscriptionNotFoundError,
    TaskNotFoundError as SocialApiTaskNotFoundError,
)
from passport.backend.core.builders.social_api.profiles import convert_task_profile
from passport.backend.core.builders.social_broker import get_best_matching_size_avatar
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.email import Email
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.email.email import is_yandex_email
from passport.backend.core.types.login.login import (
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.social_business_info import BusinessInfo
from passport.backend.core.utils.decorators import cached_property

from .base import (
    AuthProfileSocialBundleMixin,
    BaseSocialBundleView,
    BaseSocialTrackRequiredBundleView,
)
from .copier import (
    AvatarCopier,
    BirthdayCopier,
    EmailCopier,
    FirstnameCopier,
    GenderCopier,
    LastnameCopier,
)
from .exceptions import (
    NotUsersProfileError,
    ProviderNotAllowAuthError,
)
from .forms import (
    DeleteWithPasswordForm,
    SetAuthWithPasswordForm,
    SocialFillPersonalDataSubmitForm,
    SocialThumbnailForm,
    SubscriptionForm,
)


def _get_thumbnail_info_by_task_id(task_id, target_size):
    social_api = get_social_api()
    profile = social_api.get_task_data(task_id)['profile']

    thumbnail_info = dict((field, profile.get(field)) for field in ('username', 'firstname', 'lastname', 'provider'))
    thumbnail_info['avatar'] = get_best_matching_size_avatar(profile.get('avatar'), target_size)

    return thumbnail_info


class SocialThumbnailView(BaseBundleView):
    """
    Получение данных соц. аккаунта для отображения иконки
    """
    basic_form = SocialThumbnailForm

    required_grants = ['social.get_thumbnail']

    def process_request(self):
        self.process_basic_form()

        self.response_values['thumbnail'] = _get_thumbnail_info_by_task_id(
            self.form_values['task_id'],
            (self.form_values['avatar_size_x'], self.form_values['avatar_size_y']),
        )


class SocialGetListStartTrack(BaseSocialBundleView, BundleAccountResponseRendererMixin):
    def initialize_track(self):
        self.create_track(self.track_type)

        with self.track_transaction.rollback_on_error():
            self.track.uid = self.account.uid

        self.response_values['track_id'] = self.track_id

    def process_request(self):
        self.get_account_from_session(
            check_disabled_on_deletion=True,
        )

        profiles = self.social.get_profiles_by_uid(
            self.account.uid,
            subscriptions=True,
            tokens=True,
            person=True,
        )
        social_profiles = self.social_profiles_response(profiles)
        # Тут хотелось бы формировать ответ сразу в self.response_values['account']['profiles']
        # Оставлено для обратной совместимости
        self.response_values.update({
            'profiles': social_profiles,
        })
        self.initialize_track()

        self.fill_response_with_account()
        self.statbox.log(action='initialized')


class SocialGetList(BaseSocialTrackRequiredBundleView):
    def process(self):
        social_profiles = self.social_profiles_response(
            self.social.get_profiles_by_uid(
                self.account.uid,
                subscriptions=True,
                tokens=True,
                person=True,
            ),
        )
        self.response_values.update({
            'profiles': social_profiles,
        })
        self.statbox.log(action='listed')


class SocialDelete(
    AuthProfileSocialBundleMixin,
    BaseSocialTrackRequiredBundleView,
):
    basic_form = DeleteWithPasswordForm

    def process(self):
        profile_id = self.form_values['profile_id']

        profiles = self.social.get_profiles_by_uid(self.account.uid)

        social_profile_ids = [profile['profile_id'] for profile in profiles]
        if profile_id not in social_profile_ids:
            raise NotUsersProfileError()

        allow_auth_profile_ids = {p['profile_id'] for p in profiles if p['allow_auth']}
        if profile_id in allow_auth_profile_ids:
            self.state = self.check_disable_auth_profile_allowed(profiles)
            if self.state:
                return

        try:
            self.social.delete_profile(profile_id)

            self.update_cookies()
            self.statbox.log(action='deleted')
        except ProfileNotFoundError:
            self.update_cookies()
            raise ActionNotRequiredError()


class SocialSetAuth(
    MailMixin,
    UserMetaDataMixin,
    AuthProfileSocialBundleMixin,
    BaseSocialTrackRequiredBundleView,
    BundlePushMixin,
    KolmogorMixin,
):
    allowed_processes = [PROCESS_WEB_REGISTRATION]
    basic_form = SetAuthWithPasswordForm
    require_process = False

    def __init__(self):
        super(SocialSetAuth, self).__init__()
        self.profile = None
        self.account_extra_kwargs = dict(emails=True)

    def process(self):
        if self.form_values['set_auth']:
            if self.account.is_strong_password_required:
                # Привязывая к 67 сиду необходимо соблюдать политику сложного
                # пароля, которая нарушается при возможности соц. авторизации.
                raise AccountStrongPasswordPolicyError()

        profiles = self.social.get_profiles_by_uid(self.account.uid)

        self.load_profile(profiles)

        if self.form_values.get('profile_id'):
            self.check_social_profile(profiles)
        elif self.track.social_task_id and self.form_values['set_auth']:
            self.check_social_task_data()
        else:
            raise ValidationFailedError(['profile_id.empty'])

        if self.form_values['set_auth']:
            self.state = self.check_enable_auth_profile_allowed(profiles)
        else:
            self.state = self.check_disable_auth_profile_allowed(profiles)
        if self.state:
            return

        if self.form_values.get('profile_id'):
            self.social.set_authentificate_profile(self.form_values.get('profile_id'), self.form_values['set_auth'])
        elif self.track.social_task_id and self.form_values['set_auth']:
            self.social.bind_task_profile(self.track.social_task_id, self.account.uid)
        else:
            raise NotImplementedError()  # pragma: no cover

        self.update_cookies()

        if self.form_values['set_auth']:
            self.statbox.log(action='set_auth')
            self.send_account_modification_push(
                event_name='social_allow',
                context={'track_id': self.track_id},
            )
            self.send_social_allow_mail()
        else:
            self.statbox.log(action='unset_auth')

        self.send_account_modification_push(
            event_name='login_method_change',
            context={'track_id': self.track_id},
        )
        self.send_account_modification_mail(
            event_name='login_method_change',
        )

    def send_social_allow_mail(self):
        language = get_preferred_language(self.account)

        provider_name = self.profile.get('provider')
        if isinstance(provider_name, dict):
            provider_name = provider_name.get('name')

        self.send_account_modification_mail(
            event_name='social_allow',
            context=dict(
                PROVIDER=l(dict(), 'SOCIAL', provider_name, language),
            ),
        )

    def load_profile(self, profiles):
        if self.form_values.get('profile_id'):
            for profile in profiles:
                if profile['profile_id'] == self.form_values.get('profile_id'):
                    self.profile = profile
                    break
        elif self.track.social_task_id:
            self.profile = self.track.social_task_data.get('profile', dict())

    def check_social_profile(self, profiles):
        if not (self.profile and self.profile['profile_id'] == self.form_values.get('profile_id')):
            raise NotUsersProfileError()

        if (
            self.form_values['set_auth'] and
            self.profile.get('provider_code') not in settings.AUTH_ALLOWED_PROVIDERS
        ):
            raise ProviderNotAllowAuthError()

    def check_social_task_data(self):
        if (
            self.form_values['set_auth'] and
            self.profile.get('provider', dict()).get('code') not in settings.AUTH_ALLOWED_PROVIDERS
        ):
            raise ProviderNotAllowAuthError()

        # Разрешаем включить соц. вход на аккаунт через данный соц. профиль,
        # только когда у аккаунта и соц. профиля есть что-то общее, например
        # e-mail.

        social_profile_email = self.profile.get('email')
        if not social_profile_email:
            raise NotUsersProfileError()

        if (
            is_yandex_email(social_profile_email) and
            self.account.portal_alias and
            self.account.portal_alias.alias == normalize_login(raw_login_from_email(social_profile_email))
        ):
            pass
        elif (
            self.account.lite_alias and
            self.account.lite_alias.alias == Email.normalize_address(social_profile_email)
        ):
            pass
        else:
            raise NotUsersProfileError()


class SocialSetSubscription(BaseSocialTrackRequiredBundleView):
    basic_form = SubscriptionForm

    def process(self):
        profile_id = self.form_values['profile_id']
        self.check_user_profile_id(profile_id)

        try:
            self.social.create_subscription(profile_id, self.form_values['sid'])
            self.update_cookies()
            self.statbox.log(action='set_subscription')
        except SubscriptionAlreadyExistsError:
            self.update_cookies()
            raise ActionNotRequiredError()


class SocialDeleteSubscription(BaseSocialTrackRequiredBundleView):
    basic_form = SubscriptionForm

    def process(self):
        profile_id = self.form_values['profile_id']
        self.check_user_profile_id(profile_id)

        try:
            self.social.delete_subscription(profile_id, self.form_values['sid'])
            self.update_cookies()
            self.statbox.log(action='deleted_subscription')
        except SubscriptionNotFoundError:
            self.update_cookies()
            raise ActionNotRequiredError()


class _BaseSocialFillPersonalData(BundleAccountGetterMixin, BaseBundleView):
    required_headers = [
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    ]

    required_grants = ['social.fill_personal_data']

    require_track = True

    def _get_account(self):
        self.get_pinned_account_from_session_or_oauth_token(
            emails=True,
            email_attributes='all',
        )
        self.statbox.bind_context(uid=self.account.uid)

    def _get_profile(self, task_id):
        try:
            profile = self.social_api.get_task_data(task_id)
        except SocialApiTaskNotFoundError:
            raise TaskNotFoundError()
        profile = convert_task_profile(profile['profile'])

        self.statbox.bind_context(provider=profile['provider']['code'])

        return profile

    def _build_copiers(self, profile):
        return [
            FirstnameCopier(self.account, profile),
            LastnameCopier(self.account, profile),
            GenderCopier(self.account, profile),
            BirthdayCopier(self.account, profile),
            EmailCopier(self.account, profile),
            AvatarCopier(self.account, profile),
        ]

    def _check_profile_bound_to_account(self, profile):
        profiles = self.social_api.get_profiles(
            userid=profile['userid'],
            provider=profile['provider']['code'],
            uid=self.account.uid,
            business_info=BusinessInfo.from_dict(profile.get('business')),
        )
        if not profiles:
            raise NotUsersProfileError()

    def _get_data_to_fill_account_gaps_from_profile(self, copiers):
        personal_data = dict()
        for copier in copiers:
            if copier.data_is_missing_on_account():
                personal_data.update(copier.get_dict())

        profile = copiers[0].profile

        return {
            'source': {'provider_code': profile['provider']['code']},
            'values': personal_data,
        }

    def _fill_account_gaps_from_profile(self, copiers):
        for copier in copiers:
            if copier.data_is_missing_on_account():
                copier.fill_account()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='fill_account_from_social_profile',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
            step=self.STEP_NAME,
            track_id=self.track_id,
        )


class SocialFillPersonalDataSubmit(_BaseSocialFillPersonalData):
    """
    Показывает персональные данные из социальной сети, которыми можно заполнить
    пробелы в персональных данных яндексового аккаунта.
    """
    basic_form = SocialFillPersonalDataSubmitForm

    STEP_NAME = 'submit'

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        self._get_account()

        profile = self._get_profile(self.form_values['task_id'])
        self._check_profile_bound_to_account(profile)

        self.statbox.log(action='submitted')

        with self.track_transaction.rollback_on_error():
            self.track.social_task_id = self.form_values['task_id']

        copiers = self._build_copiers(profile)

        self.response_values.update({'data': [self._get_data_to_fill_account_gaps_from_profile(copiers)]})

        self.statbox.log(action='committed')


class SocialFillPersonalDataCommit(_BaseSocialFillPersonalData):
    """
    Заполняет пробелы в персональных данных на аккаунте данными из аккаунта в
    социальной сети.
    """
    STEP_NAME = 'commit'

    def process_request(self):
        self.read_track()

        self._get_account()

        profile = self._get_profile(self.track.social_task_id)
        self._check_profile_bound_to_account(profile)

        self.statbox.log(action='submitted')

        copiers = self._build_copiers(profile)

        with UPDATE(
            self.account,
            self.request.env,
            dict(action='fill_account_from_social_profile', consumer=self.consumer),
        ):
            self._fill_account_gaps_from_profile(copiers)

        self.statbox.log(action='committed')
