# -*- coding: utf-8 -*-

from datetime import datetime

from passport.backend.api.common.social_api import set_avatar_from_social_profile
from passport.backend.api.views.bundle.exceptions import YapicUnavailableError
from passport.backend.core.builders.avatars_mds_api import AvatarsMdsApiTemporaryError
from passport.backend.core.conf import settings
from passport.backend.core.models.email import Email
from passport.backend.core.types.gender import Gender


class Copier(object):
    def __init__(self, account, profile):
        self.account = account
        self.profile = profile

    def _fill_account_from_profile(self, field_name):
        value = self.profile.get(field_name)
        if value is not None:
            setattr(self.account.person, field_name, value)

    def _get_dict_from_profile(self, field_name):
        retval = dict()
        if field_name in self.profile:
            retval[field_name] = self.profile[field_name]
        return retval

    def data_is_missing_on_account(self):
        raise NotImplementedError()  # pragma: no cover

    def get_dict(self):
        raise NotImplementedError()  # pragma: no cover

    def fill_account(self):
        raise NotImplementedError()  # pragma: no cover


class FirstnameCopier(Copier):
    def fill_account(self):
        self._fill_account_from_profile('firstname')

    def get_dict(self):
        return self._get_dict_from_profile('firstname')

    def data_is_missing_on_account(self):
        return not self.account.person.firstname


class LastnameCopier(Copier):
    def fill_account(self):
        self._fill_account_from_profile('lastname')

    def get_dict(self):
        return self._get_dict_from_profile('lastname')

    def data_is_missing_on_account(self):
        return not self.account.person.lastname


class GenderCopier(Copier):
    def fill_account(self):
        self._fill_account_from_profile('gender')

    def get_dict(self):
        gender = Gender.to_char(self.profile.get('gender'))
        retval = dict()
        if gender:
            retval['gender'] = gender
        return retval

    def data_is_missing_on_account(self):
        return not self.account.person.gender


class BirthdayCopier(Copier):
    def fill_account(self):
        self._fill_account_from_profile('birthday')

    def get_dict(self):
        retval = dict()
        if 'birthday' in self.profile:
            birthday = self.profile['birthday']
            if birthday is not None:
                birthday = str(birthday)
            retval['birthday'] = birthday
        return retval

    def data_is_missing_on_account(self):
        return not self.account.person.birthday


class EmailCopier(Copier):
    def _profile_email_in_account(self):
        email_address = self.profile.get('email')
        return email_address and email_address in self.account.emails

    def fill_account(self):
        if self._profile_email_in_account():
            raise NotImplementedError()  # pragma: no cover

        email_address = self.profile.get('email')
        if email_address is not None and len(self.account.emails) < settings.MAX_EMAILS_PER_PROFILE:
            timestamp = datetime.now()
            email = Email(
                address=self.profile.get('email'),
                created_at=timestamp,
                confirmed_at=timestamp,
                bound_at=timestamp,
                is_native=False,
                is_silent=False,
                is_unsafe=True,
            )
            self.account.emails.add(email)

    def get_dict(self):
        return self._get_dict_from_profile('email')

    def data_is_missing_on_account(self):
        if self._profile_email_in_account():
            return False
        for email in self.account.emails.confirmed_external:
            if not email.is_silent:
                return False
        return True


class AvatarCopier(Copier):
    def fill_account(self):
        try:
            set_avatar_from_social_profile(self.account.uid, self.profile)
        except AvatarsMdsApiTemporaryError as e:
            # Из соображений совместимости выкинем ту ошибку, которую умеет фронт
            raise YapicUnavailableError(e)

    def get_dict(self):
        return self._get_dict_from_profile('avatar')

    def data_is_missing_on_account(self):
        return self.account.person.is_avatar_empty
