# -*- coding: utf-8 -*-


class BaseState(object):
    def update_response(self, response):
        response['state'] = self.state


class PasswordChangeForbidden(BaseState):
    """Запрещена смена пароля"""
    state = 'password_change_forbidden'


class RfcTotp(BaseState):
    """Требуется ввод одноразового пароля (RFC)"""
    state = 'rfc_totp'


class EmailCode(BaseState):
    """Требуется ввод кода из письма"""
    state = 'email_code'


class RedirectToPasswordChange(BaseState):
    """Этого пользователя надо отправить на страницу смены пароля"""
    state = 'change_password'

    def __init__(self, reason, validation_method):
        self.reason = reason
        self.validation_method = validation_method
        super(RedirectToPasswordChange, self).__init__()

    def update_response(self, response):
        super(RedirectToPasswordChange, self).update_response(response)
        response['change_password_reason'] = self.reason
        response['validation_method'] = self.validation_method


class RedirectToCompletion(BaseState):
    """Этого пользователя надо отправить на установку пароля и заполнение анкеты"""
    state = 'complete_autoregistered'


class RedirectToPDDCompletion(BaseState):
    """Этого пользователя надо отправить на завершение регистрации ПДД"""
    state = 'complete_pdd'

    def __init__(self, hint=None):
        self.hint = hint
        super(RedirectToPDDCompletion, self).__init__()

    def update_response(self, response):
        super(RedirectToPDDCompletion, self).update_response(response)
        if self.hint and self.hint.question and response.get('account') is not None:
            response['account']['hint'] = {
                'question': {
                    'id': self.hint.question.id,
                    'text': self.hint.question.text,
                },
                'answer': self.hint.answer,
            }
            response['sensitive_fields'] = response.get('sensitive_fields') or []
            response['sensitive_fields'].append('account.hint.answer')


class DomainNotServed(BaseState):
    """Домен не обслуживается, требуется обращение к администратору домена"""
    state = 'domain_not_served'


class RedirectToPDDCompletionWithPassword(RedirectToPDDCompletion):
    """Этого пользователя надо отправить на завершение регистрации ПДД с установкой пароля"""
    state = 'complete_pdd_with_password'


class RedirectToFederalCompletion(BaseState):
    """Этого пользователя надо отправить на завершение регистрации федеративного аккаунта"""
    state = 'complete_federal'


class RedirectToSocialCompletion(BaseState):
    """Этого социального пользователя надо отправить на завершение регистрации"""
    state = 'complete_social'


class RedirectToSocialCompletionWithLogin(BaseState):
    """
    Этого социального пользователя надо отправить на завершение регистрации,
    польователю нужно выставить логин
    """
    state = 'complete_social_with_login'


class RedirectToLiteCompletion(BaseState):
    """Этого lite пользователя надо отправить на завершение регистрации"""
    state = 'complete_lite'


class RedirectToNeophonishCompletion(BaseState):
    """Этого неофониша надо отправить на завершение регистрации"""
    state = 'complete_neophonish'


class AuthChallenge(BaseState):
    """Этому пользователю надо показать авторизационный челленж"""
    state = 'auth_challenge'


class ForwardAuth(BaseState):
    """Этому пользователю нужно выписать куки, закончив процесс проброса авторизации"""
    state = 'forward_auth'


class RedirectToForcedLiteCompletion(BaseState):
    """Этого lite пользователя надо отправить принудительно на завершение регистрации"""
    state = 'force_complete_lite'


class UpgradeCookies(BaseState):
    """
    Пользователя не нужно дорегистрировать, однако у него сейчас лайтовая сессия.
    """
    state = 'upgrade_cookies'


class PasswordRequired(BaseState):
    """Пользователю необходимо ввести пароль"""
    state = 'password_required'


class RateLimitExceeded(BaseState):
    """Слишком большое число попыток, нужно показать заглушку"""
    state = 'rate_limit_exceeded'


class OtpAuthNotReady(BaseState):
    """Отп-магия: данные еще не готовы для авторизации"""
    state = 'otp_auth_not_ready'


class OtpAuthFinished(BaseState):
    """Отп-магия: процесс авторизации завершился"""
    state = 'otp_auth_finished'


class ProcessRestartRequired(BaseState):
    """Требуется перезапуск длительного процесса из-за обновления"""
    state = 'process_restart_required'


class RedirectToRestorationPassed(BaseState):
    """Способ восстановления пройден, перейти на страницу ввода новых данных"""
    state = 'restoration_passed'


class Show2FAPromo(BaseState):
    """Требуется показать пользователю промо включения 2ФА"""
    state = 'show_2fa_promo'


class PhoneAliasProhibited(BaseState):
    """Продолжить процедуру с телефонным алиасом невозможно, пользователю рекомендуется вспомнить основной логин"""
    state = 'phone_alias_prohibited'


__all__ = (
    'AuthChallenge',
    'BaseState',
    'DomainNotServed',
    'PasswordRequired',
    'PasswordChangeForbidden',
    'RedirectToPasswordChange',
    'RedirectToCompletion',
    'RedirectToPDDCompletion',
    'RedirectToPDDCompletionWithPassword',
    'RedirectToFederalCompletion',
    'RedirectToForcedLiteCompletion',
    'RedirectToLiteCompletion',
    'RedirectToNeophonishCompletion',
    'RedirectToSocialCompletion',
    'RedirectToSocialCompletionWithLogin',
    'RedirectToRestorationPassed',
    'Show2FAPromo',
    'UpgradeCookies',
    'RateLimitExceeded',
    'OtpAuthNotReady',
    'OtpAuthFinished',
    'ProcessRestartRequired',
    'PhoneAliasProhibited',
    'ForwardAuth',
    'RfcTotp',
    'EmailCode',
)
