# -*- coding: utf-8 -*-
import time

from passport.backend.api.common.common import increment_track_counter
from passport.backend.api.common.processes import PROCESS_WEB_REGISTRATION
from passport.backend.api.common.suggest import (
    get_login_suggestions,
    get_mobile_language_suggest,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.suggest.forms import SuggestLoginForm
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.types.login.login import extract_clean_login_from_email
from passport.backend.core.utils.decorators import cached_property


SUGGEST_LOGIN_GRANT = 'login.suggest'
SUGGEST_LANGUAGE_GRANT = 'language.suggest'


class SuggestLoginView(BaseBundleView):
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    ]

    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]

    required_grants = [SUGGEST_LOGIN_GRANT]

    basic_form = SuggestLoginForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.process_basic_form()

        original_login = login = self.form_values['login']

        self.read_or_create_track('authorize')

        # На случай, если в поле логина ввели емейл
        if login:
            login = extract_clean_login_from_email(login)

        suggested_logins = get_login_suggestions(
            original_login=original_login,
            login=login,
            firstname=self.form_values['first_name'],
            lastname=self.form_values['last_name'],
            language=self.form_values['language'],
        )

        # В трек записываем предложенные логины и время вызовов
        with self.track_transaction.rollback_on_error() as track:
            if suggested_logins:
                track.suggested_logins.append(*suggested_logins)
            track.suggest_login_count.incr()
            if track.suggest_login_first_call is None:
                track.suggest_login_first_call = time.time()
            track.suggest_login_last_call = time.time()

        # Запись нужна для Фродо
        if not suggested_logins and login and len(login) >= 6:
            self.statbox.log(
                action='suggest_login_error',
                login=login,
                firstname=self.form_values['first_name'],
                lastname=self.form_values['last_name'],
                track_id=self.track_id,
            )

        self.response_values['suggested_logins'] = suggested_logins


class SuggestMobileLanguage(BaseBundleView):
    required_grants = [SUGGEST_LANGUAGE_GRANT]

    def process_request(self):
        try:
            language = get_mobile_language_suggest()
        finally:
            increment_track_counter(self.track_id, lambda track: track.suggest_language_count)

        self.response_values['language'] = language
