# -*- coding: utf-8 -*-

from datetime import (
    datetime,
    timedelta,
)
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    InternalPermanentError,
    InternalTemporaryError,
)
from passport.backend.api.views.bundle.mixins.account import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.support_code import exceptions
from passport.backend.api.views.bundle.support_code.forms import (
    CheckSupportCodeForm,
    CreateSupportCodeForm,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.support_code import SupportCode
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.ydb.processors.support_code import (
    find_support_code,
    insert_support_code,
    SupportCodeExistsError,
)
from passport.backend.utils.common import generate_random_code
from passport.backend.utils.time import datetime_to_integer_unixtime


log = logging.getLogger(__name__)


OAUTH_SCOPE = ['passport:support_code']


class BaseSupportCodeView(BaseBundleView):
    def check_api_enabled(self):
        if not settings.SUPPORT_CODE_API_ENABLED:
            log.debug('Support code API disabled')
            raise InternalPermanentError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            mode='support_code',
            user_agent=self.user_agent,
        )


class CreateSupportCodeView(BaseSupportCodeView, BundleAccountGetterMixin):
    basic_form = CreateSupportCodeForm
    required_grants = ['support_code.create']

    def process_request(self):
        self.check_api_enabled()
        self.process_basic_form()
        self.get_account_from_session_or_oauth_token(
            multisession_uid=self.form_values['uid'],
            required_scope=OAUTH_SCOPE,
        )
        self.statbox.bind_context(
            uid=self.account.uid,
        )
        expires_at = datetime.now() + timedelta(seconds=settings.SUPPORT_CODE_TTL)
        support_code = SupportCode(
            expires_at=expires_at,
            uid=self.account.uid,
            value=None,
        )
        log.debug('Generating support code for {}, expires at {}'.format(
            self.account.uid,
            expires_at,
        ))
        for collisions in range(settings.SUPPORT_CODE_GENERATE_RETRIES):
            support_code.value = generate_random_code(settings.SUPPORT_CODE_LENGTH)
            try:
                insert_support_code(support_code)
                break
            except SupportCodeExistsError:
                log.debug('Failed to insert %s: already exists' % support_code)
                continue
        else:
            self.statbox.log(
                action='create_support_code',
                status='too_much_collisions',
            )
            raise InternalTemporaryError()
        self.statbox.log(
            action='create_support_code',
            status='success',
            collisions=collisions,
        )
        self.response_values.update(
            support_code=support_code.value,
            expires_at=datetime_to_integer_unixtime(support_code.expires_at),
        )


class CheckSupportCodeView(BaseSupportCodeView, BundleTvmUserTicketMixin):
    basic_form = CheckSupportCodeForm
    required_grants = ['support_code.check']
    required_user_ticket_scopes = list()

    def process_request(self):
        self.check_api_enabled()
        self.process_basic_form()
        user_ticket = self.check_user_ticket()
        support_uid = self.get_uid_or_default_from_user_ticket(user_ticket, self.form_values['uid'])
        support_code = find_support_code(self.form_values['support_code'])
        log.debug('Checking support code, support uid is {}, uid by code is {}'.format(
            support_uid,
            support_code.uid if support_code else 'not found',
        ))
        if not support_code or support_code.expires_at <= datetime.now():
            self.statbox.log(
                action='check_support_code',
                status='fail',
                support_uid=support_uid,
                uid=support_code.uid if support_code else None,
            )
            raise exceptions.InvalidSupportCodeError()
        self.statbox.log(
            action='check_support_code',
            status='success',
            support_uid=support_uid,
            uid=support_code.uid,
        )
        self.response_values.update(uid=support_code.uid)
