# -*- coding: utf-8 -*-

from collections import OrderedDict
from datetime import (
    datetime,
    timedelta,
)
import json
import time

from passport.backend.api.common.format_response import json_response
from passport.backend.api.common.logbroker import get_api_logbroker_writer
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountNotFoundError,
    ActionImpossibleError,
    ActionNotRequiredError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
)
from passport.backend.api.views.bundle.takeout.exceptions import ExtractInProgressError
from passport.backend.api.views.bundle.takeout.forms import (
    DeleteExtractResultForm,
    FinishExtractForm,
    GetExtractResultForm,
    RequestExtractForm,
    StartExtractForm,
    UserInfoForm,
)
from passport.backend.core.builders.historydb_api import HistoryDBApi
from passport.backend.core.builders.social_api import get_social_api
from passport.backend.core.conf import settings
from passport.backend.core.geobase import is_valid_country_code
from passport.backend.core.historydb.account_history import AccountHistory
from passport.backend.core.historydb.account_history.account_history import HISTORYDB_API_QEURY_LIMIT
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.s3 import get_s3_client
from passport.backend.core.types.email.email import unicode_email
from passport.backend.core.types.gender import Gender
from passport.backend.core.utils.decorators import cached_property
from passport.backend.takeout.logbroker_client.extract_task_message import (
    ExtractTaskMessage,
    get_extract_id,
    get_task_id,
)
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import datetime_to_integer_unixtime


GET_USER_INFO_GRANT = 'takeout.user_info'
REQUEST_EXTRACT_GRANT = 'takeout.request_extract'
START_EXTRACT_GRANT = 'takeout.start_extract'
FINISH_EXTRACT_GRANT = 'takeout.finish_extract'
GET_EXTRACT_RESULT_GRANT = 'takeout.get_extract_result'
DELETE_EXTRACT_RESULT_GRANT = 'takeout.delete_extract_result'


HALF_A_YEAR_IN_HOURS = 24 * 365 // 2
DEFAULT_COUNTRY = 'RU'
DEFAULT_LANGUAGE = 'ru'


def to_json(data):
    return json.dumps(data, indent=2, ensure_ascii=False)


def make_ordered_dict(keys_and_values):
    data = []
    for key, value in keys_and_values:
        if not value:
            continue
        if not isinstance(value, (OrderedDict, dict, list, tuple, int, float, bool)):
            value = smart_text(value)
        data.append((key, value))

    return OrderedDict(data)


class UserInfoView(BaseBundleView, BundleAccountGetterMixin):
    required_grants = [GET_USER_INFO_GRANT]
    basic_form = UserInfoForm

    def error_response(self, errors, code=200, **extra_response_values):
        """Хотя ручка и бандловая, ошибки она отдаёт в небандловом формате: таковы требования сервиса Takeout"""
        return json_response(
            _code=code,
            status='error',
            error='; '.join(errors),
            **extra_response_values
        )

    def ok_response(self, **response_values):
        """Не кладём в ответ status=ok по умолчанию: могу быть и другие статусы"""
        return json_response(
            _code=200,
            **response_values
        )

    def account_info(self):
        language = (self.account.person.language or DEFAULT_LANGUAGE).lower()
        country = (self.account.person.country or DEFAULT_COUNTRY).lower()
        if not is_valid_country_code(country):
            country = None

        phones = [
            make_ordered_dict([
                ('number', phone.number.international),
                ('confirmed', phone.confirmed),
                ('secured', phone.secured),
            ])
            for phone in self.account.phones.all().values()
        ]

        emails = [
            make_ordered_dict([
                ('address', unicode_email(email.address)),
                ('confirmed', email.confirmed_at),
            ])
            for email in self.account.emails.external
        ]

        raw_social_profiles = get_social_api().get_profiles_by_uid(
            self.account.uid,
            expand_provider=True,
        )
        social_profiles = [
            make_ordered_dict([
                ('provider', raw_profile['provider']['name']),
                ('userid', raw_profile['userid']),
                ('username', raw_profile['username']),
                ('allow_auth', raw_profile.get('allow_auth') or None),
            ])
            for raw_profile in raw_social_profiles
        ]

        plus_info = make_ordered_dict([
            ('active', self.account.plus.has_plus),
            ('trial_used', bool(self.account.plus.trial_used_ts)),
            ('subscription_expires', self.account.plus.subscription_expire_ts),
            ('next_charge', self.account.plus.next_charge_ts),
            ('ott_subscription', self.account.plus.ott_subscription),
            ('family_role', self.account.plus.family_role),
        ])

        info = [
            ('login', self.account.display_login),
            ('registration_datetime', self.account.registration_datetime),
            ('firstname', self.account.person.firstname),
            ('lastname', self.account.person.lastname),
            ('gender', Gender.to_char(self.account.person.gender)),  # to_char вернёт None для неизвестного пола
            ('birthday', self.account.person.birthday),
            ('language', language),
            ('country', country),
            ('city', self.account.person.city),
            ('timezone', self.account.person.timezone),
            ('display_name', self.account.person.display_name.name),
            ('avatar_url', settings.GET_AVATAR_URL % (self.account.person.default_avatar, 'normal')),
            ('phones', phones),
            ('emails', emails),
            ('social_profiles', social_profiles),
            ('plus', plus_info),
        ]
        return make_ordered_dict(info)

    def account_change_history(self):
        ah = AccountHistory(uid=self.account.uid)
        events = []
        while True:
            events += ah.list(
                hours_limit=HALF_A_YEAR_IN_HOURS,
                limit=HISTORYDB_API_QEURY_LIMIT,
                to_ts=ah.next_page_timestamp,
            )
            if not ah.next_page_timestamp:
                break

        return [
            event._asdict() for event in sorted(events, key=lambda ev: ev.timestamp, reverse=True)
        ]

    def auth_history(self):
        historydb_api = HistoryDBApi(
            timeout=settings.HISTORYDB_API_TIMEOUT_FOR_TAKEOUT,
            retries=settings.HISTORYDB_API_RETRIES_FOR_TAKEOUT,
        )
        auths, _ = historydb_api.auths_aggregated(
            uid=self.account.uid,
            hours_limit=HALF_A_YEAR_IN_HOURS,
            password_auths=True,
        )
        return sorted(auths, key=lambda auth: auth.get('timestamps'), reverse=True)

    def process_request(self):
        self.process_basic_form()
        try:
            self.get_account_by_uid(
                uid=self.form_values['uid'],
                enabled_required=False,
                need_phones=True,
                emails=True,
            )
            self.response_values.update(
                status='ok',
                data={
                    'account_info.json': to_json(self.account_info()),
                    'account_change_history.json': to_json(self.account_change_history()),
                    'auth_history.json': to_json(self.auth_history()),
                },
            )
        except AccountNotFoundError:
            self.response_values.update(
                status='no_data',
            )


class EmailMixin(object):
    email_template = None
    email_subject_key = None

    def send_email_notifications(self):
        language = get_preferred_language(account=self.account)
        template_name, info, context = self.get_email_notification_data(language)
        send_mail_for_account(
            template_name,
            info, context,
            self.account,
            send_to_native=True,
            # На внешние ящики шлём, только если нет нативного
            send_to_external=not (self.account.emails.default and self.account.emails.default.is_native),
        )

    def get_email_notification_data(self, language):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = self.email_template
        info = MailInfo(
            subject=translations[self.email_subject_key],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'greeting_key': None,
                'feedback_key': 'takeout.feedback',
                'feedback_url_key': 'takeout.feedback_url',
                'uid': self.account.uid,
            },
        )
        return template_name, info, context


class RequestExtractView(BaseBundleView, BundleAccountGetterMixin, EmailMixin):
    required_grants = [REQUEST_EXTRACT_GRANT]
    basic_form = RequestExtractForm

    email_template = 'mail/takeout_extract_requested.html'
    email_subject_key = 'takeout.extract_requested.subject'

    def process_request(self):
        self.process_basic_form()
        self.get_account_from_session(
            multisession_uid=self.form_values['uid'],
            enabled_required=True,
            emails=True,
        )
        if self.account.takeout.extract_in_progress_since:
            raise ActionNotRequiredError()

        # Отправить сообщение лучше раньше, чем записать в БД признак, что оно
        # было отправлено
        self.send_extract_task_message()

        events = {
            'action': 'request_takeout_extract',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.takeout.extract_in_progress_since = datetime.now()
            self.account.takeout.archive_s3_key = None
            self.account.takeout.archive_password = None
            self.account.takeout.archive_created_at = None
            self.account.takeout.fail_extract_at = datetime.now() + timedelta(seconds=settings.TIME_FOR_ACCEPTING_EXTRACT)

        self.send_email_notifications()

    def send_extract_task_message(self):
        logbroker = get_api_logbroker_writer('takeout_tasks')
        extract_task_message = ExtractTaskMessage(
            extract_id=get_extract_id(),
            task_id=get_task_id(),
            uid=self.account.uid,
            unixtime=int(time.time()),
            source='passport_api',
        )
        logbroker.send(extract_task_message.takeout_request)


class StartExtractView(BaseBundleView, BundleAccountGetterMixin):
    required_grants = [START_EXTRACT_GRANT]
    basic_form = StartExtractForm

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
        )
        if not self.account.takeout.extract_in_progress_since:
            raise ActionNotRequiredError()

        events = {
            'action': 'start_takeout_extract',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.takeout.fail_extract_at = datetime.now() + timedelta(seconds=settings.TIME_FOR_COMPLETING_EXTRACT)


class FinishExtractView(BaseBundleView, BundleAccountGetterMixin, EmailMixin):
    required_grants = [FINISH_EXTRACT_GRANT]
    basic_form = FinishExtractForm

    email_template = 'mail/takeout_extract_complete.html'
    email_subject_key = 'takeout.extract_complete.subject'

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
            emails=True,
        )
        if not self.account.takeout.extract_in_progress_since:
            raise ActionNotRequiredError()

        events = {
            'action': 'finish_takeout_extract',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.takeout.extract_in_progress_since = None
            self.account.takeout.archive_s3_key = self.form_values['archive_s3_key']
            self.account.takeout.archive_password = self.form_values['archive_password']
            self.account.takeout.archive_created_at = datetime.now()
            self.account.takeout.fail_extract_at = None

        self.send_email_notifications()


class BaseGetExtractResultView(BaseBundleView, BundleAccountGetterMixin, BundleAccountPropertiesMixin):
    required_grants = [GET_EXTRACT_RESULT_GRANT]
    basic_form = GetExtractResultForm

    @property
    def is_archive_available(self):
        if not self.account.takeout.archive_s3_key:
            return False

        archive_valid_until = self.account.takeout.archive_created_at + timedelta(seconds=settings.TAKEOUT_ARCHIVE_TTL)
        return archive_valid_until >= datetime.now()

    def process_request(self):
        self.process_basic_form()
        self.get_account_from_session(
            multisession_uid=self.form_values['uid'],
            enabled_required=True,
        )

        self.fill_response_with_extract_results()

    def fill_response_with_extract_results(self):
        raise NotImplementedError()  # pragma: no cover


class GetExtractStatusView(BaseGetExtractResultView):
    def fill_response_with_extract_results(self):
        self.response_values.update(
            is_extract_in_progress=bool(self.account.takeout.extract_in_progress_since),
        )

        if self.is_archive_available:
            archive_valid_until = self.account.takeout.archive_created_at + timedelta(seconds=settings.TAKEOUT_ARCHIVE_TTL)
            self.response_values.update(
                archive=dict(
                    has_password=bool(self.account.takeout.archive_password),
                    created_at=datetime_to_integer_unixtime(self.account.takeout.archive_created_at),
                    valid_until=datetime_to_integer_unixtime(archive_valid_until),
                ),
            )


class GetArchiveUrlView(BaseGetExtractResultView):
    @cached_property
    def s3(self):
        return get_s3_client()

    def fill_response_with_extract_results(self):
        if self.account.takeout.extract_in_progress_since:
            raise ExtractInProgressError()

        if not self.is_archive_available:
            raise ActionImpossibleError()

        self.response_values.update(
            archive_url=self.s3.generate_presigned_url_for_file(
                key=self.account.takeout.archive_s3_key,
                expires_in=settings.S3_PRESIGNED_URL_TTL,
                content_type=settings.TAKEOUT_ARCHIVE_CONTENT_TYPE,
            ),
        )


class GetArchivePasswordView(BaseGetExtractResultView):
    def fill_response_with_extract_results(self):
        if self.account.takeout.extract_in_progress_since:
            raise ExtractInProgressError()

        if not self.is_archive_available:
            raise ActionImpossibleError()

        self.response_values.update(
            archive_password=self.account.takeout.archive_password or None,
        )


class DeleteExtractResultView(BaseBundleView, BundleAccountGetterMixin):
    required_grants = [DELETE_EXTRACT_RESULT_GRANT]
    basic_form = DeleteExtractResultForm

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
        )

        events = {
            'action': 'delete_takeout_extract_result',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            if self.account.takeout.archive_s3_key and self.account.takeout.fail_extract_at:
                # Костыль для автотестов: если выгрузка прошла быстро, то ручка start могла вызваться позже ручки
                # finish, и в базе остался лишний атрибут fail_extract_at
                self.account.takeout.fail_extract_at = None
            self.account.takeout.archive_s3_key = None
            self.account.takeout.archive_password = None
            self.account.takeout.archive_created_at = None
