# -*- coding: utf-8 -*-
from passport.backend.api.common.authorization import (
    get_session_policy_by_ttl,
    is_session_created,
    set_authorization_track_fields,
    user_session_scope,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.states import ForwardAuth
from passport.backend.api.views.bundle.track.forms import (
    InitializeInfectedTrackForm,
    InitializeTrackForm,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import always_str
from six.moves.urllib.parse import unquote


class InitializeTrackView(BaseBundleView,
                          BundleAccountGetterMixin):
    require_track = False
    required_headers = {
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
    }
    required_grants = ['track.initialize']
    basic_form = InitializeTrackForm

    def process_request(self):
        self.process_basic_form()
        self.create_track(track_type=self.form_values['track_type'], process_name=self.form_values['process_name'])

        self.get_account_from_session_or_oauth_token(
            multisession_uid=self.form_values['uid'],
        )

        with self.track_transaction.rollback_on_error() as track:
            track.uid = self.account.uid

        self.response_values['track_id'] = self.track_id
        self.statbox.log(uid=self.account.uid)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            mode='initialize_track',
            track_id=self.track_id,
            yandexuid=self.cookies.get('yandexuid'),
        )


class InitializeInfectedTrackView(BaseBundleView, BundleAccountGetterMixin):
    require_track = False
    required_headers = {
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
    }
    required_grants = ['track.initialize_infected']
    basic_form = InitializeInfectedTrackForm
    track_type = 'authorize'

    def _try_unquote_yandex_login(self, cookie_value):
        login_part, at, domain_part = always_str(cookie_value).partition('@')
        quoted_domain_part = unquote(domain_part)
        return ''.join([login_part, at, quoted_domain_part])

    def process_request(self):
        self.process_basic_form()
        self.create_track(track_type=self.track_type)
        self.get_account_from_session()

        with self.track_transaction.rollback_on_error() as track:
            track.retpath = self.form_values['retpath']
            track.authorization_session_policy = get_session_policy_by_ttl(self.session_info.ttl)
            track.is_strong_password_policy_required = self.account.is_strong_password_required
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                old_session_info=self.session_info,
                password_passed=False,
                session_scope=user_session_scope(self.session_info, self.account.uid),
            )
            track.cookie_l_value = self.request.env.cookies.get('L')
            track.cookie_my_value = self.request.env.cookies.get('my')
            track.cookie_yp_value = self.request.env.cookies.get('yp')
            track.cookie_ys_value = self.request.env.cookies.get('ys')
            track.cookie_yandex_login_value = self._try_unquote_yandex_login(
                self.request.env.cookies.get('yandex_login'),
            )
            track.cookie_yandex_gid_value = self.request.env.cookies.get('yandex_gid')
            track.use_non_auth_cookies_from_track = True

        self.response_values['track_id'] = self.track_id
        self.statbox.log(uid=self.account.uid)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            mode='initialize_infected_track',
            track_id=self.track_id,
            yandexuid=self.cookies.get('yandexuid'),
        )


class DispatchTrackView(BaseBundleView):
    required_grants = ['track.dispatch']

    require_track = True

    def process_request(self):
        self.read_track()
        self.response_values['track_id'] = self.track_id

        if self.track.allow_authorization and not is_session_created(self.track):
            return self.dispatch_auth_forwarding()

        raise InvalidTrackStateError()

    def dispatch_auth_forwarding(self):
        self.state = ForwardAuth()
