# -*- coding: utf-8 -*-
import logging

from passport.backend.core.eav_type_mapping import ATTRIBUTE_NAME_TO_TYPE as AT
from passport.backend.core.exceptions import WrongHostError
from passport.backend.core.logging_utils.loggers import PhoneLogger
from passport.backend.core.models.account import ACCOUNT_DISABLED_ON_DELETION
from passport.backend.core.types.question import Question
from passport.backend.core.utils.domains import get_keyspace_by_host
from passport.backend.utils.string import smart_text

from .exceptions import (
    HeadersEmptyError,
    InvalidHostError,
    QuestionEmptyError,
    QuestionInconsistentError,
)


log = logging.getLogger('passport.api.views.bundle.utils')
phone_logger = PhoneLogger()


def assert_valid_host(env):
    """Проверим валидность хоста из заголовка"""
    host = env.host.strip()
    if not host:
        raise HeadersEmptyError(['host'])

    try:
        get_keyspace_by_host(host)
    except WrongHostError as ex:
        raise InvalidHostError(smart_text(ex))


def write_phone_to_log(account, user_cookies, phone=None):
    if not account.phones.secure and not phone:
        return

    phone_logger.log(
        uid=account.uid,
        phone=phone or account.phones.secure.number.e164,
        yandexuid=user_cookies.get('yandexuid', ''),
    )


def make_hint_question(question=None, question_id=None, display_language=None):
    """Создаёт контрольный вопрос из question или question_id"""
    if not (question or question_id):
        raise QuestionEmptyError()
    if question and question_id:
        raise QuestionInconsistentError()
    if question:
        question = Question(text=question)
    else:
        question = Question.from_id(
            display_language,
            question_id,
        )
    return question


def is_user_disabled_on_deletion(session):
    """
    Отвечаем на вопрос: является ли данный пользователь заблокированным
    по причине наличия подписки на запрещенные SID'ы при удалении?

    Правильнее на этот вопрос отвечать после преобразования ответа
    в модель аккаунта, но это достаточно тяжеловесная операция.
    """
    attributes = session.get('attributes', {})
    is_disabled_attr = str(AT['account.is_disabled'])
    status = attributes.get(is_disabled_attr)
    return status is not None and int(status) == ACCOUNT_DISABLED_ON_DELETION
