# -*- coding: utf-8 -*-
import logging
import time

from passport.backend.api.common.common import validate_password
from passport.backend.api.common.pdd import (
    does_domain_belong_to_pdd,
    is_domain_suitable_for_directory,
)
from passport.backend.api.common.phone_number import MIN_STATBOX_PHONE_LENGTH
from passport.backend.api.common.processes import (
    PROCESS_LOGIN_RESTORE,
    PROCESS_RESTORE,
    PROCESS_WEB_REGISTRATION,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    DomainAlreadyExistsError,
    DomainInvalidTypeError,
    DomainMissingError,
    DomainNotFoundError,
    PublicIdUpdateNotAllowed,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins.account import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.common import (
    BundleCleanWebMixin,
    BundleTvmUserTicketMixin,
)
from passport.backend.api.views.bundle.mixins.password import BundlePinValidationMixin
from passport.backend.api.views.bundle.mixins.phone import (
    OctopusAvailabilityMixin,
    PhoneValidForCallStatus,
)
from passport.backend.api.views.bundle.phone import helpers as phone_helpers
from passport.backend.api.views.bundle.phone.exceptions import PhoneNotFoundError
from passport.backend.api.views.bundle.validate.forms import (
    CommonLoginForm,
    DisplayNameForm,
    FirstNameForm,
    FullNameForm,
    LastNameForm,
    LiteLoginForm,
    LoginForm,
    PasswordForm,
    PddDomainForm,
    PddLoginForm,
    PhoneIDForm,
    PhoneNumberBySquatterForm,
    PhoneNumberForm,
    PublicIdForm,
    SCENARIO_REGISTER, TotpPinForm,
)
from passport.backend.core import validators
from passport.backend.core.builders.phone_squatter import (
    BasePhoneSquatterError,
    get_phone_squatter,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
)
from passport.backend.core.types.phone_number.phone_number import mask_for_statbox
from passport.backend.core.utils.decorators import cached_property


IGNORE_STOPLIST_GRANT = 'ignore_stoplist'
SKIP_DOMAIN_EXISTENCE_CHECK_GRANT = 'login.skip_domain_existence_check'

MOBILEPROXY_CONSUMER = 'mobileproxy'

log = logging.getLogger(__name__)


class Login(BaseBundleView):
    """
    Бандловая ручка валидации логина пользователя.

    """
    require_track = True
    required_headers = [HEADER_CONSUMER_CLIENT_IP]
    required_grants = ['login.validate']
    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]

    basic_form = LoginForm

    def process_request(self):
        self.process_basic_form()

        if self.form_values['ignore_stoplist']:
            self.check_grant(IGNORE_STOPLIST_GRANT)

        if not self.form_values['require_domain_existence']:
            self.check_grant(SKIP_DOMAIN_EXISTENCE_CHECK_GRANT)

        state = validators.State(self.request.env)
        is_pdd = self.form_values['is_pdd']
        is_lite = self.form_values['is_lite']

        if (is_pdd or is_lite) and '@' not in self.form_values['login']:
            raise DomainMissingError()

        if is_pdd:
            login_type = ACCOUNT_TYPE_PDD
            login, domain = self.form_values['login'].rsplit('@', 1)

            # Валидируем логин отдельно
            self.process_form(
                PddLoginForm,
                dict(login=login),
            )
            if login.rstrip() != login:  # rstrip, так как слева пробелы убрал strip в валидаторе формы
                raise ValidationFailedError(
                    ['login.invalid'],
                )

            # Проверим на корректность переданный домен
            charset = 'default'
            if self.form_values['strict']:
                charset = 'strict'

            validator = validators.Hostname(character_set=charset)
            try:
                validator.validate_python(domain, state)
            except validators.Invalid as exc:
                raise ValidationFailedError(
                    ['domain.invalid'],
                    invalid_exception=exc,
                )

            if self.form_values['require_domain_existence']:
                domains = self.blackbox.hosted_domains(domain=domain)
                if not domains['hosted_domains']:
                    raise DomainNotFoundError()

        elif is_lite:
            login_type = ACCOUNT_TYPE_LITE
            login = self.all_values['login']
            self.process_form(
                LiteLoginForm,
                dict(login=login),
            )
            if does_domain_belong_to_pdd(login):
                raise DomainInvalidTypeError()
        else:
            login_type = ACCOUNT_TYPE_NORMAL
            self.process_form(
                CommonLoginForm,
                dict(login=self.all_values['login']),
            )

        try:
            availability_validator = validators.Availability(login_type=login_type)
            availability_validator.to_python(self.form_values, state)
        except validators.Invalid as exc:
            raise ValidationFailedError(
                ['login.not_available'],
                invalid_exception=exc,
            )

        self.read_track()
        with self.track_transaction.rollback_on_error():
            self.track.login_validation_count.incr()
            self.track.login = self.form_values['login']
            if self.track.login_validation_first_call is None:
                self.track.login_validation_first_call = time.time()
            self.track.login_validation_last_call = time.time()


class Password(BaseBundleView):
    """
    Бандловая ручка валидации пароля.
    """
    require_track = True
    required_headers = [HEADER_CONSUMER_CLIENT_IP]
    required_grants = ['password.validate']
    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]

    basic_form = PasswordForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        try:
            validated, p = validate_password(self.form_values, self.track)
            if validated.get('lt_middle_quality'):
                self.response_values.update(warnings=['password.weak'])
        except validators.Invalid as e:
            raise ValidationFailedError.from_invalid(e)

        with self.track_transaction.rollback_on_error() as track:
            track.password_validation_count.incr()
            if track.password_validation_first_call is None:
                track.password_validation_first_call = time.time()
            track.password_validation_last_call = time.time()


class PddDomain(BaseBundleView):

    required_grants = ['domain.validate']

    basic_form = PddDomainForm

    def process_request(self):
        self.process_basic_form()

        charset = 'default'
        if self.form_values['strict']:
            charset = 'strict'

        # Проверим на корректность переданный домен
        validator = validators.Hostname(character_set=charset)
        state = validators.State(self.request.env)
        try:
            validator.to_python(self.form_values['domain'], state)
        except validators.Invalid as exc:
            raise ValidationFailedError(
                ['domain.invalid'],
                invalid_exception=exc,
            )

        domains = self.blackbox.hosted_domains(domain=self.form_values['domain'])
        if bool(domains['hosted_domains']):
            raise DomainAlreadyExistsError()


class DirectoryDomain(PddDomain):
    def process_request(self):
        super(DirectoryDomain, self).process_request()

        if not is_domain_suitable_for_directory(self.form_values['domain']):
            raise DomainInvalidTypeError()


class PhoneNumber(BaseBundleView, OctopusAvailabilityMixin):

    required_grants = ['phone_number.validate']

    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
        PROCESS_LOGIN_RESTORE,
        PROCESS_RESTORE,
    ]

    basic_form = PhoneNumberForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            action='sanitize_phone_number',
            track_id=self.track_id if self.track_id else '',
        )

    def check_valid_for_call(self, phone):
        if self.consumer == MOBILEPROXY_CONSUMER:
            not_valid_for_calls = PhoneValidForCallStatus(
                valid_for_call=False,
                valid_for_flash_call=False,
            )
            return not_valid_for_calls
        return super(PhoneNumber, self).check_valid_for_call(self.consumer, phone)

    def get_phone_number(self):
        return self.form_values['phone_number']

    @property
    def to_validate_for_call(self):
        return self.form_values['validate_for_call']

    def process_request(self):
        user_phone_number = self.all_values.get('phone_number', '')
        valid_for_call_status = None
        phone_number = None
        if self.track_id:
            self.read_track()

        try:
            self.process_basic_form()
            phone_number = self.get_phone_number()

            self.response_values['phone_number'] = phone_helpers.dump_number(
                phone_number,
            )

            self.statbox.bind(sanitize_phone_result=phone_number.masked_format_for_statbox)
            if self.to_validate_for_call:
                valid_for_call_status = self.check_valid_for_call(phone_number)
                self.response_values['valid_for_call'] = valid_for_call_status.valid_for_call
                self.response_values['valid_for_flash_call'] = valid_for_call_status.valid_for_flash_call
        except ValidationFailedError as e:
            self.statbox.bind(
                sanitize_phone_result=mask_for_statbox(user_phone_number),
                sanitize_phone_error=e._invalid_exception.code,
            )
            raise
        finally:
            if self.track:
                with self.track_transaction.rollback_on_error():
                    self.track.sanitize_phone_error = 'sanitize_phone_error' in self.statbox.all_values
                    self.track.sanitize_phone_count.incr()
                    if self.track.sanitize_phone_first_call is None:
                        self.track.sanitize_phone_first_call = time.time()
                    self.track.sanitize_phone_last_call = time.time()
                    if phone_number and self.to_validate_for_call and valid_for_call_status:
                        self.track.phone_valid_for_call = valid_for_call_status.valid_for_call
                        self.track.phone_valid_for_flash_call = valid_for_call_status.valid_for_flash_call
                        self.track.phone_validated_for_call = phone_number.e164

            if len(user_phone_number) >= MIN_STATBOX_PHONE_LENGTH:
                self.statbox.log()


class PhoneNumberBySquatter(BaseBundleView):

    required_grants = ['phone_number.validate_by_squatter']

    basic_form = PhoneNumberBySquatterForm

    def process_request(self):
        self.process_basic_form()

        allow_new_flow = settings.USE_NEW_SUGGEST_BY_PHONE
        if allow_new_flow and settings.USE_PHONE_SQUATTER:
            try:
                if self.form_values['scenario'] == SCENARIO_REGISTER:
                    get_phone_squatter().start_tracking(
                        phone_number=self.form_values['phone_number'].e164,
                        request_id=self.request.env.request_id,
                        is_draft=True,
                    )
                else:
                    get_phone_squatter().get_change_status(
                        phone_number=self.form_values['phone_number'].e164,
                        request_id=self.request.env.request_id,
                        allow_cached=True,
                    )
            except BasePhoneSquatterError:
                if settings.PHONE_SQUATTER_DRY_RUN:
                    log.debug('Phone not accepted by PhoneSquatter, but still allowing new flow due to dry run')
                else:
                    allow_new_flow = False

        self.response_values.update(
            require_flow_with_fio=not allow_new_flow,
        )


class PhoneID(PhoneNumber, BundleAccountGetterMixin):

    required_grants = ['phone_id.validate']

    basic_form = PhoneIDForm

    require_track = True

    def get_phone_number(self):
        phone_id = self.form_values['phone_id']

        if self.form_values['uid']:
            self.check_grant('phone_id.by_uid')
            self.get_account_by_uid(self.form_values['uid'], need_phones=True)
        else:
            self.get_account_from_track(need_phones=True)

        if self.account.phones.has_id(phone_id):
            phone_number = self.account.phones.by_id(phone_id).number
        else:
            raise PhoneNotFoundError()

        return phone_number

    @property
    def to_validate_for_call(self):
        return True

    def process_request(self):
        super(PhoneID, self).process_request()
        self.response_values = {
            k: self.response_values[k]
            for k in self.response_values
            if k in ['valid_for_call', 'valid_for_flash_call']
        }


class TotpPin(BaseBundleView, BundlePinValidationMixin):

    required_grants = ['otp.validate']

    basic_form = TotpPinForm

    sensitive_fields = [
        'pin',
    ]

    def process_request(self):
        self.process_basic_form()
        self.response_values['pin'] = self.validate_pin(pin=self.form_values['pin'])
        self.response_values.update(
            sensitive_fields=self.sensitive_fields,
        )


class TotpPinV2(TotpPin):
    require_track = True

    def process_request(self):
        self.read_track()
        super(TotpPinV2, self).process_request()


class DisplayName(BaseBundleView, BundleCleanWebMixin):
    """
    Бандловая ручка валидации display name.
    """
    required_grants = ['display_name.validate']
    basic_form = DisplayNameForm

    def process_request(self):
        self.process_basic_form()
        self.clean_web_check_form_values()


class FirstName(BaseBundleView, BundleCleanWebMixin):
    """
    Бандловая ручка валидации firstname.
    """
    required_grants = ['firstname.validate']
    basic_form = FirstNameForm

    def process_request(self):
        self.process_basic_form()
        self.clean_web_check_form_values()


class LastName(BaseBundleView, BundleCleanWebMixin):
    """
    Бандловая ручка валидации lastname.
    """
    required_grants = ['lastname.validate']
    basic_form = LastNameForm

    def process_request(self):
        self.process_basic_form()
        self.clean_web_check_form_values()


class FullName(BaseBundleView, BundleCleanWebMixin):
    """
    Бандловая ручка валидации firstname и lastname.
    """
    required_grants = ['firstname.validate', 'lastname.validate']
    basic_form = FullNameForm

    def process_request(self):
        self.process_basic_form()
        self.clean_web_check_form_values()


class PublicId(BaseBundleView, BundleCleanWebMixin, BundleAccountGetterMixin, BundleTvmUserTicketMixin):
    """
    Бандловая ручка валидации public_id
    """
    required_grants = ['public_id.validate']
    basic_form = PublicIdForm

    def process_request(self):
        self.process_basic_form()
        self.clean_web_check_form_values()

        self.get_account_from_available_media(
            multisession_uid=self.form_values['multisession_uid'],
            ignore_grants=True,
        )

        if self.account.public_id_alias.alias and len(self.account.public_id_alias.old_public_ids):
            raise PublicIdUpdateNotAllowed()
        try:
            availability_validator = validators.PublicIdAvailability(uid=self.account.uid)
            availability_validator.to_python(self.form_values, validators.State(self.request.env))
        except validators.Invalid as exc:
            raise ValidationFailedError(
                ['public_id.not_available'],
                invalid_exception=exc,
            )


__all__ = (
    'PddDomain',
    'PhoneNumber',
    'TotpPin',
    'TotpPinV2',
)
