# -*- coding: utf-8 -*-

from passport.backend.core import validators


SCENARIO_AUTH = 'auth'
SCENARIO_REGISTER = 'register'


class LoginForm(validators.Schema):
    """Форма корректности логина"""
    login = validators.String(not_empty=True, strip=True)
    ignore_stoplist = validators.StringBool(if_missing=False, not_empty=True)
    is_pdd = validators.StringBool(if_missing=False)
    is_lite = validators.StringBool(if_missing=False)
    strict = validators.StringBool(if_missing=False)
    require_domain_existence = validators.StringBool(if_missing=True)

    chained_validators = [
        validators.RequireSome(
            [
                'is_pdd',
                'is_lite',
            ],
            min_=0,
            max_=1,
            empty_values=[False],
        ),
    ]


class PasswordForm(validators.Schema):
    password = validators.String(not_empty=True)
    login = validators.String(if_missing=None)
    policy = validators.PasswordPolicy(if_missing=None, not_empty=True)
    phone_number = validators.String(if_missing=None, not_empty=True)
    country = validators.CountryCode(if_missing=None)


class PddLoginForm(validators.Schema):
    login = validators.PddLogin()


class LiteLoginForm(validators.Schema):
    login = validators.LiteLogin()


class CommonLoginForm(validators.Schema):
    login = validators.Login()


class PddDomainForm(validators.Schema):
    """Форма проверки незанятости ПДД-домена"""
    domain = validators.String(not_empty=True)
    strict = validators.StringBool(if_missing=False)


class BasePhoneNumberForm(validators.Schema):
    country = validators.CountryCode(if_missing=None, strip=True)
    phone_number = validators.String(not_empty=True, strip=True)

    chained_validators = [
        validators.PhoneNumber(),
    ]


class PhoneNumberForm(BasePhoneNumberForm):
    validate_for_call = validators.StringBool(if_empty=True, if_missing=False, strip=True)


class PhoneNumberBySquatterForm(BasePhoneNumberForm):
    scenario = validators.OneOf(
        [SCENARIO_AUTH, SCENARIO_REGISTER],
        if_missing=SCENARIO_REGISTER,
        not_empty=True,
    )


class PhoneIDForm(validators.Schema):
    phone_id = validators.PhoneId()
    uid = validators.Uid(if_missing=None)


class TotpPinForm(validators.Schema):
    pin = validators.String(not_empty=True)


class DisplayNameForm(validators.Schema):
    pre_validators = [
        validators.GroupDisplayName(),
    ]

    force_clean_web = validators.StringBool(if_missing=False)
    display_name = validators.DisplayName()

    chained_validators = [
        validators.RequireSome([
            'display_name',
        ]),
    ]


class FirstNameForm(validators.Schema):
    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(not_empty=True, max=50)


class LastNameForm(validators.Schema):
    force_clean_web = validators.StringBool(if_missing=False)
    lastname = validators.AntiFraudLastName(not_empty=True, max=50)


class FullNameForm(validators.Schema):
    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(not_empty=True, max=50, if_missing=None)
    lastname = validators.AntiFraudLastName(not_empty=True, max=50, if_missing=None)

    chained_validators = [
        validators.RequireSome(
            [
                'firstname',
                'lastname',
            ],
            min_=1,
            max_=2,
        ),
    ]


class PublicIdForm(validators.Schema):
    force_clean_web = validators.StringBool(if_missing=False)
    public_id = validators.PublicId(not_empty=True)

    uid = validators.Uid(if_missing=None, not_empty=True)
    multisession_uid = validators.Uid(if_missing=None, not_empty=True)
