# -*- coding: utf-8 -*-
import base64
from datetime import datetime
import logging
import os

from passport.backend.api.common.authorization import build_cookie_wcid
from passport.backend.api.common.webauthn import get_suggested_webauthn_credentials
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionImpossibleError,
    InvalidTrackStateError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.phone.exceptions import (
    PhoneNotConfirmedError,
    PhoneNotFoundError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.webauthn import WebauthnCredential
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.domains import build_passport_domain
from passport.backend.utils.string import always_str
from passport.backend.utils.time import datetime_to_integer_unixtime
import webauthn
from webauthn.webauthn import (
    AuthenticationRejectedException,
    RegistrationRejectedException,
)

from .exceptions import (
    WebauthnCredentialExistsError,
    WebauthnCredentialLimitReachedError,
    WebauthnCredentialNotFoundError,
    WebauthnCredentialOccupiedError,
    WebauthnRegistrationRejectedError,
    WebauthnVerificationRejectedError,
)
from .forms import (
    WebauthnAddCredentialCommitForm,
    WebauthnRemoveCredentialForm,
    WebauthnVerifyCommitForm,
    WebauthnVerifySubmitForm,
)


log = logging.getLogger(__name__)


AVATAR_SIZE = 'normal'


class BaseWebauthnView(BaseBundleView, BundleAccountGetterMixin):
    require_track = True
    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            mode='webauthn',
            yandexuid=self.cookies.get('yandexuid'),
        )

    def get_rp_id(self):
        return build_passport_domain(self.request.env.host)

    @staticmethod
    def generate_challenge(length):
        # TODO: на py3 заменить на secrets.token_urlsafe
        challenge_bytes = os.urandom(length)
        challenge_base64 = base64.urlsafe_b64encode(challenge_bytes).decode()
        return challenge_base64.rstrip('=')


class BaseWebauthnManageView(BaseWebauthnView, BundlePhoneMixin):
    required_grants = ['webauthn.manage']
    require_confirmed_phone = True

    def assert_creds_limit_not_reached(self):
        if len(self.account.webauthn_credentials.all()) >= settings.WEBAUTHN_CREDENTIALS_MAX_COUNT:
            raise WebauthnCredentialLimitReachedError()

    def process_request(self):
        if self.basic_form:
            self.process_basic_form()

        self.read_track()
        if not self.track.uid:
            raise InvalidTrackStateError()

        self.get_account_from_session(
            check_disabled_on_deletion=True,
            need_phones=True,
            get_webauthn_credentials=True,
            multisession_uid=int(self.track.uid),
        )
        self.statbox.bind_context(uid=self.account.uid)

        if self.require_confirmed_phone:
            if not self.account.phones.secure:
                raise PhoneNotFoundError()
            if not self.is_secure_phone_confirmed_in_track(allow_by_call=False):
                raise PhoneNotConfirmedError()

        self._process()

    def _process(self):
        raise NotImplementedError()


class WebauthnListCredentialsView(BaseWebauthnManageView):
    require_confirmed_phone = False

    @staticmethod
    def credential_to_response(credential, is_suggested):
        return {
            'external_id': credential.external_id,
            'device_name': credential.device_name or None,
            'os_family_name': settings.OS_FAMILY_DECODE.get(credential.os_family_id),
            'browser_name': settings.BROWSER_DECODE.get(credential.browser_id),
            'is_device_mobile': bool(credential.is_device_mobile),
            'is_device_tablet': bool(credential.is_device_tablet),
            'created_at': datetime_to_integer_unixtime(credential.created_at),
            'is_suggested': is_suggested,
        }

    def _process(self):
        suggested_credentials = get_suggested_webauthn_credentials(account=self.account, env=self.request.env)
        self.response_values.update(
            webauthn_credentials=[
                self.credential_to_response(
                    credential,
                    is_suggested=credential in suggested_credentials,
                )
                for credential in self.account.webauthn_credentials.all()
            ],
        )


class WebauthnAddCredentialSubmitView(BaseWebauthnManageView):
    def _process(self):
        self.assert_creds_limit_not_reached()

        challenge = self.generate_challenge(length=settings.WEBAUTHN_CHALLENGE_DEFAULT_BYTE_LEN)
        avatar_url = settings.GET_AVATAR_URL % (
            settings.DEFAULT_AVATAR_KEY,
            AVATAR_SIZE,
        )
        make_credential_options = webauthn.WebAuthnMakeCredentialOptions(
            challenge=challenge,
            rp_name=settings.WEBAUTHN_RELYING_PARTY_NAME,
            rp_id=self.get_rp_id(),
            user_id=str(self.account.uid),
            username=self.account.login,
            display_name=self.account.person.display_name.name,
            icon_url=avatar_url,
            user_verification='required' if settings.WEBAUTHN_IS_USER_VERIFICATION_REQUIRED else 'preferred',
        ).registration_dict
        if settings.WEBAUTHN_AUTHENTICATOR_ATTACHMENT is not None:
            # либа не умеет эту опцию, приходится добавлять вручную
            make_credential_options['authenticatorSelection'].update(
                authenticatorAttachment=settings.WEBAUTHN_AUTHENTICATOR_ATTACHMENT,
            )

        with self.track_transaction.rollback_on_error():
            self.track.webauthn_challenge = challenge

        self.response_values.update(
            track_id=self.track_id,
            make_credential_options=make_credential_options,
        )


class WebauthnAddCredentialCommitView(BaseWebauthnManageView):
    basic_form = WebauthnAddCredentialCommitForm

    def _process(self):
        self.assert_creds_limit_not_reached()
        if not self.track.webauthn_challenge:
            raise InvalidTrackStateError()

        rp_id = self.get_rp_id()
        webauthn_registration_response = webauthn.WebAuthnRegistrationResponse(
            rp_id=rp_id,
            origin=self.form_values['origin'],
            registration_response={
                'clientData': self.form_values['client_data'],
                'attObj': self.form_values['attestation_object'],
            },
            challenge=self.track.webauthn_challenge,
            trusted_attestation_cert_required=settings.WEBAUTHN_IS_TRUSTED_ATTESTATION_CERT_REQUIRED,
            self_attestation_permitted=settings.WEBAUTHN_IS_SELF_ATTESTATION_PERMITTED,
            none_attestation_permitted=settings.WEBAUTHN_IS_NONE_ATTESTATION_PERMITTED,
            uv_required=settings.WEBAUTHN_IS_USER_VERIFICATION_REQUIRED,
        )

        try:
            webauthn_credential_raw = webauthn_registration_response.verify()
        except RegistrationRejectedException as e:
            log.debug('Failed to verify client response while adding webauthn credential: %s', e)
            raise WebauthnRegistrationRejectedError()

        credential_external_id = always_str(webauthn_credential_raw.credential_id)
        bb_response = self.blackbox.webauthn_credentials(
            credential_external_id=credential_external_id,
        )
        owner_uid = bb_response[credential_external_id]['uid']
        if owner_uid == self.account.uid:
            log.debug('Credential %s already belongs to current user %s', credential_external_id, self.account.uid)
            raise WebauthnCredentialExistsError()
        elif owner_uid is not None:
            log.debug('Credential %s already belongs to another user %s', credential_external_id, owner_uid)
            raise WebauthnCredentialOccupiedError()

        user_agent_info = self.request.env.user_agent_info
        browser_id = settings.BROWSER_ENCODE.get(user_agent_info.get('BrowserName')) or None
        os_family_id = settings.OS_FAMILY_ENCODE.get(user_agent_info.get('OSFamily')) or None

        with UPDATE(self.account, self.request.env, {'action': 'add_webauthn_credential'}):
            cred = WebauthnCredential(
                external_id=credential_external_id,
                public_key=always_str(webauthn_credential_raw.public_key),
                relying_party_id=rp_id,
                device_name=self.form_values['device_name'],
                browser_id=browser_id,
                os_family_id=os_family_id,
                is_device_mobile=user_agent_info.get('isMobile'),
                is_device_tablet=user_agent_info.get('isTablet'),
                created_at=datetime.now(),
            )
            self.account.webauthn_credentials.add(cred)

        self.statbox.log(
            action='add_credential',
            credential_external_id=credential_external_id,
            relying_party_id=rp_id,
            device_name=self.form_values['device_name'],
        )

        cookies = [
            build_cookie_wcid(
                self.request.env,
                webauthn_credential_external_id=credential_external_id,
            ),
        ]
        self.response_values.update(
            cookies=cookies,
        )


class WebauthnRemoveCredentialView(BaseWebauthnManageView):
    basic_form = WebauthnRemoveCredentialForm
    require_confirmed_phone = False

    def _process(self):
        credential_external_id = self.form_values['credential_external_id']
        if credential_external_id not in self.account.webauthn_credentials:
            raise ActionImpossibleError()

        with UPDATE(self.account, self.request.env, {'action': 'remove_webauthn_credential'}):
            self.account.webauthn_credentials.remove(credential_external_id)

        if self.request.env.cookies.get('wcid') == credential_external_id:
            # Удалим неактуальную куку
            cookies = [
                build_cookie_wcid(
                    self.request.env,
                    webauthn_credential_external_id='',
                    expires=1,
                ),
            ]
            self.response_values.update(
                cookies=cookies,
            )


class BaseWebauthnVerifyView(BaseWebauthnView):
    required_grants = ['webauthn.verify']

    def make_webauthn_user(self, credential):
        avatar_url = settings.GET_AVATAR_URL % (
            settings.DEFAULT_AVATAR_KEY,
            AVATAR_SIZE,
        )
        return webauthn.WebAuthnUser(
            user_id=str(self.account.uid),
            username=self.account.login,
            display_name=self.account.person.display_name.name,
            icon_url=avatar_url,
            credential_id=credential.external_id,
            public_key=credential.public_key,
            sign_count=credential.sign_count,
            rp_id=self.get_rp_id(),
        )

    def process_request(self):
        if self.basic_form:
            self.process_basic_form()

        self.read_track()
        uid = self.form_values['uid'] or self.track.uid
        if not uid:
            raise ValidationFailedError(['uid.empty'])

        self.get_account_by_uid(
            uid=uid,
            check_disabled_on_deletion=True,
            need_phones=True,
            get_webauthn_credentials=True,
        )
        self.statbox.bind_context(uid=self.account.uid)

        cred_external_id = self.form_values['credential_external_id']
        cred = self.account.webauthn_credentials.by_external_id(cred_external_id)
        if cred is None:
            raise WebauthnCredentialNotFoundError()

        self._process(cred)

    def _process(self, credential):
        raise NotImplementedError()


class WebauthnVerifySubmitView(BaseWebauthnVerifyView):
    basic_form = WebauthnVerifySubmitForm

    def _process(self, credential):
        challenge = self.generate_challenge(length=settings.WEBAUTHN_CHALLENGE_DEFAULT_BYTE_LEN)

        webauthn_user = self.make_webauthn_user(credential)
        webauthn_assertion_options = webauthn.WebAuthnAssertionOptions(
            webauthn_user,
            challenge,
            userVerification='required' if settings.WEBAUTHN_IS_USER_VERIFICATION_REQUIRED else 'preferred',
        ).assertion_dict

        with self.track_transaction.rollback_on_error():
            self.track.webauthn_challenge = challenge

        self.response_values.update(
            track_id=self.track_id,
            assertion_options=webauthn_assertion_options,
        )


class WebauthnVerifyCommitView(BaseWebauthnVerifyView):
    basic_form = WebauthnVerifyCommitForm

    def _process(self, credential):
        webauthn_user = self.make_webauthn_user(credential)
        webauthn_assertion_response = webauthn.WebAuthnAssertionResponse(
            webauthn_user=webauthn_user,
            assertion_response={
                'clientData': self.form_values['client_data'],
                'authData': self.form_values['auth_data'],
                'signature': self.form_values['signature'],
            },
            challenge=self.track.webauthn_challenge,
            origin=self.form_values['origin'],
            uv_required=settings.WEBAUTHN_IS_USER_VERIFICATION_REQUIRED,
        )

        try:
            sign_count = webauthn_assertion_response.verify()
        except AuthenticationRejectedException as e:
            log.debug('Failed to verify client response while authenticating via webauthn: %s', e)
            raise WebauthnVerificationRejectedError()

        with UPDATE(self.account, self.request.env, {'action': 'webauthn_verify'}):
            credential.sign_count = sign_count

        with self.track_transaction.rollback_on_error():
            # Кладём в трек значение без паддинга (либа ожидает его в таком формате)
            self.track.webauthn_confirmed_secret_external_id = credential.external_id
