# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.processes import PROCESS_YAKEY_BACKUP
from passport.backend.api.forms.base import DeviceInfoForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    YaKeyBackupAlreadyExistsError,
    YaKeyBackupNotFoundError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundlePhoneMixin
from passport.backend.api.views.bundle.yakey_backups import forms
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_YAKEY_BACKUP_EXISTS_STATUS,
    BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.yakey_backup import YaKeyBackup
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import smart_str


YAKEY_BACKUP_GRANT = 'allow_yakey_backup'


log = logging.getLogger(__name__)


class BackupBaseView(BaseBundleView, BundlePhoneMixin):
    """
    Базовый класс для работы с загрузкой/выгрузкой бэкапов секретов Ключа
    """
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )
    require_process = True
    require_track = True
    required_grants = [YAKEY_BACKUP_GRANT]

    process_name = PROCESS_YAKEY_BACKUP

    basic_form = forms.BaseBackupForm()

    mode = 'yakey_backup'
    step = None

    number = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.mode,
            step=self.step,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def assert_number_confirmed(self):
        # Убедимся, что переданный номер совпадает с тем, что подтвержден и сохранен в треке
        if not self.track.phone_confirmation_phone_number_original:
            raise InvalidTrackStateError()
        track_number = PhoneNumber.parse(
            self.track.phone_confirmation_phone_number_original,
            self.track.country,
        )
        if not (track_number.e164 == self.number.e164 and self.is_phone_confirmed_in_track()):
            raise InvalidTrackStateError()

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.number = self.form_values['number']
        self.assert_number_confirmed()
        # Запишем информацию об устройстве
        device_info = {}
        for device_param in DeviceInfoForm.DEVICE_INFO_FIELD_NAMES:
            value = self.form_values.get(device_param)
            device_info[device_param] = smart_str(value) if value else None

        self.statbox.bind_context(
            track_id=self.track_id,
            number=self.number.masked_format_for_statbox,
            **device_info
        )
        self._process()
        self.statbox.log(status='ok')

    def _process(self):
        raise NotImplementedError()     # pragma: no cover


class BackupUploadView(BackupBaseView):
    """
    Второй шаг сохранения бэкапа секретов Ключа в базу Паспорта.
    Принимает бэкап, валидирует его и сохраняет в базу.
    Если бэкап для данного телефона уже есть, сначала нужно подтверждение
    пользователя перезаписать старый. Данные об этом получаем отдельным параметром
    """

    step = 'upload'
    basic_form = forms.UploadCommitForm()

    def _process(self):

        response = self.blackbox.yakey_backup(self.number.digital)
        exists = response['status'] == BLACKBOX_YAKEY_BACKUP_EXISTS_STATUS

        if not exists:
            with CREATE(YaKeyBackup(), self.request.env, {}) as yakey_backup:
                yakey_backup.phone_number = self.number
                yakey_backup.backup = self.form_values['backup']
                yakey_backup.updated = datetime.now()
                yakey_backup.device_name = self.form_values.get('device_name')
        elif self.form_values['force_update']:
            yakey_backup = YaKeyBackup().parse(response['yakey_backup'])
            with UPDATE(yakey_backup, self.request.env, {}):
                yakey_backup.backup = self.form_values['backup']
                yakey_backup.updated = datetime.now()
                yakey_backup.device_name = self.form_values.get('device_name')
        else:
            raise YaKeyBackupAlreadyExistsError()


class BackupDownloadView(BackupBaseView):
    """
    Второй шаг выгрузки бэкапа секретов Ключа из базы Паспорта на телефон пользователя.
    Если бэкап существует, отдает бэкап из базы.
    """
    step = 'download'

    def _process(self):
        response = self.blackbox.yakey_backup(self.number.digital)

        if response['status'] == BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS:
            raise YaKeyBackupNotFoundError()
        self.response_values['backup'] = response['yakey_backup']['backup']
        self.response_values['backup_info'] = {
            'updated': response['yakey_backup']['updated'],
            'device_name': response['yakey_backup']['device_name'],
        }


class BackupInfoView(BackupBaseView):
    """
    Получение метаниформации о бэкапе
    """
    basic_form = forms.BaseNumberedForm()

    def _process(self):
        response = self.blackbox.yakey_backup(self.number.digital, meta=True)

        if response['status'] == BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS:
            raise YaKeyBackupNotFoundError()
        self.response_values['backup_info'] = {
            'updated': response['yakey_backup']['updated'],
            'device_name': response['yakey_backup']['device_name'],
        }
