# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.processes import PROCESS_YAKEY_BACKUP
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    CodeInvalidError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAssertCaptchaMixin,
    BundlePhoneMixin,
    BundleTrackedPhoneConfirmationMixin,
    TrackRestricter,
)
from passport.backend.api.views.bundle.phone import forms as phone_forms
from passport.backend.api.views.bundle.phone.exceptions import (
    SendingLimitExceededError,
    SmsNotSentError,
)
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.core.counters.sms_per_phone import get_per_phone_buckets
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property

from .forms import ConfirmSubmitForm


log = logging.getLogger(__name__)


YAKEY_BACKUP_MODE = 'yakey_backup'
YAKEY_BACKUP_GRANT = 'allow_yakey_backup'


class BaseBackupPhoneConfirmationView(BaseBundleView, BundlePhoneMixin):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    process_name = PROCESS_YAKEY_BACKUP
    require_process = True
    required_grants = [YAKEY_BACKUP_GRANT]

    track_type = 'register'

    mode = YAKEY_BACKUP_MODE
    step = None

    number = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.mode,
            step=self.step,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )


class BackupRestricter(TrackRestricter):
    def __init__(self, *args):
        super(BackupRestricter, self).__init__(*args)
        self._config.update(
            phone_number=False,
        )

    def restrict_phone_number(self):
        self._config['phone_number'] = True

    def check(self):
        super(BackupRestricter, self).check()
        counter = get_per_phone_buckets()

        if counter.hit_limit(self._phone_number.digital):
            self._statbox.stash(error='sms_limit.exceeded')
            log.debug('Unable to send code because of phone number limit')
            raise SendingLimitExceededError()

    def update(self):
        super(BackupRestricter, self).update()
        get_per_phone_buckets().incr(self._phone_number.digital)


class BackupSendSmsView(
    BaseBackupPhoneConfirmationView,
    BundleAssertCaptchaMixin,
    BundleTrackedPhoneConfirmationMixin,
):

    basic_form = ConfirmSubmitForm
    step = 'send_code'

    @cached_property
    def restricter(self):
        rest = BackupRestricter(
            self.confirmation_info,
            self.request.env,
            self.statbox,
            self.consumer,
            self.number,
            self.track,
        )
        rest.restrict_ip()
        rest.restrict_rate()
        rest.restrict_reconfirmation()
        rest.restrict_phone_number()
        return rest

    def process_request(self):
        self.process_basic_form()
        self.read_or_create_track(self.track_type, process_name=self.process_name)

        self.number = self.form_values['number']
        self.statbox.bind_context(
            track_id=self.track_id,
            number=self.number.masked_format_for_statbox,
        )
        with self.track_transaction.commit_on_error():
            self.response_values['track_id'] = self.track.track_id
            self.response_values['number'] = dump_number(self.number)

            # Проверяем, что номер, на который уже высылали код, не поменялся
            if (self.track.phone_confirmation_phone_number_original and
                    self.track.phone_confirmation_phone_number_original != self.number.original):
                self.confirmation_info.reset_phone()
                self.confirmation_info.save()

            self.track.phone_confirmation_phone_number_original = self.number.original
            self.track.country = self.form_values['country']
            self.track.display_language = self.form_values['display_language']

            try:
                self.send_code(
                    self.number,
                    restricter=self.restricter,
                    language=self.form_values['display_language'],
                )
            except:
                self.statbox.dump_stashes()
                raise

        self.statbox.log(status='ok')


class BackupCheckSmsView(
    BaseBackupPhoneConfirmationView,
    BundleAssertCaptchaMixin,
    BundleTrackedPhoneConfirmationMixin,
):
    require_track = True
    basic_form = phone_forms.ConfirmCommitForm
    step = 'check_code'

    def process_request(self):
        self.process_basic_form()
        self.statbox.bind_context(
            track_id=self.track_id,
        )
        self.response_values['track_id'] = self.track_id
        self.read_track()

        original_number = self.track.phone_confirmation_phone_number_original
        if not original_number:
            raise InvalidTrackStateError()
        self.number = PhoneNumber.parse(original_number, self.track.country)
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)

        with self.track_transaction.commit_on_error():
            self.confirm_code(self.form_values['code'])

        self.statbox.log(status='ok')

    def confirm_code(self, code):
        if not self.track.phone_confirmation_sms_count.get(default=0):
            raise SmsNotSentError()

        if self.is_phone_confirmed_in_track():
            raise ActionNotRequiredError()

        try:
            super(BackupCheckSmsView, self).confirm_code(code)
        except CodeInvalidError:
            self.statbox.log(
                action=self.statbox.Link('confirm_phone'),
                error='code.invalid',
                code_checks_count=self.track.phone_confirmation_confirms_count.get(default=0),
            )
            raise
