# -*- coding: utf-8 -*-
import logging
import time

from flask import request
from passport.backend.api import forms
from passport.backend.api.common import ip
from passport.backend.api.common.decorators import (
    headers_required,
    validate,
)
from passport.backend.api.common.format_response import (
    format_error,
    ok_response,
)
from passport.backend.api.common.login import allow_special_test_yandex_login_registration
from passport.backend.api.exceptions import EmptyCaptchaKeyError
from passport.backend.api.views.bundle.constants import CHANGE_PASSWORD_REASON_HACKED
from passport.backend.core.builders.captcha import get_captcha
from passport.backend.core.conf import settings
from passport.backend.core.counters import registration_karma
from passport.backend.core.counters.change_password_counter import get_per_user_ip_buckets
from passport.backend.core.logging_utils.loggers.statbox import (
    StatboxLogger,
    to_statbox,
)
from passport.backend.core.tracks.track_manager import TrackManager

from .grants import grants


log = logging.getLogger('passport.api.views')


def is_captcha_valid(track):
    # Капча еще не создавалась - невалидна
    if not track.captcha_generate_count.get() or track.captcha_generated_at is None:
        return False
    # Сгенерированная капча уже проверялась - невалидна
    if track.is_captcha_checked:
        return False
    # В треке закеширован урл капчи?
    return bool(track.captcha_image_url)


@validate(forms.CaptchaGenerateForm())
@grants(['captcha'])
def captcha_generate(args):
    language = args['display_language']
    country = args['country']
    checks = args['checks']
    voice = args['voice']
    scale_factor = args['scale_factor']
    use_cached = args['use_cached']
    track_id = args['track_id']
    image_type = args['type']

    statbox = StatboxLogger(
        mode='captcha_generate',
        track_id=track_id,
    )
    response = {}
    if voice:
        if language and language not in settings.CAPTCHA_VOICE_LANGUAGE_MATCHING.keys():
            voice = False
            response.update({
                'voice': {
                    'warnings': [format_error(
                        'unsupportedvoicecaptchalanguage',
                        'Unsupported language=%s for voice captcha' % language,
                    )],
                },
            })

        if country and country not in settings.CAPTCHA_VOICE_COUNTRY_MATCHING.keys():
            voice = False
            response.update({
                'voice': {
                    'warnings': [format_error(
                        'unsupportedvoicecaptchacountry',
                        'Unsupported country=%s for voice captcha' % country,
                    )],
                },
            })

    if track_id:
        with TrackManager().transaction(track_id).rollback_on_error() as track:
            statbox.bind_context(uid=track.uid)

            if use_cached and is_captcha_valid(track):
                # возвращаем закешированную в треке капчу
                response.update({'image_url': track.captcha_image_url, 'key': track.captcha_key})
                if voice:
                    response.update({
                        'voice': {
                            'url': track.captcha_voice_url,
                            'intro_url': track.captcha_voice_intro_url,
                        },
                    })
                    statbox.log(
                        key=track.captcha_key,
                        cached=True,
                    )
                return ok_response(response)

    captcha_result = get_captcha().generate(
        language=language,
        country=country,
        image_type=image_type,
        checks=checks,
        https=True,
        voice=voice,
        scale_factor=scale_factor,
        request_id=request.env.request_id,
    )

    if track_id:
        with TrackManager().transaction(track_id).rollback_on_error() as track:
            track.captcha_key = captcha_result.key
            track.image_captcha_type = captcha_result.image_captcha.type
            track.captcha_image_url = captcha_result.image_captcha.url
            if voice and captcha_result.voice_captcha.type:
                track.voice_captcha_type = captcha_result.voice_captcha.type
                track.captcha_voice_url = captcha_result.voice_captcha.url
                track.captcha_voice_intro_url = captcha_result.voice_captcha.intro_url
            track.captcha_generate_count.incr()
            track.captcha_generated_at = time.time()
            # Временный костыль, PASSP-6126
            if track.is_captcha_checked:
                track.is_captcha_checked = False
            if track.is_captcha_recognized:
                track.is_captcha_recognized = False

    response.update({'image_url': captcha_result.image_captcha.url, 'key': captcha_result.key})
    if voice:
        response.update({
            'voice': {
                'url': captcha_result.voice_captcha.url,
                'intro_url': captcha_result.voice_captcha.intro_url,
            },
        })
    statbox.log(
        key=captcha_result.key,
        image_type=captcha_result.image_captcha.type,
        voice_type=captcha_result.voice_captcha.type,
    )
    return ok_response(response)


@validate(forms.CaptchaCheckForm())
@headers_required('Ya-Consumer-Client-Ip')
@grants(['captcha'])
def captcha_check(args):
    answer = args.get('answer')
    key = args.get('key')
    track_id = args['track_id']

    captcha = get_captcha()

    uid = None

    with TrackManager().transaction(track_id).rollback_on_error() as track:
        key = key or track.captcha_key
        if not key:
            raise EmptyCaptchaKeyError('No captcha key')

        track.is_captcha_checked = True
        login = track.login or ''
        uid = track.uid

        # Для интеграционного тестирования
        if allow_special_test_yandex_login_registration(login, request.env.user_ip):
            is_recognized = True
            reason = 'test_login'
        else:
            registration_karma_status = registration_karma.get_status(request.env.user_ip)
            is_user_ip_bad = (
                ip.is_ip_blacklisted(request.env.user_ip) and
                track.track_type in settings.DISABLED_FOR_BLACK_AS_CAPTCHA_TRACK_TYPES
            )

            # TODO: отправлять как-то информацию о таких запросах во фродо
            if registration_karma_status.is_bad:
                log.info('User IP %s considered bad. Ignoring received captcha answer %s for key %s.',
                         request.env.user_ip, answer, key)
                is_recognized = False
                reason = 'bad_registration_karma'
            elif is_user_ip_bad:
                log.info('User AS %s considered bad. Ignoring received captcha answer %s for key %s.',
                         request.env.user_ip, answer, key)
                is_recognized = False
                reason = 'bad_registration_as'
            else:
                is_recognized = captcha.check(
                    answer=answer,
                    key=key,
                    request_id=request.env.request_id,
                )
                reason = 'check'

            if registration_karma_status is not registration_karma.WHITE:
                to_statbox({
                    'action': 'captcha_check',
                    'track_id': track_id,
                    'onliner_status': registration_karma_status.status,
                })

            # Имитация неправильно введенной капчи при превышении количества
            # принудительных смен пароля с одного IP по причине взлома
            if (track.track_type == 'authorize' and
                    track.change_password_reason == CHANGE_PASSWORD_REASON_HACKED):
                user_ip = request.env.user_ip
                per_ip_counter = get_per_user_ip_buckets()
                if per_ip_counter.hit_limit_by_ip(user_ip):
                    is_recognized = False
                    reason = 'hacked_change_attempts_exceeded'
                    to_statbox({
                        'action': 'hacked_change_attempts_exceeded',
                        'track_id': track_id,
                        'user_ip': user_ip,
                        'attempts': per_ip_counter.get(user_ip),
                    })

        if is_recognized:
            track.is_captcha_recognized = True
        track.captcha_check_count.incr()
        track.captcha_checked_at = time.time()

    to_statbox({
        'mode': 'captcha_check',
        'action': 'captcha_check',  # оставлен для совместимости со старыми потребителями
        'track_id': track_id,
        'key': key,
        'is_recognized': is_recognized,
        'reason': reason,
        'uid': uid,
    })

    return ok_response(correct=is_recognized)


__all__ = (
    'captcha_generate',
    'captcha_check',
)
