# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.authorization import (
    authorize_oauth,
    is_oauth_token_created,
)
from passport.backend.api.common.common import get_logout_datetime
from passport.backend.api.common.decorators import (
    headers_required,
    validate,
)
from passport.backend.api.common.format_response import ok_response
from passport.backend.api.exceptions import (
    AuthorizationNotAllowedError,
    OAuthTokenAlreadyCreatedError,
)
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.utils.time import get_unixtime

from .grants import grants


log = logging.getLogger('passport.api.views')


@validate(forms.OAuthTokenCreate())
@headers_required('Ya-Consumer-Client-Ip')
@grants(['oauth.token_create'])
def create_oauth_token(args):
    track_id = args['track_id']
    with TrackManager().transaction(track_id).rollback_on_error() as track:
        if not track.allow_oauth_authorization:
            raise AuthorizationNotAllowedError('Authorization is not allowed for track: %s' % track_id)

        if is_oauth_token_created(track):
            raise OAuthTokenAlreadyCreatedError('OAuth token already created for track: %s' % track_id)

        from passport.backend.api.views.bundle.mixins.common import BundleDeviceInfoMixin
        thing = BundleDeviceInfoMixin()
        thing.track = track
        device_info = thing.track_to_oauth_params(thing.get_device_params_from_track())

        response = authorize_oauth(
            client_id=args['client_id'],
            client_secret=args['client_secret'],
            track=track,
            env=request.env,
            logout_datetime=get_logout_datetime(track.uid),
            **device_info
        )

        if 'error' not in response:
            track.oauth_token_created_at = track.oauth_token_created_at or get_unixtime()
            to_statbox({
                'action': 'oauth_token_created',
                'client_id': args['client_id'],
                'ip': request.env.user_ip,
                'track_id': track_id,
                'uid': track.uid,
            })

        return ok_response(
            oauth=response,
            sensitive_fields=['oauth.access_token'],
        )


__all__ = ('create_oauth_token',)
