# -*- coding: utf-8 -*-
from functools import partial
import logging
import random
import time

from flask import request
from passport.backend.api import (
    exceptions,
    forms,
)
from passport.backend.api.common.decorators import (
    headers_required,
    validate,
)
from passport.backend.api.common.format_response import ok_response
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.common.phone_number import sanitize_phone_number
from passport.backend.api.common.processes import is_process_allowed
from passport.backend.api.common.yasms import (
    build_route_advice,
    generate_fake_global_sms_id,
    log_sms_not_delivered,
)
from passport.backend.api.yasms.api import (
    SaveSimplePhone,
    Yasms,
)
from passport.backend.core.builders import (
    blackbox,
    yasms,
)
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    sms_per_ip,
    sms_per_ip_for_consumer,
)
from passport.backend.core.logging_utils.loggers.statbox import (
    StatboxLogger,
    to_statbox,
)
from passport.backend.core.models.account import Account
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.types.account.account import ACCOUNT_TYPE_PHONISH
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.utils.common import normalize_code

from .grants import grants


log = logging.getLogger('passport.api.views')

PHONE_CONFIRMATION_STATE = 'phone_confirmation'


@validate(forms.PhoneConfirmation())
@headers_required('Ya-Consumer-Client-Ip')
@grants(['phone_number.confirm'])
def phone_confirmation_send_code(args):
    track_id = args['track_id']
    language = args['display_language']
    exception = None

    with TrackManager().transaction(track_id).commit_on_error() as track:
        try:
            phone_number = sanitize_phone_number(track, args['phone_number'],
                                                 args['country'], args['ignore_phone_compare'])
            if not is_process_allowed([], track=track):
                raise exceptions.InvalidTrackStateError()
            if not track.state:
                track.state = PHONE_CONFIRMATION_STATE

            # Проверяем, что стэйт трэка правильный
            if track.state != PHONE_CONFIRMATION_STATE:
                raise exceptions.InvalidTrackStateError()

            if track.phone_confirmation_is_confirmed:
                raise exceptions.PhoneAlreadyConfirmedError()

            user_ip = request.env.user_ip
            consumer = args['consumer']

            log_sms_not_delivered_error = partial(
                log_sms_not_delivered,
                client_ip=user_ip,
                user_agent=request.env.user_agent,
                number=phone_number,
                global_sms_id=generate_fake_global_sms_id(),
                caller=consumer,
            )

            if sms_per_ip_for_consumer.exists(consumer):
                if sms_per_ip_for_consumer.get_counter(consumer).hit_limit_by_ip(user_ip):
                    log.info('Sms limit per ip "%s", and consumer "%s" exceeded' % (user_ip, consumer))
                    track.phone_confirmation_send_ip_limit_reached = True
                    log_sms_not_delivered_error(reason='limits')
                    raise exceptions.SmsSendLimitExceededError()
            elif sms_per_ip.get_counter(user_ip).hit_limit_by_ip(user_ip):
                log.info('Sms limit per ip "%s" exceeded', user_ip)
                track.phone_confirmation_send_ip_limit_reached = True
                log_sms_not_delivered_error(reason='limits')
                raise exceptions.SmsSendLimitExceededError()

            sms_count = track.phone_confirmation_sms_count.get(default=0)
            if sms_count >= settings.SMS_VALIDATION_MAX_SMS_COUNT:
                log.info('Sms limit per track exceeded')
                track.phone_confirmation_send_count_limit_reached = True
                log_sms_not_delivered_error(reason='limits')
                raise exceptions.SmsSendLimitExceededError()

            if sms_count > 0 and track.phone_confirmation_last_send_at:
                """
                >>> t1 = 1377195511.966000
                >>> t2 = 1377195511.967000
                >>> t2 - float(str(t1)) < 0  # Как будто мы пишем в redis и читаем из редиса
                <<< True  # Поэтому мы используем abs()
                """
                last_send_at = float(track.phone_confirmation_last_send_at)
                if abs(time.time() - last_send_at) < settings.SMS_VALIDATION_RESEND_TIMEOUT:
                    raise exceptions.SmsSendTooEarlyError()

            yasms_client = yasms.get_yasms()

            old_phone_number = track.phone_confirmation_phone_number
            if old_phone_number is None or PhoneNumber.parse(old_phone_number) != phone_number:
                code_length = settings.SMS_LEGACY_VALIDATION_CODE_LENGTH
                code = str(random.SystemRandom().randrange(10 ** (code_length - 1), 10 ** code_length))

                track.phone_confirmation_is_confirmed = False

                track.phone_confirmation_code = code
                track.phone_confirmation_phone_number = phone_number.e164
                track.phone_confirmation_phone_number_original = phone_number.original

                track.phone_confirmation_confirms_count.reset()

            response = yasms_client.send_sms(
                phone_number.e164,
                settings.translations.SMS[language]['APPROVE_CODE'],
                text_template_params={'code': track.phone_confirmation_code},
                caller=consumer,
                identity='send_confirmation_code',
                route=build_route_advice(phone_number, consumer, track),
                user_agent=request.env.user_agent,
                client_ip=request.env.user_ip,
            )
            sms_id = response['id']

            now = time.time()

            if not track.phone_confirmation_first_send_at:
                track.phone_confirmation_first_send_at = now
            track.phone_confirmation_last_send_at = now

            track.phone_confirmation_sms_count.incr()
            if sms_per_ip_for_consumer.exists(consumer):
                sms_per_ip_for_consumer.get_counter(consumer).incr(user_ip)
            else:
                sms_per_ip.get_counter(user_ip).incr(user_ip)

            to_statbox({
                'action': 'send_code',
                'track_id': track_id,
                'uid': '-',
                'number': phone_number.masked_format_for_statbox,
                'sms_count': track.phone_confirmation_sms_count.get(default=0),
                'sms_id': sms_id,
            })

        except exceptions.SmsSendLimitExceededError as ex:
            exception = ex

    if exception:
        raise exception
    return ok_response(code_length=len(str(track.phone_confirmation_code)))


@validate(forms.RequiredTrackedConsumerForm())
@grants(['phone_number.confirm'])
def phone_confirmation_can_resend(args):
    track_id = args['track_id']

    with TrackManager().transaction(track_id).rollback_on_error() as track:
        sms_count = track.phone_confirmation_sms_count.get(default=0)
        is_resent_forbidden = (
            sms_count == 0 or
            sms_count >= settings.SMS_VALIDATION_MAX_SMS_COUNT or
            track.phone_confirmation_is_confirmed
        )
        if is_resent_forbidden:
            return ok_response(result=False)

        if track.phone_confirmation_last_send_at:
            last_send_at = float(track.phone_confirmation_last_send_at)
            result = time.time() - last_send_at > settings.SMS_VALIDATION_RESEND_TIMEOUT
            return ok_response(result=result)

        return ok_response(result=True)


@validate(forms.ConfirmPhone())
@grants(['phone_number.confirm'])
def phone_confirmation_confirm(args):
    track_id = args['track_id']
    code = args['code']

    exception = None
    with TrackManager().transaction(track_id).rollback_on_error() as track:
        try:
            if not is_process_allowed([], track=track):
                raise exceptions.InvalidTrackStateError()
            # FIXME: удалить после выкладки в прод.
            # Сделано, чтобы не сломать текущее поведение
            if not track.state:
                track.state = PHONE_CONFIRMATION_STATE

            # Проверяем, что стэйт трэка правильный
            if track.state != PHONE_CONFIRMATION_STATE:
                raise exceptions.InvalidTrackStateError()

            sms_count = track.phone_confirmation_sms_count.get(default=0)
            if sms_count == 0:
                raise exceptions.SmsNotSentError()
            if track.phone_confirmation_is_confirmed:
                raise exceptions.PhoneAlreadyConfirmedError()
            confirms_count = track.phone_confirmation_confirms_count.get(default=0)
            if confirms_count >= settings.SMS_VALIDATION_MAX_CHECKS_COUNT:
                track.phone_confirmation_confirms_count_limit_reached = True
                raise exceptions.ConfirmationsLimitExceededError()

            now = time.time()
            if track.phone_confirmation_first_checked is None:
                track.phone_confirmation_first_checked = now
            result = normalize_code(track.phone_confirmation_code) == normalize_code(code)
            track.phone_confirmation_confirms_count.incr()
            track.phone_confirmation_last_checked = now
            if result:
                track.phone_confirmation_is_confirmed = True

            time_passed = None
            if track.phone_confirmation_last_send_at:
                time_passed = int(now - float(track.phone_confirmation_last_send_at))

            to_statbox({
                'action': 'enter_code',
                'track_id': track_id,
                'number': PhoneNumber.parse(track.phone_confirmation_phone_number).masked_format_for_statbox,
                'uid': '-',
                'time_passed': time_passed,
                'good': result,
            })
        except exceptions.ConfirmationsLimitExceededError as ex:
            exception = ex

    if exception:
        raise exception
    return ok_response(result=result)


@validate(forms.PhoneCopy())
@grants(['phone_number.copy'])
def phone_copy(args):
    def userphones(account, yasms_client):
        return dict((phone_info['number'], phone_info)
                    for phone_info in yasms_client.userphones(account))

    src_token = args['src_oauth_token']
    dst_token = args['dst_oauth_token']
    phone_number = args['phone_number']

    bb = blackbox.get_blackbox()

    bb_kwargs = add_phone_arguments()

    src_response = bb.oauth(src_token, **bb_kwargs)
    if src_response['status'] != blackbox.BLACKBOX_OAUTH_VALID_STATUS:
        raise exceptions.InvalidSrcOAuthTokenError()

    dst_response = bb.oauth(dst_token, **bb_kwargs)
    if dst_response['status'] != blackbox.BLACKBOX_OAUTH_VALID_STATUS:
        raise exceptions.InvalidDstOAuthTokenError()

    src_account = Account().parse(src_response)
    dst_account = Account().parse(dst_response)
    setup_log_prefix(src_account)
    if src_account.uid == dst_account.uid:
        raise exceptions.SrcAndDstUidEqualError()

    yasms_client = Yasms(bb, yasms.get_yasms(), request.env)
    phone_number_e164 = phone_number.e164

    # Проверяем есть ли этот телефон у src-пользователя
    src_phones = userphones(src_account, yasms_client)
    if phone_number_e164 not in src_phones:
        raise exceptions.PhoneNotFoundError()
    if src_phones[phone_number_e164]['valid'] != 'valid':
        raise exceptions.PhoneNotConfirmedError()

    # Проверяем что номер уже не подтвержден у dst-пользователя
    dst_phones = userphones(dst_account, yasms_client)
    if (phone_number_e164 in dst_phones and
            dst_phones[phone_number_e164]['valid'] == 'valid'):
        raise exceptions.PhoneAlreadyConfirmedError()

    # Т.к. к фонишу нельзя привязать больше одного номера, нужно проверить что
    # на фонише ещё нет номеров (хоть это и невозможно сейчас).
    if dst_account.is_phonish and dst_account.phones.bound():
        raise exceptions.PhoneAlreadyConfirmedError()

    # ignore_bindlimit автоматически проставляется для PHONISH
    ignore_bindlimit = dst_account.type == ACCOUNT_TYPE_PHONISH

    statbox = StatboxLogger(
        mode='phone_copy',
        uid=dst_account.uid,
        login=dst_account.normalized_login,
        src_uid=src_account.uid,
    )

    save_simple_phone = SaveSimplePhone(
        account=dst_account,
        phone_number=phone_number,
        consumer=args['consumer'],
        env=request.env,
        statbox=statbox,
        blackbox=bb,
        yasms=yasms_client,
        should_ignore_binding_limit=ignore_bindlimit,
    )
    save_simple_phone.submit()
    with UPDATE(dst_account, request.env, {'action': 'copy_phone', 'consumer': args['consumer']}):
        save_simple_phone.commit()
    save_simple_phone.after_commit()

    log.debug(
        'YaSms register phone_number=%s for uid=%s (copy from uid=%s)',
        phone_number_e164,
        dst_account.uid,
        src_account.uid,
    )

    return ok_response()


__all__ = (
    'phone_confirmation_send_code',
    'phone_confirmation_can_resend',
    'phone_confirmation_confirm',
    'phone_copy',
    'PHONE_CONFIRMATION_STATE',
)
