# -*- coding: utf-8 -*-
import json
import logging
import os

from flask import (
    make_response,
    request,
)
from passport.backend.api.common.format_response import (
    error_response,
    format_error,
)
from passport.backend.core.builders import blackbox
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.manager import (
    get_db_errors,
    get_dbm,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.redis_manager.redis_manager import get_redis_mgr
from passport.backend.core.useragent.sync import (
    RequestError,
    UserAgent,
)
from passport.backend.utils.string import smart_text
from redis import RedisError


log = logging.getLogger('passport.api.views.ping')


def ping():
    def error(code, text):
        error_list.append(format_error(code, text))
        log.error(text)
    check = [value.strip().lower() for value in request.values.to_dict().get('check', '').split(',')]
    error_list = []  # список ошибок, которые отдадим наружу в ответе

    # Проверяем доступность файла. Если его нет или он недоступен - значит, мы должны закрыться от балансера
    if not os.access(settings.PING_FILE, os.R_OK):
        error('BackendUnavailable', u'Server is down')

    if 'db' in check:
        # Пингуем только базы из разрешённого списка
        # Пингуем только мастера
        for dbname in settings.PING_DB_NAMES:
            engine = get_dbm(dbname).get_engine(force_master=True)
            connect_url = engine.url
            allowed_exceptions = get_db_errors()
            try:
                log.debug(u'Checking database availability: "%s" (%s)' % (connect_url.database,
                                                                          connect_url.host))
                # connect выполняется в обход dbmanager.safe_execute,
                # поэтому надо явно ловить все возможные исключения
                conn = engine.connect()

                # Делаем запрос, чтобы Алхимия реально сходила в СУБД
                conn.execute('select 1')

                # Возвращаем соединение в пул
                conn.close()
            except allowed_exceptions:
                error('DatabaseUnavailable',
                      u'Database is unavailable: "%s" (%s)' % (connect_url.database,
                                                               connect_url.host))
    if 'blackbox' in check:
        try:
            bbox = blackbox.Blackbox(retries=settings.BLACKBOX_AVAILABILITY_TEST_RETRIES)
            log.debug(u'Checking blackbox availability: "%s", retries="%d", uid="%s"' % (bbox.url,
                                                                                         settings.BLACKBOX_AVAILABILITY_TEST_RETRIES,
                                                                                         settings.BLACKBOX_AVAILABILITY_TEST_UID))

            data = bbox.userinfo(uid=settings.BLACKBOX_AVAILABILITY_TEST_UID)
            if not data.get('uid'):
                log.error(u'Blackbox is available, but there is no test user with uid "%s" in instance "%s"' % (settings.BLACKBOX_AVAILABILITY_TEST_UID,
                                                                                                                bbox.url))
        except (blackbox.BaseBlackboxError, RequestError):
            error('BlackboxUnavailable', u'Blackbox is unavailable: failed to get userinfo by uid "%s" from instance "%s"' % (settings.BLACKBOX_AVAILABILITY_TEST_UID,
                                                                                                                              bbox.url))
    # пингуем фронтенд
    if 'frontend' in check:
        useragent = UserAgent()
        try:
            log.debug(u'Checking frontend availability')
            response = useragent.get(settings.FRONTEND_HOST + '/ping', timeout=settings.FRONTEND_PING_TIMEOUT)
            if response.content.lower().strip() != 'pong':
                error('FrontendNoPong', u'Frontend at %s is unavailable: response is not Pong - %s' % (settings.FRONTEND_HOST, smart_text(response.content)))
        except RequestError as e:
            error('FrontendUnavailable', u'Frontend at %s is unavailable: %s' % (settings.FRONTEND_HOST, smart_text(e)))
    # пингуем redis
    if 'redis' in check:
        try:
            log.debug(u'Checking local redis availability')
            current_host_id = get_current_host().get_id()
            r = get_redis_mgr(current_host_id)
            if not r.ping():
                error('RedisUnavailable', u'Redis ping response is not True at %s' % current_host_id)
        except RedisError as e:
            error('RedisUnavailable', u'Redis at %s is unavailable: %s' % (current_host_id, smart_text(e)))
    # пингуем лингво-апи
    if 'lingvo' in check:
        useragent = UserAgent()
        try:
            log.debug(u'Checking lingvo api availability')
            response = useragent.post(settings.FIO_SUGGEST_API_URL + 'passport?facts=passport',
                                      data=u'Иванов Иван'.encode('utf-8'),
                                      timeout=settings.LINGVO_API_PING_TIMEOUT).content
            fio = json.loads(response)['passport'][0]
            fio = fio['BasicFio']
            assert fio['FirstName'] == u'Иван'
            assert fio['LastName'] == u'Иванов'
        except (KeyError, AssertionError):
                error('LingvoApiBadResponse',
                      u'Lingvo api at %s is unavailable: response is in incorrect format - %s' % (settings.FIO_SUGGEST_API_URL, smart_text(response)))
        except RequestError as e:
            error('LingvoApiUnavailable', u'Lingvo api at %s is unavailable: %s' % (settings.FIO_SUGGEST_API_URL, smart_text(e)))

    if error_list:
        return error_response(503, errors=error_list)
    # Такой ответ сделан специально для балансера.
    # Надо отдавать 200 и Pong в теле, и никак иначе. Иначе балансер будет считать, что мы лежим.
    return make_response('Pong\n', 200, {'Content-Type': 'text/plain'})


__all__ = ('ping',)
