# -*- coding: utf-8 -*-

from datetime import datetime
import json
import logging

from flask import request
from passport.backend.api import forms
from passport.backend.api.common import ip
from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
    set_password_with_experiment,
    unsubscribe_from_maillists_if_nessesary,
)
from passport.backend.api.common.authorization import (
    authorize_oauth,
    SessionScope,
    set_authorization_track_fields,
)
from passport.backend.api.common.common import (
    check_spammer,
    CleanWebChecker,
    get_email_domain,
    validate_password,
)
from passport.backend.api.common.decorators import (
    headers_required,
    static_statbox,
    validate,
)
from passport.backend.api.common.format_response import (
    error_response,
    ok_response,
)
from passport.backend.api.common.login import build_available_phonish_login
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.common.pdd import does_domain_belong_to_pdd
from passport.backend.api.common.phonish import get_phonish_namespace
from passport.backend.api.common.processes import (
    is_process_allowed,
    PROCESS_WEB_REGISTRATION,
)
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.exceptions import (
    AccountAlreadyCompletedError,
    AccountDisabledError,
    DomainInvalidTypeError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    InvalidTrackTypeError,
    OAuthTokenNotFoundError,
    PhoneNotConfirmedError,
    RegistrationAlreadyCompletedError,
    RegistrationSmsSendPerIPLimitExceededError,
    UnknownAlternativeRegistrationTypeError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleLastAuthMixin,
    BundlePhoneMixin,
)
from passport.backend.api.yasms.api import (
    SaveSecurePhone,
    SaveSimplePhone,
    Yasms,
)
from passport.backend.core import validators
from passport.backend.core.builders import (
    blackbox,
    historydb_api,
    yasms,
)
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    registration_karma,
    sms_per_ip,
    sms_per_phone,
    uncompleted_registration_captcha,
)
from passport.backend.core.grants.grants_config import check_ip_counters_always_full
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    MailInfo,
    render_to_sendmail,
)
from passport.backend.core.models.account import Account
from passport.backend.core.models.email import Email
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.types.email.email import get_default_native_emails
from passport.backend.core.types.phone_number.phone_number import (
    parse_phone_number,
    PhoneNumber,
)
from passport.backend.core.types.question import (
    Question,
    USER_QUESTION_ID,
)
from passport.backend.utils.time import unixtime_to_datetime

from .grants import grants


log = logging.getLogger('passport.api.views')

MOBILEPROXY_CONSUMER = 'mobileproxy'


def send_plus_promo_email(account, plus_promo_code):
    if plus_promo_code and len(plus_promo_code) > settings.PLUS_PROMO_CODE_MAX_LENGTH:
        log.debug('plus_promo_code is too long: {}; Will not send email'.format(len(plus_promo_code)))
        return

    address = '{}@{}'.format(account.login, get_email_domain(request.env.host))
    tld = 'ru'

    info = MailInfo(
        subject=u'А мы не с пустыми руками! 3 месяца Яндекс.Плюс в подарок 🎁',
        from_=(u'Яндекс.Плюс', 'hello@plus.yandex.ru'),
        tld=tld,
        reply_to=('', 'hello@plus.yandex.ru'),
    )
    render_to_sendmail(
        'mail/email_plus_promo.html',
        info=info,
        recipients=[address],
        context={'promocode': plus_promo_code},
        is_plain_text=False,
    )


def statbox_registration_sms_per_ip_limit_exceeded(env, track, counter, mode):
    statbox = StatboxLogger(mode=mode)
    statbox.log(
        action='registration_with_sms',
        error='registration_sms_per_ip_limit_has_exceeded',
        track_id=track.track_id,
        ip=env.user_ip,
        counter_prefix=counter.prefix,
        counter_current_value=counter.get(env.user_ip),
        counter_limit_value=counter.limit,
        is_special_testing_ip=check_ip_counters_always_full(env.user_ip),
    )


def statbox_registration_sms_per_phone_limit_exceeded(phone_number, track, counter, mode, is_phonish):
    statbox = StatboxLogger(mode=mode)
    statbox.log(
        action='registration_with_sms',
        error='registration_sms_per_phone_limit_has_exceeded',
        track_id=track.track_id,
        is_phonish=is_phonish,
        counter_prefix=counter.prefix,
        counter_current_value=counter.get(phone_number),
        counter_limit_value=counter.limit,
    )


def is_registration_sms_per_ip_counter_exceed(env, track, mode=None):
    """
    Проверяем значение счетчика "телефонных" регистраций с одного ip-адреса,
    если счетчик превысил предел, пишем в statbox
    """
    counter = sms_per_ip.get_registration_completed_with_phone_counter(user_ip=env.user_ip)
    exceed = counter.hit_limit_by_ip(env.user_ip)
    if exceed:
        if not track.phone_confirmation_send_ip_limit_reached:
            # Установка флажка в треке происходит во вложенном context-manager'е потому,
            # что переданный track не сохранится при выходе из внешнего context_manager'а по исключению.
            # Не перезаписываем track, чтобы избежать нежелательных сайд-эффектов
            with TrackManager().transaction(track.track_id, allow_nested=True).rollback_on_error() as track_instance:
                track_instance.phone_confirmation_send_ip_limit_reached = True

        statbox_registration_sms_per_ip_limit_exceeded(request.env, track, counter, mode)

    return exceed


def is_registration_sms_per_phone_counter_exceed(phone_number, track, mode=None, is_phonish=False):
    """
    Проверяем значение счетчика "телефонных" регистраций с одного телефона,
    если счетчик превысил предел, пишем в statbox
    """
    phone_number = parse_phone_number(phone_number).e164
    counter = sms_per_phone.get_per_phone_on_registration_buckets(is_phonish=is_phonish)
    exceed = counter.hit_limit(phone_number)
    if exceed:
        if not track.phone_confirmation_send_ip_limit_reached:
            with TrackManager().transaction(track.track_id, allow_nested=True).rollback_on_error() as track_instance:
                track_instance.phone_confirmation_send_ip_limit_reached = True

        statbox_registration_sms_per_phone_limit_exceeded(phone_number, track, counter, mode, is_phonish)

    return exceed


def incr_registration_sms_per_ip_counter(env):
    """Увеличиваем значение счетчика "телефонных" регистраций с одного ip-адреса"""
    sms_per_ip.get_registration_completed_with_phone_counter(user_ip=env.user_ip).incr(env.user_ip)


def incr_registration_sms_per_phone_counter(phone_number, is_phonish=False):
    """Увеличиваем значение счетчика "телефонных" регистраций с одного ip-адреса"""
    phone_number = parse_phone_number(phone_number).e164
    sms_per_phone.get_per_phone_on_registration_buckets(is_phonish=is_phonish).incr(phone_number)


def validate_password_on_registration(args, track, allow_phone_validation):
    is_phone_confirmed = None
    if allow_phone_validation:
        is_phone_confirmed = track.phone_confirmation_is_confirmed

    password_args = dict(args)
    password_args['emails'] = get_default_native_emails(args['login'])
    if is_phone_confirmed:
        # Если телефон был подтвержден в процессе регистрации - используем его для валидации пароля
        password_args['phone_number'] = track.phone_confirmation_phone_number

    validated, _ = validate_password(password_args, track)
    args['quality'] = validated['quality']
    return args


def common_portal_registration(args,
                               allow_phone_validation,
                               allow_captcha_validation,
                               require_confirmed_email,
                               mode,
                               verify_password_with_phone=True,
                               alias_type='portal',
                               with_phonenumber_alias=False,
                               plus_promo_code=None):
    """
    Функция для обобщенной регистрации.

    TODO: Порефакторить этого динозавра и разбить на составные части.

    @param args: Параметры формы, неиспользуемые параметры, должны быть отфильтрованы
    @param allow_phone_validation: Разрешена верификация пользователя с помощью телефона
    @param allow_captcha_validation: Разрешена верификация пользователя с помощью капчи
    @param require_confirmed_email: Требуется верификация пользователя с помощью письма на эл. почту
    @param mode: Регистрирующий режим
    @param verify_password_with_phone: Требуется проверить несовпадение пароля и телефона
    @param alias_type: Тип алиаса, с которым надо создать аккаунт.
    @return: Response
    """

    track_id = args['track_id']
    with TrackManager().transaction(track_id).rollback_on_error() as track:

        args = validate_password_on_registration(args, track, verify_password_with_phone)

        if track.is_successful_registered:
            raise RegistrationAlreadyCompletedError()

        eula_accepted = args['eula_accepted']

        if not eula_accepted:
            raise EulaIsNotAcceptedError()

        email = None
        if require_confirmed_email:
            # Требуется подтверждённый почтовый адрес, но он не был подтверждён
            if not track.email_confirmation_passed_at:
                raise UserNotVerifiedError()
            # не пустой email_confirmation_address гарантируется во вьюхе account_register_require_confirmed_email
            email = Email(
                address=track.email_confirmation_address,
                is_unsafe=False,
                confirmed_at=unixtime_to_datetime(track.email_confirmation_passed_at),
            )

        is_phone_confirmed = None

        if allow_phone_validation:
            is_phone_confirmed = track.phone_confirmation_is_confirmed
            # PASSP-9817 если телефон не валидировали,
            # не нужно делать проверки sms-счетчиков
            if is_phone_confirmed:
                if (
                    is_registration_sms_per_ip_counter_exceed(request.env, track, mode=mode) or
                    is_registration_sms_per_phone_counter_exceed(
                        track.phone_confirmation_phone_number,
                        track,
                        mode=mode,
                    )
                ):
                    raise RegistrationSmsSendPerIPLimitExceededError()

        is_captcha_passed = None
        if allow_captcha_validation:
            is_captcha_passed = track.is_captcha_recognized

        if not is_phone_confirmed and not is_captcha_passed:
            raise UserNotVerifiedError()

        question = None
        if allow_captcha_validation:
            if args['hint_question']:
                question = Question(text=args['hint_question'])
            else:
                question = Question.from_id(args['display_language'], args['hint_question_id'])

        # Пытаемся взять телефон из формы или из трека, если он уже подтвержден
        phone_number = None
        if is_phone_confirmed:
            phone_number = track.phone_confirmation_phone_number

        # http://wiki.yandex-team.ru/passport/statbox#chtopishem
        statbox = StatboxLogger(
            mode=mode,
            track_id=args['track_id'],
            user_agent=request.env.user_agent,
            ip=request.env.user_ip,
            consumer=args['consumer'],
        )
        CleanWebChecker().check_form_values(args, statbox=statbox)

        timestamp = datetime.now()

        account = default_account(
            args['login'],
            timestamp,
            args,
            build_default_person_registration_info(request.env.user_ip),
            alias_type=alias_type,
        )

        if phone_number is not None:
            save_secure_phone = SaveSecurePhone(
                account=account,
                is_new_account=True,
                phone_number=PhoneNumber.parse(phone_number),
                consumer=args['consumer'],
                env=request.env,
                statbox=statbox,
                blackbox=blackbox.get_blackbox(),
                yasms=Yasms(blackbox.get_blackbox(), yasms.get_yasms(), request.env),
                phone_confirmation_datetime=timestamp,
                aliasify=with_phonenumber_alias,
                allow_to_take_busy_alias_from_any_account=args.get('allow_to_take_busy_alias', False),
                enable_search_alias=False,
                check_account_type_on_submit=False,
            )
            save_secure_phone.submit()

        with CREATE(
            account,
            request.env,
            {'action': 'account_register', 'consumer': args['consumer']},
            datetime_=timestamp,
        ) as account:
            if alias_type == 'portal':
                add_subscription(account, get_service(slug='mail'))
            if email:
                email.bound_at = timestamp
                email.created_at = timestamp
                account.emails.add(email)

            if question:
                account.hint.question = question
                account.hint.answer = args['hint_answer']

            if phone_number is not None:
                save_secure_phone.commit()

            unsubscribe_from_maillists_if_nessesary(
                account=account,
                form_values=args,
                track=track,
            )

        if phone_number is not None:
            save_secure_phone.after_commit()

        track.is_successful_registered = True
        set_authorization_track_fields(
            account,
            track,
            allow_create_session=True,
            allow_create_token=True,
            password_passed=True,
            session_scope=SessionScope.xsession,
        )
        track.user_entered_login = args['login']
        setup_log_prefix(account)

        frodo_args = dict(x for x in args.items() if x[1] is not None)
        frodo_args['action'] = mode

        account, _ = check_spammer(
            account,
            request.env,
            frodo_args,
            track,
            args['consumer'],
        )

        if account.karma.suffix == 100:
            registration_karma.incr_bad(request.env.user_ip)
        elif account.karma.value == 0:
            registration_karma.incr_good(request.env.user_ip)

        statbox.bind_context(
            uid=account.uid,
            login=account.normalized_login,
        )
        statbox.bind(
            action='account_created',
            karma=account.karma.value,
            password_quality=args['quality'],
            is_suggested_login=account.login in track.suggested_logins.get(),
            suggest_generation_number=track.suggest_login_count.get(default=0),
            country=args['country'],
            retpath=track.retpath,
        )

        if allow_captcha_validation:
            # Разрешена валидация пользователя через captcha
            statbox.bind(
                captcha_generation_number=track.captcha_generate_count.get(default=0),
                question_id=args['hint_question_id'] or USER_QUESTION_ID,
                is_voice_generated=bool(track.voice_captcha_type),
                captcha_key=track.captcha_key,
            )

        if plus_promo_code:
            send_plus_promo_email(account, plus_promo_code)
            statbox.bind(with_plus_promo_code='1')

        statbox.log()

        # Только для телефонных регистраций увеличиваем счетчик СМС на ip-адрес
        if phone_number is not None:
            incr_registration_sms_per_ip_counter(request.env)
            incr_registration_sms_per_phone_counter(phone_number)

        process_env_profile(account, track=track)

        return ok_response(uid=account.uid)


@static_statbox({'action': 'submitted'})
@headers_required('Ya-Client-User-Agent', 'Ya-Consumer-Client-Ip')
@validate(forms.ConsumerForm(filter_extra_fields=False))
@grants(['account.register_require_confirmed_phone'])
@validate(forms.AccountRegisterRequireConfirmedPhone())
def account_register_require_confirmed_phone(args):
    return common_portal_registration(
        args,
        allow_phone_validation=True,
        allow_captcha_validation=False,
        require_confirmed_email=False,
        mode='phonereg',
    )


@static_statbox({'action': 'submitted', 'mode': 'register_by_email'})
@headers_required('Ya-Client-User-Agent', 'Ya-Consumer-Client-Ip')
@validate(forms.AccountRegisterSelectAlternative(filter_extra_fields=False))
@grants(['account.register_require_confirmed_email'])
def account_register_require_confirmed_email(args):
    validation_method = args['validation_method']
    allow_phone_validation = validation_method == forms.PHONE_VALIDATION_METHOD
    allow_captcha_validation = validation_method == forms.CAPTCHA_VALIDATION_METHOD

    if validation_method == forms.PHONE_VALIDATION_METHOD:
        form = forms.AccountRegisterRequireConfirmedPhoneAndEmail
        mode = 'register_by_email_with_phone'
    elif validation_method == forms.CAPTCHA_VALIDATION_METHOD:
        form = forms.AccountRegisterWithHintAndEmail
        mode = 'register_by_email_with_hint'
    else:
        raise UnknownAlternativeRegistrationTypeError()

    args = form().to_python(args, validators.State(request.env))

    if not is_process_allowed(
        allowed_processes=[PROCESS_WEB_REGISTRATION],
        is_process_required=True,
        track_id=args['track_id'],
    ):
        return error_response(400, 'track.invalid_state')

    track = TrackManager().read(args['track_id'])
    login = args['login']
    if track.email_confirmation_address != login:
        raise InvalidTrackStateError()

    if does_domain_belong_to_pdd(login):
        raise DomainInvalidTypeError()

    return common_portal_registration(
        args,
        allow_phone_validation=allow_phone_validation,
        allow_captcha_validation=allow_captcha_validation,
        require_confirmed_email=True,
        alias_type='lite',
        mode=mode,
    )


@static_statbox({'action': 'submitted', 'mode': 'alternative'})
@headers_required('Ya-Client-User-Agent', 'Ya-Consumer-Client-Ip')
@validate(forms.AccountRegisterSelectAlternative(filter_extra_fields=False))
@grants(['account.register_alternative'])
def account_register_alternative(args):
    validation_method = args['validation_method']
    allow_phone_validation = validation_method == forms.PHONE_VALIDATION_METHOD
    allow_captcha_validation = validation_method == forms.CAPTCHA_VALIDATION_METHOD

    if validation_method == forms.PHONE_VALIDATION_METHOD:
        form = forms.AccountRegisterAlternativeWithConfirmedPhone
        mode = 'alternative_phone'
    elif validation_method == forms.CAPTCHA_VALIDATION_METHOD:
        form = forms.AccountRegisterAlternativeWithHint
        mode = 'alternative_hint'
    else:
        raise UnknownAlternativeRegistrationTypeError()

    args = form().to_python(args, validators.State(request.env))
    return common_portal_registration(
        args,
        allow_phone_validation=allow_phone_validation,
        allow_captcha_validation=allow_captcha_validation,
        require_confirmed_email=False,
        mode=mode,
        # PASSP-15801, для АМ разрешаем не проверять совпадение пароля и телефона
        verify_password_with_phone=args['consumer'] != MOBILEPROXY_CONSUMER,
        with_phonenumber_alias=True,
        plus_promo_code=args['plus_promo_code'],
    )


@static_statbox({'action': 'submitted', 'mode': 'account_register_phonish'})
@headers_required('Ya-Client-User-Agent', 'Ya-Consumer-Client-Ip')
@validate(forms.AccountRegisterPhonishForm())
@grants(['account.register_phonish'])
def account_register_phonish(args):
    """
    Создание телефонного аккаунта.

    Документация
    http://wiki.yandex-team.ru/passport/backend/api/frontend#parkovkiregistracijaizmobilnogotelefona
    """
    return _RegisterPhonish(args).process_request()


class _RegisterPhonish(BundleAccountGetterMixin, BundleLastAuthMixin, BundlePhoneMixin):
    def __init__(self, args):
        self._args = args
        self._track_id = args['track_id']
        self._consumer = args['consumer']
        self._app_id = args['app_id']
        self._statbox = StatboxLogger(
            mode='account_register_phonish',
            track_id=self._track_id,
            user_agent=request.env.user_agent,
            ip=request.env.user_ip,
            consumer=self._consumer,
        )

    def process_request(self):
        if ip.is_ip_blacklisted(request.env.user_ip):
            log.debug('Phonish creation attempt from blacklisted ip %s', request.env.user_ip)
            raise RegistrationSmsSendPerIPLimitExceededError()

        with TrackManager().transaction(self._track_id).rollback_on_error() as track:
            self.track = track

            if track.is_successful_registered:
                raise RegistrationAlreadyCompletedError()

            if not self.is_phone_confirmed_in_track():
                raise UserNotVerifiedError()

            phone_number = PhoneNumber.parse(track.phone_confirmation_phone_number)

            phonish_namespace = get_phonish_namespace(app_id=self._app_id)
            account, phone_number = self.get_account_by_phone_number(
                phone_number,
                phonish_namespace=phonish_namespace,
            )

            if account is not None:
                is_new_account = False
                self._set_account_to_logging_context(account)

                if not account.is_enabled:
                    raise AccountDisabledError()

                # СИБ: если видим, что отдаём тот же Phonish-аккаунт, то
                # делаем global logout, чтобы отозвать токены со старого
                # устройства (токены должны жить только на 1 устройстве).
                #
                # Правильно не отзывать токены полученные на данное устройство.
                #
                # Такое поведение не подходит дорегистрированным фонишам. Т.к.
                # на 1 устройстве фониш может входить СМСкой, а на других
                # паролем. Здесь было правильно отзывать только те токены с
                # других устройств, которые были получены через вход по СМС.
                need_logout = not account.portal_alias

                with UPDATE(
                    account,
                    request.env,
                    {
                        'action': 'login_phonish',
                        'consumer': self._consumer,
                    },
                ):
                    if need_logout:
                        account.global_logout_datetime = datetime.now()

                    phone = account.phones.by_number(phone_number)
                    phone.confirm()

                if need_logout:
                    # Чтобы с треком можно было работать дальше, нужно сообщить,
                    # что global logout был сделан в нём.
                    track.is_web_sessions_logout = True

                self._statbox.log(
                    action='login_phonish',
                    phone_number=phone_number.masked_format_for_statbox,
                    phone_id=account.phones.by_number(phone_number).id,
                )
            else:
                is_new_account = True
                if (
                    is_registration_sms_per_ip_counter_exceed(
                        request.env,
                        track,
                        mode='account_register_phonish',
                    ) or
                    is_registration_sms_per_phone_counter_exceed(
                        phone_number,
                        track,
                        mode='account_register_phonish',
                        is_phonish=True,
                    )
                ):
                    raise RegistrationSmsSendPerIPLimitExceededError()
                account = self._create_account(phone_number, track, namespace=phonish_namespace)
                incr_registration_sms_per_ip_counter(request.env)
                incr_registration_sms_per_phone_counter(phone_number, is_phonish=True)

            track.is_successful_registered = True

            set_authorization_track_fields(
                account,
                track,
                allow_create_session=False,
                allow_create_token=True,
                password_passed=True,
            )

            process_env_profile(account, track=self.track)

            return ok_response(uid=account.uid, is_new_account=is_new_account)

    def _set_account_to_logging_context(self, account):
        setup_log_prefix(account)
        self._statbox.bind_context(
            uid=account.uid,
            login=account.normalized_login,
        )

    def _create_account(self, phone_number, track, namespace=None):
        login = build_available_phonish_login(
            settings.PHONISH_LOGIN_GENERATION_RETRIES,
            request.env,
        )

        # Если пользователь заводит фониша используя старый номер, для которого
        # уже действует новый, то создаём фониша с новым номером, для того чтобы
        # прекратить появление новых фонишей со старыми номерами.
        phone_number = PhoneNumber.from_deprecated(phone_number)

        with CREATE(
            default_account(
                login,
                datetime.now(),
                self._args,
                build_default_person_registration_info(request.env.user_ip),
                alias_type='phonish',
            ),
            request.env,
            {
                'action': 'account_register',
                'consumer': self._consumer,
            },
        ) as account:
            add_subscription(
                account,
                service=get_service(slug='phonish'),
                login=login,
            )
            save_simple_phone = SaveSimplePhone(
                account=account,
                phone_number=phone_number,
                consumer=self._consumer,
                env=request.env,
                statbox=self._statbox,
                blackbox=self.blackbox,
                yasms=self.yasms_api,
                is_new_account=True,
                should_ignore_binding_limit=True,
            )
            save_simple_phone.submit()
            save_simple_phone.commit()
            account.phonish_namespace = namespace

        self._set_account_to_logging_context(account)

        self._statbox.log(
            action='account_created',
            country=account.person.country,
            retpath=track.retpath,
        )

        save_simple_phone.after_commit()

        return account

    @property
    def blackbox(self):
        return blackbox.get_blackbox()

    @property
    def yasms_api(self):
        return Yasms(self.blackbox, yasms.get_yasms(), request.env),

    @property
    def historydb_api(self):
        return historydb_api.get_historydb_api()


@static_statbox({'action': 'submitted', 'mode': 'uncompleted'})
@headers_required('Ya-Client-User-Agent', 'Ya-Consumer-Client-Ip')
@validate(forms.AccountRegisterUncompleted())
@grants(['account.register_uncompleted'])
def account_register_uncompleted(args):
    """
    Первый этап регистрации от ритэйлеров
    Документация: http://wiki.yandex-team.ru/passport/backend/api/frontend#vyzovynajetapei
    """
    statbox = StatboxLogger(
        action='account_created',
        mode='uncompleted',
        track_id=args['track_id'],
        country=args['country'],
    )
    CleanWebChecker().check_form_values(args, statbox=statbox)
    with TrackManager().transaction(args['track_id']).rollback_on_error() as track:

        if track.is_successful_registered:
            raise RegistrationAlreadyCompletedError()

        captcha_required = uncompleted_registration_captcha.is_required(request.env.user_ip)
        captcha_passed = track.is_captcha_recognized
        if captcha_required and not captcha_passed:
            raise UserNotVerifiedError()

        now = datetime.now()
        with CREATE(default_account(args['login'], now, args, build_default_person_registration_info(request.env.user_ip)),
                    request.env, {'action': 'account_register',
                                  'consumer': args['consumer']}) as account:
            pass

        # Авторизация для данного пользователя запрещена
        track.is_successful_registered = True
        track.allow_authorization = False
        track.uid = account.uid
        track.login = account.login
        track.have_password = False
        if account.person.language:
            track.language = account.person.language

        setup_log_prefix(account)
        statbox.log(**{
            'uid': account.uid,
            'login': account.normalized_login,
            'karma': account.karma.value,
            'user_agent': request.env.user_agent,
            'is_suggested_login': account.login in track.suggested_logins.get(),
            'suggest_generation_number': track.suggest_login_count.get(default=0),
            'is_voice_generated': bool(track.voice_captcha_type),
            'captcha_key': track.captcha_key,
            'retpath': track.retpath,
        })

        oauth_application = settings.OAUTH_APPLICATION_TOKEN_TO_TRACK
        client_id = oauth_application['client_id']
        client_secret = oauth_application['client_secret']
        oauth_response = authorize_oauth(
            client_id=client_id,
            client_secret=client_secret,
            track=track,
            env=request.env,
        )

        if 'access_token' not in oauth_response:
            raise OAuthTokenNotFoundError('Missing field access_token in OAuth response')

        return ok_response(
            access_token=oauth_response['access_token'],
            sensitive_fields=['access_token'],
        )


@static_statbox({'action': 'submitted', 'mode': 'uncompleted'})
@headers_required('Ya-Client-User-Agent', 'Ya-Consumer-Client-Ip')
@validate(forms.SetPasswordUncompleted())
@grants(['account.uncompleted_set_password'])
def account_uncompleted_set_password(args):
    """
    Второй этап регистрации от ритэйлеров
    Документация http://wiki.yandex-team.ru/passport/backend/api/frontend#vyzovynajetapeii
    """
    track_id = args['track_id']

    eula_accepted = args['eula_accepted']

    if not eula_accepted:
        raise EulaIsNotAcceptedError()

    with TrackManager().transaction(track_id).rollback_on_error() as track:
        args['emails'] = get_default_native_emails(track.login)
        validate_password(args, track)

        if track.track_type != 'complete':
            raise InvalidTrackTypeError('Invalid track type "%s".' % track.track_type)

        if track.uid is None:
            raise InvalidTrackStateError('Uid is None')

        if track.is_successful_completed:
            raise AccountAlreadyCompletedError('Account already completed')

        # Проверяем что телефон подтвержден
        if not track.phone_confirmation_is_confirmed:
            raise PhoneNotConfirmedError('Phone is not confirmed')
        if track.phone_confirmation_phone_number is None:
            raise InvalidTrackStateError('Phone number is None')

        phone_number = PhoneNumber.parse(track.phone_confirmation_phone_number)
        if (
            is_registration_sms_per_ip_counter_exceed(request.env, track, mode='uncompleted') or
            is_registration_sms_per_phone_counter_exceed(phone_number, track, mode='uncompleted')
        ):
            raise RegistrationSmsSendPerIPLimitExceededError()

        blackbox_kwargs = dict(uid=track.uid)
        blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)

        data = blackbox.get_blackbox().userinfo(**blackbox_kwargs)

        account = Account().parse(data)
        setup_log_prefix(account)

        statbox = StatboxLogger(
            mode='uncompleted',
            track_id=track_id,
            uid=account.uid,
            login=account.normalized_login,
        )

        save_secure_phone = SaveSecurePhone(
            account=account,
            phone_number=phone_number,
            consumer=args['consumer'],
            env=request.env,
            statbox=statbox,
            blackbox=blackbox.get_blackbox(),
            yasms=Yasms(blackbox.get_blackbox(), yasms.get_yasms(), request.env),
            check_account_type_on_submit=False,
        )
        save_secure_phone.submit()

        events = {
            'action': 'change_password',
            'consumer': args['consumer'],
        }
        with UPDATE(account, request.env, events):
            set_password_with_experiment(
                account,
                args['password'],
                args['quality'],
                uid=account.uid,
            )
            add_subscription(account, get_service(slug='mail'))
            save_secure_phone.commit()

        statbox.log(
            action='uncompleted_set_password',
            password_quality=args['quality'],
        )

        save_secure_phone.after_commit()

        frodo_args = dict(x for x in args.items() if x[1] is not None)
        frodo_args.update(
            action='uncompleted_set_password',
            login=account.login,
            firstname=account.person.firstname,
            lastname=account.person.lastname,
            language=account.person.language,
            country=account.person.country,
            phone_number=phone_number,
        )
        account, _ = check_spammer(
            account,
            request.env,
            frodo_args,
            track,
            args['consumer'],
        )

        track.is_successful_completed = True
        set_authorization_track_fields(
            account,
            track,
            allow_create_session=True,
            allow_create_token=True,
            password_passed=True,
            session_scope=SessionScope.xsession,
        )

        incr_registration_sms_per_ip_counter(request.env)
        incr_registration_sms_per_phone_counter(phone_number)

        process_env_profile(account, track=track)

    return ok_response()


__all__ = (
    'is_registration_sms_per_ip_counter_exceed',
    'is_registration_sms_per_phone_counter_exceed',
    'incr_registration_sms_per_ip_counter',
    'incr_registration_sms_per_phone_counter',
    'account_register_require_confirmed_phone',
    'account_register_require_confirmed_email',
    'account_register_alternative',
    'account_register_phonish',
    'account_register_uncompleted',
    'account_uncompleted_set_password',
)
