# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.authorization import (
    build_auth_cookies_and_session,
    is_session_created,
    is_session_valid,
)
from passport.backend.api.common.common import (
    get_logout_datetime,
    track_to_response,
)
from passport.backend.api.common.decorators import (
    headers_required,
    validate,
)
from passport.backend.api.common.format_response import (
    error_response,
    ok_response,
)
from passport.backend.api.exceptions import (
    AuthorizationNotAllowedError,
    SessionAlreadyCreatedError,
    SessionOverflowError,
)
from passport.backend.core.builders import blackbox
from passport.backend.core.crypto.utils import compare_strings
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.types.account.account import ACCOUNT_TYPE_NORMAL
from passport.backend.core.types.display_name import DisplayName

from .grants import grants


log = logging.getLogger('passport.api.views')


@validate(forms.RequiredTrackedConsumerForm())
@headers_required('Ya-Client-Host', 'Ya-Consumer-Client-Ip')
@grants(['session.create'])
def create_session(args):
    track_id = args['track_id']
    with TrackManager().transaction(track_id).rollback_on_error() as track:

        if not track.allow_authorization:
            raise AuthorizationNotAllowedError('Authorization is not allowed for track: %s' % track.track_id)

        if is_session_created(track):
            raise SessionAlreadyCreatedError('Session already created for track: %s' % track.track_id)

        extend_session = False
        multi_session_users = None
        # TODO: нужно ли ТРЕБОВАТЬ заголовок Ya-Client-Cookie (может сломать обратную совместимость)?
        if request.environ.get('HTTP_YA_CLIENT_COOKIE') is None:
            log.warn('Called create_session without Ya-Client-Cookie header')

        sessionid = request.env.cookies.get('Session_id')
        ssl_sessionid = request.env.cookies.get('sessionid2')
        sessguard = request.env.cookies.get('sessguard')
        if sessionid:
            track.old_session = sessionid
            track.old_ssl_session = ssl_sessionid
            track.old_sessguard = sessguard
            to_statbox({
                'mode': 'check_cookies',
                'host': request.env.host,
                'consumer': args['consumer'],
                'have_sessguard': sessguard is not None,
                'sessionid': mask_sessionid(sessionid),
            })
            bb_response = blackbox.get_blackbox().sessionid(
                sessionid=sessionid,
                ip=request.env.user_ip,
                host=request.env.host,
                sslsessionid=ssl_sessionid,
                sessguard=sessguard,
                multisession=True,
                dbfields=[],
                request_id=request.env.request_id,
            )
            if is_session_valid(bb_response['cookie_status']):
                multi_session_users = bb_response.get('users', {})
                if not bb_response.get('allow_more_users', True) and (
                    int(track.uid) not in multi_session_users
                ):
                    raise SessionOverflowError()
                extend_session = True
                track.old_session_ttl = bb_response['ttl']

        cookies, session, _ = build_auth_cookies_and_session(
            request.env,
            track,
            account_type=ACCOUNT_TYPE_NORMAL,
            extend_session=extend_session,
            multi_session_users=multi_session_users,
            display_name=DisplayName(name=track.login),
            logout_datetime=get_logout_datetime(track.uid),
        )
        # Сохраняем обратную совместимость: ЧЯ скоро перестанет отдавать эти поля
        for cookie_name in ('session', 'sslsession', 'sessguard'):
            if cookie_name in session:
                session[cookie_name].update({
                    'secure': True,
                    'http-only': True,
                })

        sensitive_fields = [
            'cookies',
            'session.value',
            'sslsession.value',
            'sessguard.value',
        ]
        return ok_response(
            cookies=cookies,
            sensitive_fields=sensitive_fields,
            default_uid=int(track.uid),
            **session
        )


@validate(forms.SessionCheck())
@grants(['session.check'])
def check_session(args):
    track = TrackManager().read(args['track_id'])
    session = track.session
    sslsession = track.sslsession
    sessguard = track.sessguard

    if session is None:
        return error_response(400, error_code='invalidTrack',
                              error_message='No session data found in track')

    result = compare_strings(args['session'], session)
    if args['sslsession'] is not None and sslsession is not None:
        sslresult = compare_strings(args['sslsession'], sslsession)
        result = result and sslresult

    if args['sessguard'] is not None and track.sessguard is not None:
        result = result and compare_strings(args['sessguard'], sessguard)

    response = dict(session_is_correct=result)
    response.update(track_to_response(track, ['retpath', 'fretpath', 'clean']))

    if result:
        response.update(session_has_users=session != '')

    return ok_response(**response)


__all__ = ('create_session', 'check_session',)
