# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.decorators import validate
from passport.backend.api.common.format_response import (
    error_response,
    ok_response,
)
from passport.backend.api.common.grants import add_grants_for_service
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.common.mail import is_mail_occupied_by_another_user
from passport.backend.core import Undefined
from passport.backend.core.builders import blackbox
from passport.backend.core.builders.blackbox.utils import user_exists
from passport.backend.core.models.account import Account
from passport.backend.core.models.alias import YandexoidAlias
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.subscription import (
    add_subscription,
    can_be_subscribed,
    delete_subscription as core_delete_subscription,
    SubscriptionImpossibleError,
    SubscriptionNotAllowedError,
    SubscriptionRequiresAccountWithLoginError,
    SubscriptionRequiresAccountWithPasswordError,
    update_subscription,
)

from .grants import grants


log = logging.getLogger('passport.api.views')


@validate(forms.Subscription())
@grants([add_grants_for_service(['create', 'update'])])
def subscription(args):
    uid = args['uid']
    service = args['service']

    bb = blackbox.get_blackbox()
    data = bb.userinfo(uid=uid)

    account = Account().parse(data)
    setup_log_prefix(account)

    # TODO: в реальных условиях у аккаунта всегда должны быть подписки
    # возможно стоит добавить __contains__ в _Undefined?
    if account.subscriptions == Undefined:
        account.subscriptions = {}

    # Пользователь уже подписан
    is_subscribed_already = account.is_subscribed(service)

    if service.slug == 'mail':
        if is_mail_occupied_by_another_user(bb, account):
            return error_response(400, error_code='EmailAlreadyInUse',
                                  error_message='Mailbox is already used by another user')

    if service.sid == 669 and is_subscribed_already:
        if account.yandexoid_alias == YandexoidAlias(account, login=args['yastaff_login']):
            return ok_response()
        return error_response(400, error_code='YastaffLoginInvalid',
                              error_message='Account has another yastaff login already')

    # Ничего не делаем, если пользователь уже подписан и подписка не может обновляться
    elif is_subscribed_already and not service.is_renewable:
        return ok_response()

    try:
        can_be_subscribed(account, service)
    except SubscriptionNotAllowedError:
        return error_response(400, error_code='SubscriptionNotAllowed',
                              error_message='Subscription to sid %d for %s users is not allowed' % (service.sid, 'PDD'))
    except SubscriptionRequiresAccountWithLoginError:
        return error_response(400, error_code='RequiresAccountWithLogin',
                              error_message='Subscription to sid %s requires account with login' % service.sid)
    except SubscriptionRequiresAccountWithPasswordError:
        return error_response(400, error_code='RequiresAccountWithPassword',
                              error_message='Subscription to sid %s requires account with password' % service.sid)
    except SubscriptionImpossibleError:
        return error_response(
            400,
            error_code='SubscriptionImpossible',
            error_message='Subscription to sid %s is impossible' % service.sid
        )

    # Если логин занят на sid=669 - ошибка
    if args['yastaff_login'] and user_exists(args['yastaff_login'], sid=669):
        return error_response(400, error_code='YastaffLoginOccupied',
                              error_message='Yastaff login %s is occupied' % args['yastaff_login'])

    events = {
        'action': 'subscription',
        'consumer': args['consumer'],
        'service': service.slug,
    }

    with UPDATE(account, request.env, events):
        if is_subscribed_already:
            update_subscription(
                account,
                service,
                login_rule=args['login_rule'],
                host_id=args['wmode'],
            )
        else:
            # Костыль для galatasaray
            if service.sid == 61 and not account.is_normal:
                return error_response(
                    400,
                    error_code='SubscriptionImpossible',
                    error_message='Subscription to sid %s is impossible for uid %s' % (
                        service.sid,
                        account.uid,
                    ),
                )
            add_subscription(
                account,
                service,
                login_rule=args['login_rule'],
                login=args['yastaff_login'],
                host_id=args['wmode'],
            )

    return ok_response()


@validate(forms.DeleteSubscription())
@grants([add_grants_for_service(['delete'])])
def delete_subscription(args):
    uid = args['uid']
    service = args['service']

    data = blackbox.get_blackbox().userinfo(uid=uid, force_show_mail_subscription=True)

    account = Account().parse(data)
    setup_log_prefix(account)

    if account.subscriptions == Undefined:
        account.subscriptions = {}

    # У пользователя нет подписки. Нечего отписывать.
    if not account.is_subscribed(service):
        return error_response(404, error_code='NoSubscription',
                              error_message="Subscription with sid %d doesn't exist" % service.sid)

    events = {
        'action': 'delete_subscription',
        'consumer': args['consumer'],
        'service': service.slug,
    }

    with UPDATE(account, request.env, events):
        core_delete_subscription(account, service)

    return ok_response()


__all__ = ('subscription', 'delete_subscription',)
