# -*- coding: utf-8 -*-
import logging
import time

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.common import increment_track_counter
from passport.backend.api.common.decorators import (
    headers_required,
    validate,
)
from passport.backend.api.common.format_response import ok_response
from passport.backend.api.common.suggest import (
    get_countries_suggest,
    get_language_suggest,
    get_login_suggestions,
)
from passport.backend.core import validators
from passport.backend.core.builders.suggest import get_fio_suggest
from passport.backend.core.conf import settings
from passport.backend.core.geobase import Region
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.types.login.login import extract_clean_login_from_email

from .grants import grants


log = logging.getLogger('passport.api.views')


@validate(forms.SuggestLogin())
@grants(['login.suggest'])
def suggest_login(args):
    firstname = args.get('firstname')
    lastname = args.get('lastname')
    original_login = login = args.get('login')
    language = args.get('language')

    # В саджесторе телефонный номер не участвует, но когда-то зачем-то принимался.
    # Продолжаем его принимать, поскольку не хотим менять интерфейс ручки и валидацию.
    phone_number = args.get('phone_number')
    logins = []

    if login:
        login = extract_clean_login_from_email(login)

    logins += get_login_suggestions(
        original_login=original_login,
        firstname=firstname,
        lastname=lastname,
        login=login,
        language=language,
    )

    statbox = StatboxLogger()
    # в tskv логи пишем событие об отсутствии логинов только в случае,
    # если логин был не короче 6 символов
    if not logins and login is not None and len(login) >= 6:
        statbox.bind(
            action='suggest_login_error',
            login=login,
            firstname=firstname,
            lastname=lastname,
            phone_number=phone_number.masked_format_for_statbox if phone_number else None,
        )

    track_id = args.get('track_id')
    if track_id:
        if statbox.has_data:
            statbox.bind(track_id=track_id)

        with TrackManager().transaction(track_id).rollback_on_error() as track:
            if logins:
                track.suggested_logins.append(*logins)
            track.suggest_login_count.incr()
            if track.suggest_login_first_call is None:
                track.suggest_login_first_call = time.time()
            track.suggest_login_last_call = time.time()

    statbox.log()

    return ok_response(logins=logins)


@validate(forms.SuggestName())
@grants(['name.suggest'])
def suggest_name(args):
    firstname, lastname, _ = get_fio_suggest().get(args['name'])

    increment_track_counter(args.get('track_id'), lambda track: track.suggest_name_count)
    return ok_response({
        'firstname': validators.FirstName(if_empty='').to_python(firstname or ''),
        'lastname': validators.LastName(if_empty='').to_python(lastname or ''),
    })


@validate(forms.SuggestGender())
@grants(['gender.suggest'])
def suggest_gender(args):
    name = args['name'] or u"%s %s" % (args['lastname'] or '', args['firstname'] or '')
    _, _, gender = get_fio_suggest().get(name)
    increment_track_counter(args.get('track_id'), lambda track: track.suggest_gender_count)
    return ok_response(gender=gender or 'unknown')


@validate(forms.TrackedConsumerForm())
@grants(['country.suggest'])
def suggest_country(args):
    ret = get_countries_suggest()
    increment_track_counter(args.get('track_id'), lambda track: track.suggest_country_count)
    return ok_response(country=list(ret))


@validate(forms.TrackedConsumerForm())
@headers_required('Ya-Consumer-Client-Ip')
@grants(['timezone.suggest'])
def suggest_timezone(args):

    ret = list()

    user_ip = request.env.user_ip
    timezone = Region(ip=user_ip).timezone
    if timezone is not None:
        ret.append(timezone)

    increment_track_counter(args.get('track_id'), lambda track: track.suggest_timezone_count)
    return ok_response(timezone=list(ret))


@validate(forms.TrackedConsumerForm())
@headers_required('Ya-Client-Host')
@grants(['language.suggest'])
def suggest_language(args):

    try:
        language = get_language_suggest()
    finally:
        increment_track_counter(args.get('track_id'), lambda track: track.suggest_language_count)

    return ok_response(language=language)


@validate(forms.ControlQuestions())
@grants(['control_questions'])
def control_questions(args):
    language = args.get('display_language')
    available_control_questions = settings.translations.QUESTIONS[language]
    allowed_ids = settings.HINT_QUESTION_IDS_FOR_LANGUAGES[language]

    control_questions = [
        {'id': q_id, 'value': available_control_questions[q_id]} for q_id in allowed_ids
    ]
    increment_track_counter(args.get('track_id'), lambda track: track.control_questions_count)
    return ok_response(questions=control_questions)


__all__ = (
    'suggest_login',
    'suggest_name',
    'suggest_gender',
    'suggest_country',
    'suggest_timezone',
    'suggest_language',
    'control_questions',
)
