# -*- coding: utf-8 -*-
from copy import deepcopy
import logging

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.common import parse_args_for_track
from passport.backend.api.common.decorators import validate
from passport.backend.api.common.format_response import ok_response
from passport.backend.api.exceptions import (
    InvalidOAuthScopeError,
    InvalidOAuthTokenError,
)
from passport.backend.core import validators
from passport.backend.core.builders import blackbox
from passport.backend.core.conf import settings
from passport.backend.core.tracks.track_manager import TrackManager

from .grants import grants


log = logging.getLogger('passport.api.views')


@validate(forms.CreateTrackStartForm())
@grants(['track'])
def track_create(args):
    track_type = args['track_type']
    process_name = args['process_name']

    to_track = {}
    if track_type == 'register':
        form_class = forms.CreateRegisterTrackForm
    elif track_type == 'complete':
        form_class = forms.CreateCompleteTrackForm
    else:
        form_class = forms.CreateOtherTrackForm

    args = form_class.to_python(args, validators.State(request.env))

    if track_type == 'complete':
        token = args['token']

        oauth_response = blackbox.get_blackbox().oauth(token, ip=request.env.user_ip)

        if oauth_response['status'] != blackbox.BLACKBOX_OAUTH_VALID_STATUS:
            raise InvalidOAuthTokenError('Invalid oauth token passed')
        if settings.OAUTH_TOKEN_TO_TRACK_SCOPE not in oauth_response['oauth']['scope']:
            raise InvalidOAuthScopeError('Invalid oauth scope passed')

        to_track['uid'] = oauth_response['uid']
        to_track['login'] = oauth_response['login']

    to_track.update(parse_args_for_track(args, exclude=('track_type', 'token')))

    # FIXME: PASSP-11574 Здесь только для совместимости, как только последние
    # потребители этих треков уйдут - выпилить.
    if track_type in settings.DELETED_TRACK_TYPES:
        track_type = 'authorize'

    track_manager = TrackManager()
    track = track_manager.create(track_type, args['consumer'], process_name=process_name)

    if to_track:
        with track_manager.transaction(track.track_id).rollback_on_error() as track:
            track.parse(to_track)
    return ok_response(id=track.track_id)


@validate(forms.RequiredTrackedConsumerForm())
@grants(['track'])
def track_get(args):
    track_manager = TrackManager()
    track_id = args['track_id']

    track = track_manager.read(track_id)
    track_data = deepcopy(track._data)
    for secret_field in track.sensitive_fields:
        if secret_field in track_data:
            track_data[secret_field] = '*****'

    return ok_response(track_data)


@validate(forms.SaveTrackForm())
@grants(['track'])
def track_save(args):
    track_id = args['track_id']
    to_track = parse_args_for_track(args, exclude=('track_id', 'uid'))

    with TrackManager().transaction(track_id).rollback_on_error() as track:
        track.parse(to_track)
    return ok_response()


@validate(forms.RequiredTrackedConsumerForm())
@grants(['track'])
def track_delete(args):
    track_id = args['track_id']
    with TrackManager().transaction(track_id).delete():
        pass
    return ok_response()


__all__ = (
    'track_create',
    'track_get',
    'track_save',
    'track_delete',
)
