# -*- coding: utf-8 -*-
import logging
import time

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.common import (
    increment_track_counter,
    validate_password,
)
from passport.backend.api.common.decorators import (
    headers_required,
    validate,
)
from passport.backend.api.common.format_response import (
    format_error,
    format_errors,
    ok_response,
)
from passport.backend.api.common.grants import add_grants
from passport.backend.api.common.phone_number import sanitize_phone_number
from passport.backend.core import validators
from passport.backend.core.exceptions import ValueChanged
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.validators import State

from .grants import grants


log = logging.getLogger('passport.api.views')


@validate(forms.SimpleLoginValidation())
@headers_required('Ya-Consumer-Client-Ip')
@grants(['login.validate', add_grants(['ignore_stoplist'], if_arg='ignore_stoplist')])
def login_validation(args):
    result = {}
    try:
        forms.LoginValidation().to_python({
            'login': args['login'],
            'ignore_stoplist': args['ignore_stoplist'],
        }, State(request.env))
    except validators.Invalid as e:
        result['validation_errors'] = format_errors(e)

    with TrackManager().transaction(args['track_id']).rollback_on_error() as track:
        track.login_validation_count.incr()
        track.login = args['login']
        if track.login_validation_first_call is None:
            track.login_validation_first_call = time.time()
        track.login_validation_last_call = time.time()

    return ok_response(result)


@validate(forms.SimplePasswordValidation())
@grants(['password.validate'])
def password_validation(args):
    result = {}
    track_id = args.get('track_id')
    track = TrackManager().read(track_id) if track_id else None
    try:
        validated, p = validate_password(args, track)
        if validated.get('lt_middle_quality'):
            result['validation_warnings'] = [format_error(*p.message('weak', None), field='password')]
    except validators.Invalid as e:
        result['validation_errors'] = format_errors(e)

    if track_id:
        with TrackManager().transaction(track_id).rollback_on_error() as track:
            track.password_validation_count.incr()
            if track.password_validation_first_call is None:
                track.password_validation_first_call = time.time()
            track.password_validation_last_call = time.time()

    return ok_response(result)


@validate(forms.SimplePhoneNumberValidation())
@grants(['phone_number.validate'])
def phone_number_validation(args):
    result = {}
    with TrackManager().transaction(args['track_id']).rollback_on_error() as track:
        try:
            sanitized_number = sanitize_phone_number(track, args['phone_number'],
                                                     args['country'], args['ignore_phone_compare'])
            result = {'phone_number': sanitized_number.e164}
        except ValueChanged as e:
            result['validation_errors'] = format_errors(e, field=e.field)
        except validators.Invalid as e:
            result['validation_errors'] = format_errors(e)
        track.phone_number_validation_count.incr()
    return ok_response(result)


@validate(forms.SimpleHintValidation())
@grants(['hint.validate'])
def hint_validation(args):

    result = {}

    def validation_helper(field, validator):
        if args[field] is None:
            return
        try:
            result[field] = validator.to_python(args[field])
        except validators.Invalid as e:
            errors = format_errors(e, field=field)
            result.setdefault('validation_errors', []).extend(errors)

    validation_helper('hint_question', validators.HintQuestion())
    validation_helper('hint_question_id', validators.HintQuestionId())
    validation_helper('hint_answer', validators.HintAnswer(if_missing=None))

    increment_track_counter(args.get('track_id'), lambda track: track.hint_validation_count)
    return ok_response(result, sensitive_fields=['hint_answer'])


@validate(forms.RetPathValidation())
@grants(['retpath.validate'])
def retpath_validation(args):
    result = {}
    track_id = args.get('track_id')
    try:
        retpath = validators.RetPath()
        result = {'retpath': retpath.to_python(args['retpath'])}
        if track_id:
            with TrackManager().transaction(track_id).rollback_on_error() as track:
                track.retpath = result['retpath']
    except validators.Invalid as e:
        result['validation_errors'] = format_errors(e, field='retpath')

    increment_track_counter(track_id, lambda track: track.retpath_validation_count)
    return ok_response(result)


__all__ = (
    'login_validation',
    'password_validation',
    'phone_number_validation',
    'hint_validation',
    'retpath_validation',
)
