# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import (
    datetime,
    timedelta,
)
from functools import partial
import logging
from operator import itemgetter

from passport.backend.core.builders.blackbox.exceptions import (
    BlackboxInvalidResponseError,
    BlackboxTemporaryError,
    BlackboxUnknownError,
)
from passport.backend.core.builders.yasms.utils import (
    accept_non_e164_phone_numbers,
    normalize_phone_number,
)
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers import DummyLogger
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.phones.phones import (
    ConfirmationLimitExceeded,
    NumberBoundAlready,
    OperationInapplicable,
    PhoneConfirmedAlready,
    RemoveSecureOperation,
    ReplaceSecurePhoneWithBoundPhoneOperation,
    ReplaceSecurePhoneWithNonboundPhoneOperation,
    SecureBindOperation,
    SecureBindToSimpleBindError,
    SecureNumberBoundAlready,
    SecurifyOperation,
    SimpleBindOperation,
    SingleSecureOperationError,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.yasms.notifications import (
    notify_about_phone_changes,
    notify_user_by_email_that_secure_phone_bound,
)
from passport.backend.core.yasms.phonenumber_alias import (
    Aliasification,
    PhoneAliasManager,
)
from passport.backend.core.yasms.unbinding import unbind_old_phone
from passport.backend.utils.common import (
    generate_random_code,
    method_decorator,
)

from . import exceptions
from .utils import (
    build_legacy_send_confirmation_code,
    build_send_confirmation_code,
    does_binding_allow_washing,
    get_many_accounts_with_phones_by_uids,
    remove_equal_phone_bindings,
    Restricter,
)


__all__ = (
    u'SaveSecurePhone',
    u'SaveSimplePhone',
    u'Yasms',
)

VALIDATION_STATE = namedtuple(
    u'VALIDATION_STATE',
    u'VALID MESSAGE_SENT DELIVERED',
)._make([u'valid', u'msgsent', u'delivered'])

DELETE_PHONE_STATUS = namedtuple(
    u'DELETE_PHONE_STATUS',
    u'OK STARTED NOT_FOUND INTERNAL_ERROR',
)._make([u'ok', u'started', u'notfound', u'interror'])

DROP_PHONE_STATUS = namedtuple(
    u'DROP_PHONE_STATUS',
    u'OK NOT_FOUND',
)._make([u'ok', u'notfound'])

PROLONG_VALID_STATUS = namedtuple(
    u'PROLONG_VALID_STATUS',
    u'OK NO_PHONE',
)._make([u'ok', u'nophone'])

log = logging.getLogger(u'passport.backend.api.yasms')


def adapt_return_value_to_yasms_builder_form(history):
    """
    Адаптирует ответ phone_bindings_history к форме Я.Смсного билдера.
    """
    history = [
        {
            u'uid': binding[u'uid'],
            u'phone': binding[u'phone_number'].e164,
            u'ts': binding[u'binding_time'],
        }
        for binding in history
    ]
    return {u'status': u'ok', u'history': history}


def userphones_is_phone_confirmed(phone):
    return bool(phone.bound)


def get_validation_state(phone):
    if phone.bound:
        state = VALIDATION_STATE.VALID
    elif is_phone_being_bound(phone):
        state = VALIDATION_STATE.MESSAGE_SENT
    elif phone.confirmed:
        state = VALIDATION_STATE.DELIVERED
    else:
        raise ValueError(u'Unexpected phone state')
    return state


def get_time_of_last_confirmation_code_sending(phone):
    """
    Вычисляет validation_date телефона (время отправки последнего сообщения
    с кодом подтверждения) для ручки userphones.
    """
    if phone.bound:
        time = phone.admitted or phone.confirmed or phone.bound
    elif is_phone_being_bound(phone):
        # Телефон может быть на аккаунте, но не привязанным, а находящимся в
        # процессе привязки. Код подтверждения также может быть ещё не выслан.
        time = phone.operation.code_last_sent or phone.created
    elif phone.confirmed:
        time = phone.admitted or phone.confirmed
    else:
        raise ValueError(u'Unexpected phone state')
    return time


def userphones_get_validations_left(phone):
    if userphones_is_phone_confirmed(phone):
        return 0
    if not is_phone_being_bound(phone):
        raise ValueError(
            u'You can call validations left for confirmed phone number '
            u'or phone number taking part in binding only.'
        )
    return settings.SMS_VALIDATION_MAX_CHECKS_COUNT - phone.operation.code_checks_count


def is_phone_being_bound(phone):
    return phone.operation and phone.operation.type == u'bind'


def check_phone_is_binding_limit_exceeded(current_bindings):
    """
    Вычисляет высказывание "Лимит привязок телефона достигнут".
    """
    bindings_count = sum(
        1 for b in current_bindings
        if not b['should_ignore_binding_limit']
    )
    return bindings_count >= settings.YASMS_PHONE_BINDING_LIMIT


class Yasms(object):
    """
    Предназначен для выполнения высокоуровневых операций с телефонами
    пользователей.
    """
    def __init__(self, blackbox_builder, yasms_builder, env, view=None):
        self._yasms_builder = yasms_builder
        self._blackbox_builder = blackbox_builder
        self._env = env
        self.view = view

    @method_decorator(accept_non_e164_phone_numbers)
    def phone_bindings_history(self, phone_numbers):
        """
        :raises: YaSmsPhoneNumberValueError
        """
        if not phone_numbers:
            raise exceptions.YaSmsPhoneNumberValueError()
        history = self._blackbox_builder.phone_bindings(
            need_current=False,
            need_unbound=False,
            phone_numbers=phone_numbers,
            should_ignore_binding_limit=False,
        )
        history = adapt_return_value_to_yasms_builder_form(history)
        result = {
            u'status': u'ok',
            u'history': remove_equal_phone_bindings(history[u'history']),
        }
        return result

    def register(self, account, phone_number, user_ip, statbox, consumer,
                 language=None, revalidate=False, without_sms=False, user_agent=None,
                 ignore_bindlimit=False, timestamp=None, secure=False):
        """
        Замечание
            Если подпрограмма выбросит исключение, непротиворечивое состояние
            аккаунта не гарантируется.
        """
        if secure and not account.is_password_set_or_promised:
            raise exceptions.YaSmsSecureNumberNotAllowed()

        phone = account.phones.by_number(phone_number)

        if (
            account.is_phonish and
            not (phone and phone.bound) and
            account.phones.bound()
        ):
            raise exceptions.YaSmsPhoneBindingsLimitExceeded()

        if not phone:
            phone = account.phones.create(
                number=phone_number,
                created=datetime.now(),
            )

        code = generate_random_code(settings.SMS_LEGACY_VALIDATION_CODE_LENGTH)
        with statbox.make_context(
            action=statbox.Link(u'register'),
            uid=account.uid,
            ip=self._env.user_ip,
        ), notify_about_phone_changes(
            account=account,
            yasms_builder=self._yasms_builder,
            statbox=statbox,
            consumer=consumer,
            language=language,
            client_ip=self._env.user_ip,
            user_agent=self._env.user_agent,
        ):
            try:
                logical_operation = phone.get_logical_operation(statbox)
                if logical_operation:
                    if secure:
                        logical_operation = logical_operation.to_secure_bind(
                            phone,
                            code,
                            ignore_bindlimit,
                        )
                    else:
                        logical_operation = logical_operation.to_simple_bind(
                            phone,
                            code,
                        )
                else:
                    op_class = SecureBindOperation if secure else SimpleBindOperation
                    logical_operation = op_class.create(
                        phone_manager=account.phones,
                        phone_id=phone.id,
                        code=code,
                        should_ignore_binding_limit=ignore_bindlimit,
                        statbox=statbox,
                    )
            except NumberBoundAlready:
                raise exceptions.YaSmsAlreadyVerified()
            except (
                SingleSecureOperationError,
                SecureNumberBoundAlready,
                SecureBindToSimpleBindError,
            ):
                raise exceptions.YaSmsSecureNumberExists()

            # NOTE: Над номером идёт операция привязки (самостоятельная или как
            # часть составной операции).

            if without_sms:
                # Привязать номер без проверок
                confirmation_info = logical_operation.get_confirmation_info(phone.id)
                with statbox.make_context(without_sms=True):
                    if not confirmation_info.code_confirmed:
                        # Для сохранения обратной совместимости не станем
                        # ругаться, если номер уже подтверждён.
                        if timestamp is not None:
                            confirmation_time = datetime.fromtimestamp(timestamp)
                        else:
                            confirmation_time = datetime.now()
                        logical_operation.confirm_phone(
                            phone.id,
                            logical_operation.get_confirmation_info(phone.id).code_value,
                            timestamp=confirmation_time,
                        )
                    _, changes = logical_operation.apply(
                        need_authenticated_user=False,
                    )

                    if (account.phonenumber_alias.number and
                            account.phonenumber_alias.number.e164 in changes.unbound_numbers):
                        phone_alias = PhoneAliasManager(
                            consumer,
                            environment=self._env,
                            statbox=statbox,
                            need_update_tx=False,
                        )
                        phone_alias.delete_alias(
                            account,
                            phone_alias.ALIAS_DELETE_REASON_OFF,
                        )

                    bindings_history = self.phone_bindings_history([phone.number.e164])
                    if does_binding_allow_washing(
                        account,
                        phone.number.e164,
                        phone.bound,
                        logical_operation.flags.should_ignore_binding_limit,
                        bindings_history,
                    ):
                        # Обеляем учётную запись
                        account.karma.prefix = settings.KARMA_PREFIX_WASHED

                    return {
                        u'id': phone.id,
                        u'number': phone_number,
                        u'uid': account.uid,
                        u'is_revalidated': False,
                    }

            confirmation_info = logical_operation.get_confirmation_info(phone.id)
            send_confirmation_code = build_legacy_send_confirmation_code(
                phone.number.e164,
                confirmation_info,
                account,
                self._env,
                self._yasms_builder,
                consumer,
                statbox,
                language,
            )
            try:
                send_confirmation_code()
            except (
                exceptions.YaSmsPermanentBlock,
                exceptions.YaSmsTemporaryBlock,
            ) as exc:
                statbox.dump_stashes()
                raise exc
            except (
                # Ограничение по номеру телефона
                exceptions.YaSmsLimitExceeded,
                # Ограничение по идентификатору пользователя
                exceptions.YaSmsUidLimitExceeded,
                # Ограничение по IP-адресу
                exceptions.YaSmsIpLimitExceeded,
            ):
                statbox.dump_stashes()
                raise exceptions.YaSmsCodeLimitError()
            except exceptions.YaSmsError:
                statbox.dump_stashes()
                raise exceptions.YaSmsTemporaryError()
            except:
                statbox.dump_stashes()
                raise

            logical_operation.set_confirmation_info(phone.id, confirmation_info)
            is_code_resent = confirmation_info.code_send_count > 1

        return {
            u'id': phone.id,
            u'number': phone_number,
            u'uid': account.uid,
            u'is_revalidated': revalidate and is_code_resent,
        }

    def confirm(self, account, code, phone_number=None, phone_id=None,
                statbox=None, consumer=None, event_timestamp=None, user_ip=None, user_agent=None):
        """
        Исключения
            YaSmsImpossibleConfirm
            YaSmsValueError
            YaSmsCodeLimitError
            OperationExpired
        """
        if phone_id:
            phone = account.phones.by_id(phone_id, assert_exists=False)
        elif phone_number:
            try:
                phone = account.phones.by_number(phone_number)
            except InvalidPhoneNumber:
                # Такого номера не может быть на учётной записи
                raise exceptions.YaSmsImpossibleConfirm()
        else:
            raise exceptions.YaSmsValueError(u'At least one of phone_id or phone_number is needed')

        if not (phone and not phone.bound):
            raise exceptions.YaSmsImpossibleConfirm()

        with statbox.make_context(
            action=statbox.Link(u'confirm'),
            uid=account.uid,
        ), notify_about_phone_changes(
            account=account,
            yasms_builder=self._yasms_builder,
            statbox=statbox,
            consumer=consumer,
            client_ip=self._env.user_ip,
            user_agent=self._env.user_agent,
            view=self.view,
        ):
            logical_op = phone.get_logical_operation(statbox)
            if not (logical_op and type(logical_op) in (SimpleBindOperation, SecureBindOperation)):
                # Ручка предназначена только для привязки простых и защищённых
                # телефонов. С её помощью нельзя исполнить другие операции.
                raise exceptions.YaSmsImpossibleConfirm()

            try:
                is_phone_confirmed, code_checks_left = logical_op.confirm_phone(phone.id, code)
            except ConfirmationLimitExceeded:
                raise exceptions.YaSmsCodeLimitError()
            except PhoneConfirmedAlready:
                raise exceptions.YaSmsImpossibleConfirm()

            if is_phone_confirmed:
                if type(logical_op) is SecureBindOperation and not logical_op.password_verified:
                    raise exceptions.YaSmsImpossibleConfirm()

                logical_op.apply()

                bindings_history = self.phone_bindings_history([phone.number.e164])
                if does_binding_allow_washing(
                    account,
                    phone.number.e164,
                    phone.bound,
                    logical_op.flags.should_ignore_binding_limit,
                    bindings_history,
                ):
                    # Обеляем учётную запись
                    account.karma.prefix = settings.KARMA_PREFIX_WASHED

        if is_phone_confirmed:
            code_checks_left = None
        response = {
            u'id': phone.id,
            u'phone_number': phone.number.e164,
            u'uid': account.uid,
            u'is_valid': is_phone_confirmed,
            u'is_current': phone == account.phones.default,
            u'code_checks_left': code_checks_left,
        }
        return response, logical_op.flags

    def userphones(self, account):
        userphones = []
        for phone in account.phones.all().values():
            if userphones_is_phone_confirmed(phone) or is_phone_being_bound(phone):
                is_phone_secure = phone == account.phones.secure
                logical_op = phone.get_logical_operation(statbox=None)
                is_secure_phone_being_bound = (
                    not phone.bound and
                    logical_op and
                    logical_op.is_binding and
                    # Защищённость логической операции не всегда означает, что
                    # привязывается защищённый номер. Например, в результате
                    # операции замены защищённого номера на привязываемый,
                    # появляется простой номер над которым идёт операция замены
                    # защищённого на простой. Поэтому, чтобы убедиться что
                    # привязывается защищённый номер, смотрим на защищённость
                    # физической операции на телефоне.
                    phone.operation.is_secure
                )
                userphones.append({
                    u'id': int(phone.id),
                    u'number': phone.number.e164,
                    u'active': phone == account.phones.default,
                    u'secure': is_phone_secure or is_secure_phone_being_bound,
                    u'cyrillic': True,

                    u'valid': get_validation_state(phone),
                    u'validation_date': get_time_of_last_confirmation_code_sending(phone),
                    u'validations_left': userphones_get_validations_left(phone),

                    u'autoblocked': False,
                    u'permblocked': False,
                    u'blocked': False,
                })
        return userphones

    def check_phone(self, phone_number, secure_only=True):
        bindings = self._blackbox_builder.phone_bindings(
            phone_numbers=[phone_number],
            need_history=False,
            need_unbound=not secure_only,
        )

        uids = {b[u'uid'] for b in bindings}

        # Получаем сведения об учётных записях. Игнорируем учётные записи,
        # которые уже удалили.
        accounts, _ = get_many_accounts_with_phones_by_uids(uids, self._blackbox_builder)

        items = []
        for account in accounts:
            if not account.phones.has_number(phone_number):
                # Телефон уже отвязали.
                continue
            phone = account.phones.by_number(phone_number)
            if secure_only and phone is not account.phones.secure:
                # Телефон не защищён, такой телефон не должен отдаваться.
                continue
            if not secure_only and not phone.confirmed:
                # Телефон ещё не подтверждён.
                continue
            active = True if secure_only else phone == account.phones.secure  # признак защищённости номера
            items.append({
                u'uid': account.uid,
                u'active': active,
                u'phoneid': phone.id,
                u'phone': phone.number.e164,
                u'valid': get_validation_state(phone),
                u'validation_date': get_time_of_last_confirmation_code_sending(phone),
            })

        current_bindings = bindings if secure_only else [b for b in bindings if b[u'type'] == u'current']

        return {
            u'binding_limit_exceeded': check_phone_is_binding_limit_exceeded(current_bindings),
            u'items': items,
        }

    def prolong_valid(self, account, phone_number):
        try:
            phone = account.phones.by_number(phone_number)
        except InvalidPhoneNumber:
            phone = None

        if not (phone and phone.bound and phone.confirmed):
            return {u'status': PROLONG_VALID_STATUS.NO_PHONE, u'uid': account.uid}

        phone.admitted = datetime.now()
        return {u'status': PROLONG_VALID_STATUS.OK, u'uid': account.uid}

    def drop_phone(self, account, phone_id, statbox, consumer):
        if not account.phones.has_id(phone_id):
            return {u'status': DROP_PHONE_STATUS.NOT_FOUND}
        phone_on_account = account.phones.by_id(phone_id)
        if not (phone_on_account.bound and phone_on_account.confirmed):
            return {u'status': DROP_PHONE_STATUS.NOT_FOUND}

        if account.is_phonish and len(account.phones.bound()) == 1:
            raise exceptions.YaSmsPhoneBindingsLimitExceeded()

        if phone_on_account == account.phones.secure:
            if account.totp_secret.is_set or account.sms_2fa_on:
                raise exceptions.YaSmsPhoneBindingsLimitExceeded()

            if account.is_neophonish:
                raise exceptions.YaSmsPhoneBindingsLimitExceeded()

        if phone_on_account.secured and account.phonenumber_alias.alias:
            phone_alias_manager = PhoneAliasManager(
                consumer,
                environment=None,
                statbox=statbox,
                need_update_tx=False,
            )
            phone_alias_manager.delete_alias(
                account,
                phone_alias_manager.ALIAS_DELETE_REASON_OFF,
            )
        account.phones.remove(phone_on_account)
        return {u'status': DROP_PHONE_STATUS.OK}

    def remove_userphones(self, account, statbox, consumer, block=False, drop_bank_phone=True, unsecure_bank_phone=False):
        """
        Удаляет все телефоны с аккаунта.
        """
        is_2fa_enabled = account.totp_secret and account.totp_secret.is_set
        if account.phones.secure and (is_2fa_enabled or account.sms_2fa_on):
            raise exceptions.YaSmsOperationInapplicable()

        if account.is_phonish or account.is_neophonish:
            raise exceptions.YaSmsPhoneBindingsLimitExceeded()

        if account.phonenumber_alias.alias:
            phone_alias = PhoneAliasManager(
                consumer,
                environment=None,
                statbox=statbox,
                need_update_tx=False,
            )
            phone_alias.delete_alias(account, phone_alias.ALIAS_DELETE_REASON_OFF)

        for phone in account.phones.all().values():
            if phone.is_bank and not drop_bank_phone:
                if phone.secured and unsecure_bank_phone:
                    phone.secured = None
                    account.phones.secure = None
                continue
            account.phones.remove(phone)

        return {u'status': u'ok'}

    def check_user(self, account):
        if account.phones.default is not None:
            user_info = {
                u'uid': account.uid,
                u'has_current_phone': True,
                u'number': account.phones.default.number.e164,

                # Легаси. 14 января 2015 года не используется и для всех новых
                # номеров равняется True.
                u'cyrillic': True,

                # 14 января 2015 года механизм блокировки не работает
                # и признаётся неудачным, планируется его замена, поэтому здесь
                # отдаём всегда False.
                u'blocked': False,
            }
        else:
            user_info = {
                u'uid': account.uid,
                u'has_current_phone': False,
                u'number': None,
                u'cyrillic': None,
                u'blocked': None,
            }
        return user_info

    def delete_phone(self, account, phone_number, user_ip, statbox, consumer, user_agent=None):
        """
        Замечание: после вызова метода нужно вызвать statbox.dump_stashes
        передав туда хотя бы operation_id.
        """
        phone_number = normalize_phone_number(phone_number)
        try:
            phone_number = PhoneNumber.parse(phone_number).e164
        except InvalidPhoneNumber:
            return DELETE_PHONE_STATUS.NOT_FOUND

        if not account.phones.has_number(phone_number):
            return DELETE_PHONE_STATUS.NOT_FOUND

        phone_on_account = account.phones.by_number(phone_number)

        if not (phone_on_account.bound and phone_on_account.confirmed):
            return DELETE_PHONE_STATUS.NOT_FOUND
        if phone_on_account.operation:
            return DELETE_PHONE_STATUS.INTERNAL_ERROR

        if account.is_phonish:
            return DELETE_PHONE_STATUS.INTERNAL_ERROR

        if phone_on_account != account.phones.secure:
            account.phones.remove(phone_on_account)
            return DELETE_PHONE_STATUS.OK

        if account.totp_secret.is_set or account.sms_2fa_on:
            return DELETE_PHONE_STATUS.INTERNAL_ERROR

        if account.is_neophonish:
            return DELETE_PHONE_STATUS.INTERNAL_ERROR

        with statbox.make_context(
            action=statbox.Link(u'delete_phone'),
            uid=account.uid,
        ):
            notification = notify_about_phone_changes(
                account=account,
                yasms_builder=self._yasms_builder,
                statbox=statbox,
                consumer=consumer,
                client_ip=self._env.user_ip,
                user_agent=self._env.user_agent,
                view=self.view,
            )
            notification.start()

            remove_op = RemoveSecureOperation.create(
                phone_manager=account.phones,
                phone_id=phone_on_account.id,
                code=generate_random_code(settings.SMS_LEGACY_VALIDATION_CODE_LENGTH),
                statbox=statbox,
            )
            confirmation_info = remove_op.get_confirmation_info(
                phone_on_account.id,
            )
            restricter = Restricter(
                confirmation_info=confirmation_info,
                consumer=consumer,
                env=self._env,
                phone_number=phone_on_account.number,
                statbox=statbox,
            )
            restricter.restrict_ip()
            send_confirmation_code = build_legacy_send_confirmation_code(
                phone_on_account.number.e164,
                confirmation_info,
                account,
                self._env,
                self._yasms_builder,
                consumer,
                statbox,
                restricter=restricter,
            )
            try:
                send_confirmation_code()
            except exceptions.YaSmsError:
                statbox.dump_stashes()
                return DELETE_PHONE_STATUS.INTERNAL_ERROR
            except:
                statbox.dump_stashes()
                raise
            remove_op.set_confirmation_info(
                phone_on_account.id,
                confirmation_info,
            )

            notification.finish()
        return DELETE_PHONE_STATUS.STARTED

    def validations_number_of_user_phones(self, account, get_account_by_uid):
        """
        Сделать краткую сводку о номерах, привязанных к uid.

        Возвращает список словарей вида
            {
                u'number': номер телефона,
                u'valid': подтверждённость номера пользователем с данным uid,
                u'confirmed_date': дата и время подтверждения, или None,
                u'validations_number': число когда-либо сделанных подтверждений
                                       номера с любых учётных записей,
                u'other_accounts': число других учётных записей для которых номер
                                   сейчас подтверждён,
            }
        """
        uid_phone_state = {}
        user_phones = self.userphones(account)

        if user_phones:
            users_confirmed_phone_number = self.get_validations_info(
                {phone[u'number'] for phone in user_phones},
            )

        ret = []
        for phone in sorted(user_phones, key=itemgetter(u'number')):
            phone_number = phone[u'number']
            validation_state = phone[u'valid']

            if validation_state == u'valid':
                confirmation_date = phone[u'validation_date']
            else:
                confirmation_date = None

            validations_number = (
                users_confirmed_phone_number[phone_number][u'bindings_count']
            )
            if validations_number == 0 and validation_state == u'valid':
                # Фикс случая, когда номера нет в ответе phone_bindings_history,
                # но он есть и подтверждён в ответе user_phones.
                validations_number = 1

            # Подсчёт other_accounts
            other_uids_who_owns_and_confirmed = set()
            all_uids = users_confirmed_phone_number[phone_number][u'uids']

            other_uids = all_uids - {account.uid}
            for other_uid in other_uids:
                if other_uid not in uid_phone_state:
                    # Чтобы не вызывать userphones для одного и того же uid'а
                    # несколько раз, запоминаем ответ.
                    try:
                        other_account = get_account_by_uid(other_uid)
                    except UnknownUid:
                        user_phones = []
                    else:
                        user_phones = self.userphones(other_account)
                    phone_state = {phone[u'number']: phone[u'valid']
                                   for phone in user_phones}
                    uid_phone_state[other_uid] = phone_state
                else:
                    phone_state = uid_phone_state[other_uid]
                if (phone_number in phone_state and
                        phone_state[phone_number] == u'valid'):
                    other_uids_who_owns_and_confirmed.add(other_uid)
            other_accounts_count = len(other_uids_who_owns_and_confirmed)

            ret.append({
                u'number': phone_number,
                u'valid': validation_state,
                u'confirmed_date': confirmation_date,
                u'validations_number': validations_number,
                u'other_accounts': other_accounts_count,
            })
        return ret

    def have_user_once_validated_phone(self, account):
        user_phones = self.userphones(account)

        if not user_phones:
            return {
                u'have_user_once_validated_phone': False,
                u'reason': u'no-phone',
            }

        confirmed_phones = [
            phone for phone in user_phones if phone[u'valid'] == u'valid'
        ]
        if not confirmed_phones:
            return {
                u'have_user_once_validated_phone': False,
                u'reason': u'no-confirmed-phone',
            }

        validations_info = self.get_validations_info(
            {phone[u'number'] for phone in confirmed_phones},
        )

        for phone in validations_info:
            bindings_count = validations_info[phone][u'bindings_count']
            if bindings_count <= settings.YASMS_CLEAN_PHONE_NUMBER_BINDINGS_LIMIT:
                once_validated_phone = phone
                break
        else:
            once_validated_phone = None

        if once_validated_phone is None:
            return {
                u'have_user_once_validated_phone': False,
                u'reason': u'no-quality-confirmed-phone',
            }

        return {
            u'have_user_once_validated_phone': True,
            u'reason': u'ok',
        }

    def get_validations_info(self, phone_numbers):
        """
        Примечание: Этот метод не часть интерфейса перлового Я.Смса.

        Строим словарь, ключи которого данные номера телефонов, а значения
        словарь вида
        {
            u'uids': множество uid'ов, которые когда-либо привязывали номер,
            u'bindings_count': общее число привязок номера,
        }
        """
        def build_dict():
            return {u'uids': set(), u'bindings_count': 0}

        if not phone_numbers:
            return {}

        response = self.phone_bindings_history(phone_numbers)

        # Поскольку в ответе может не быть некоторых номеров, явно перечислим их в
        # словаре, чтобы клиент мог перебрать все номера и получить по каждому
        # сведения.
        users_confirmed_phone_number = {
            phone_number: build_dict() for phone_number in phone_numbers
        }
        for landmark in response[u'history']:
            phone_number = landmark[u'phone']
            phone = users_confirmed_phone_number[phone_number]
            phone[u'uids'].add(landmark[u'uid'])
            phone[u'bindings_count'] += 1
        return users_confirmed_phone_number


class SaveSimplePhone(object):
    def __init__(
        self,
        account,
        phone_number,
        consumer,
        env,
        statbox,
        blackbox,
        yasms,
        should_ignore_binding_limit=False,
        phone_confirmation_datetime=None,
        is_new_account=False,
        washing_enabled=True,
        should_book_confirmation_to_statbox=True,
    ):
        phone = account.phones.by_number(phone_number)
        if phone and phone.bound:
            self._executor = ConfirmBoundSimplePhone(
                account=account,
                phone_confirmation_datetime=phone_confirmation_datetime,
                phone_number=phone_number,
                should_book_confirmation_to_statbox=should_book_confirmation_to_statbox,
                statbox=statbox,
            )
        else:
            self._executor = BindSimplePhone(
                account=account,
                blackbox=blackbox,
                consumer=consumer,
                env=env,
                is_new_account=is_new_account,
                phone_confirmation_datetime=phone_confirmation_datetime,
                phone_number=phone_number,
                should_book_confirmation_to_statbox=should_book_confirmation_to_statbox,
                should_ignore_binding_limit=should_ignore_binding_limit,
                statbox=statbox,
                washing_enabled=washing_enabled,
                yasms=yasms,
            )

    def submit(self):
        self._executor.submit()

    def commit(self, validation_period=0):
        self._executor.commit()

    def after_commit(self):
        self._executor.after_commit()


class BindSimplePhone(object):
    def __init__(
        self,
        account,
        blackbox,
        consumer,
        env,
        phone_number,
        statbox,
        yasms,
        is_new_account=False,
        phone_confirmation_datetime=None,
        should_book_confirmation_to_statbox=True,
        should_ignore_binding_limit=False,
        washing_enabled=True,
    ):
        """
        Стратегия для привязывания простого телефона к аккаунту

        Основной сценарий использования

        1. Вызвать submit, чтобы убедится что все необходимые условия для
        привязывания простого телефона выполнены.

        2. Вызвать commit под сериализатором изменений модели account, чтобы
        привязать данный телефон к аккаунту.

        3. Вызвать after_commit, чтобы выполнить дополнительные действия
        связанные с привязкой простого телефона к аккаунту.

        Входные параметры

        - is_new_account -- Признак, что привязка происходит в контексте
        регистрации нового аккаунта. Нужен для оптимизации запросов в БД.

        - should_ignore_binding_limit -- Признак, что привязываемый номер не
        нужно учитывать в подсчёте числа связок номера с аккаунтами. Полезно
        для фонишей, т.к. от них нельзя отвязать телефон.

        - washing_enabled -- Признак, что нужно обелить карма аккаунта, если
        привязываемый телефон "чистый".
        """
        timestamp = datetime.now()

        self._account = account
        self._blackbox = blackbox
        self._consumer = consumer
        self._env = env
        self._is_new_account = is_new_account
        self._phone_confirmation_datetime = phone_confirmation_datetime or timestamp
        self._phone_number = phone_number
        self._should_book_confirmation_to_statbox = should_book_confirmation_to_statbox
        self._should_ignore_binding_limit = should_ignore_binding_limit
        self._statbox = statbox
        self._washing_enabled = washing_enabled
        self._yasms = yasms

        self._bound_at = timestamp
        self._code = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)
        self._logical_op = None
        self._phone = None

        # Признак, что карму аккаунта нужно обелить
        self.should_wash_account = False

    def submit(self):
        if not self._is_new_account:
            if self._washing_enabled:
                self.make_wash_decision()
            self.acquire_phone()

    def commit(self):
        if self._is_new_account:
            self.create_bind_operation()

        self.confirm_phone()
        self.apply_bind_operation()

        if self.should_wash_account:
            self.wash_karma()

    def after_commit(self):
        self._statbox.dump_stashes(
            operation_id=self._logical_op.id,
            uid=self._account.uid,
        )

        if not self._should_ignore_binding_limit:
            self.unbind_old_phone()

    def acquire_phone(self):
        """
        Заводим на телефоне операцию, чтобы не было возможности начат привязку
        этого телефона несколько раз или не было других параллельных процессов.
        """
        # Предусловия

        # Не верно, что регистрируется новый аккаунт
        # Когда регистрируется новый аккаунт, создавать телефонную операцию
        # не обязательно, т.к. параллельные телефонные операции невозможны.
        assert not self._is_new_account

        with UPDATE(
            self._account,
            self._env,
            {'action': 'acquire_phone', 'consumer': self._consumer},
            allow_nested=True,
        ):
            self._cancel_conflict_operations()
            self.create_bind_operation()
        self._statbox.dump_stashes(operation_id=self._logical_op.id)

        # Постусловия

        # На аккаунте есть телефон с данным номером
        assert self._phone
        assert self._phone.number == self._phone_number

        # Над телефоном начата операция привязки простого телефона
        assert self._logical_op
        assert type(self._logical_op) is SimpleBindOperation

    def make_wash_decision(self):
        """
        Решает, что карму аккаунта нужно обелить, если привязываемый номер
        "чистый".
        """
        # Предусловия

        # Неверное, что регистрируется новый аккаунт
        # Когда регистрируется новый аккаунт, его карма и так чистая и ничего
        # можно не обелять.
        assert not self._is_new_account

        bindings_history = self._yasms.phone_bindings_history([self._phone_number.e164])
        self.should_wash_account = does_binding_allow_washing(
            self._account,
            self._phone_number.e164,
            self._bound_at,
            self._should_ignore_binding_limit,
            bindings_history,
        )

        # Постусловия

        # assert self.should_wash_account is "Данный номер чистый"

    def create_bind_operation(self):
        """
        Создаёт телефоную операция привязки простого телефона
        """
        # Низкоуровневая логика привязывания телефона инкапсулируются в классе
        # SimpleBindOperation, поэтому нужно создавать телефонную операцию,
        # даже когда не требуется защищаться от гонок.

        # Предусловия

        # assert "Над данным телефоном не идёт других операций"

        self._phone = self._account.phones.by_number(self._phone_number)
        if not self._phone:
            self._phone = self._account.phones.create(number=self._phone_number)

        short_ttl = timedelta(seconds=settings.YASMS_MARK_OPERATION_TTL)

        self._logical_op = SimpleBindOperation.create(
            phone_manager=self._account.phones,
            phone_id=self._phone.id,
            code=self._code,
            should_ignore_binding_limit=self._should_ignore_binding_limit,
            statbox=self._statbox,
            ttl=short_ttl,
        )

        # Постусловия

        # На аккаунте есть телефон с данным номером
        assert self._phone
        assert self._phone.number == self._phone_number

        # Над телефоном начата операция привязки простого телефона
        assert type(self._logical_op) is SimpleBindOperation

    def confirm_phone(self):
        """
        Подтверждает опрерацию привязки простого телефона кодом
        """
        # Для того чтобы привязать простой телефон обязательно требуется
        # подтвердить владение телефона кодом.

        # Предусловия

        assert type(self._logical_op) is SimpleBindOperation
        # assert "Телефонная операция не подтверждена кодом"

        self._logical_op.confirm_phone(
            self._phone.id,
            None,
            timestamp=self._phone_confirmation_datetime,
            should_check_code=False,
            statbox_enabled=self._should_book_confirmation_to_statbox,
        )

        # Постусловия

        # assert "Телефонная операция подтверждена кодом в момент времени self._phone_confirmation_datetime"

    def apply_bind_operation(self):
        """
        Применяет операцию приязки простого телефона
        """
        # Предусловия

        assert type(self._logical_op) is SimpleBindOperation
        # assert "Телефонная операция подтверждена кодом"

        self._logical_op.apply(timestamp=self._bound_at)

        # Постусловия

        assert self._phone
        assert self._phone.bound == self._bound_at
        assert self._phone.confirmed >= self._phone_confirmation_datetime
        assert self._account.phones.secure != self._phone
        assert not self._phone.operation

    def wash_karma(self):
        """
        Обеляет карму данного аккаунта
        """
        self._account.karma.prefix = settings.KARMA_PREFIX_WASHED

    def unbind_old_phone(self):
        """
        Отвязывает телефон от других учётных записей, если число привязок
        достигло предела.

        Данное действие является дополнительным относительно привязки телефона,
        поэтому в случаи временных отказов отвязывание пропускается.
        """
        # Предусловия
        assert self._phone
        assert self._phone.bound

        try:
            unbind_old_phone(
                subject_phone=self._phone,
                blackbox_builder=self._blackbox,
                statbox=self._statbox,
                consumer=self._consumer,
                event_timestamp=self._bound_at,
                environment=self._env,
            )
        except (
            BlackboxInvalidResponseError,
            BlackboxTemporaryError,
            BlackboxUnknownError,
        ) as e:
            log.warning(u'Blackbox error occured while unbind old phones: %s', e)
            return

    def _cancel_conflict_operations(self):
        conflict_ops = SimpleBindOperation.get_conflict_operations(
            self._account.phones,
            self._phone_number.e164,
            self._statbox,
        )
        for conflict_op in conflict_ops:
            conflict_op.cancel()


class ConfirmBoundSimplePhone(object):
    def __init__(
        self,
        account,
        phone_number,
        statbox,
        phone_confirmation_datetime=None,
        should_book_confirmation_to_statbox=True,
    ):
        """
        Стратегия для переподтверждения привязанного простого телефона

        Основной сценарий использования

        1. Вызвать submit, чтобы убедится что все необходимые условия для
        переподтверждения простого телефона выполнены.

        2. Вызвать commit под сериализатором изменений модели account, чтобы
        переподтвердить данный телефон.

        3. Вызвать after_commit, чтобы выполнить дополнительные действия
        связанные с переподтверждением телефона.
        """
        timestamp = datetime.now()

        self._account = account
        self._phone_confirmation_datetime = phone_confirmation_datetime or timestamp
        self._phone_number = phone_number
        self._should_book_confirmation_to_statbox = should_book_confirmation_to_statbox
        self._statbox = statbox

        self._phone = self._account.phones.by_number(self._phone_number)

    def submit(self):
        # Предусловия

        assert self._phone
        assert self._phone.bound
        assert self._account.phones.secure != self._phone

        # Если одновременно удалять телефон, то возможно гонка после которой в
        # БД останется мусорный атрибут подверждения телефона, но телефона не
        # будет. Считаю появляение мусорного атрибута незначительной проблемой.
        #
        # Проблемы гонок телефонных процессов решаются через блокирование
        # телефона операцией. В этом случае проблема решается введением
        # телефонной операции удаления простого номера и выполнением
        # переподтверждения по следующему алгоритму:
        #
        # - Если на телефоне нет операций: создать MarkOperation
        #
        # - Если идёт удаление простого телефона: отменить операцию и создать
        #   создать MarkOperation
        #
        # - Если идёт другая операция, то обновить время подтверждения телефона
        #   не создавая телефонную операция

    def commit(self):
        self.confirm_phone()

    def after_commit(self):
        if self._should_book_confirmation_to_statbox:
            self._statbox.log(
                action='phone_confirmed',
                code_checks_count=0,
                confirmation_time=self._phone_confirmation_datetime,
                phone_id=self._phone.id,
            )

    def confirm_phone(self):
        """
        Обновляет время подтверждения простого телефона
        """
        # Предусловия

        assert self._phone
        assert self._phone.bound
        assert self._account.phones.secure != self._phone

        self._phone.confirm(self._phone_confirmation_datetime)

        # Постусловия

        assert self._phone.confirmed >= self._phone_confirmation_datetime


class _BaseSaveSecurePhone(object):
    def __init__(self, account, phone_number, consumer, env, statbox, blackbox,
                 yasms, should_ignore_binding_limit=False,
                 phone_confirmation_datetime=None, is_new_account=False,
                 washing_enabled=True, should_book_confirmation_to_statbox=True):
        self._phone = None
        self._logical_op = None
        self._should_wash_account = None

        self._account = account
        self._phone_number = phone_number
        self._consumer = consumer
        self._env = env
        self._statbox = statbox
        self._blackbox = blackbox
        self._yasms = yasms
        self._should_ignore_binding_limit = should_ignore_binding_limit
        self._is_new_account = is_new_account
        self._washing_enabled = washing_enabled
        self._should_book_confirmation_to_statbox = should_book_confirmation_to_statbox

        phone = self._account.phones.by_number(self._phone_number)
        self._is_bound = phone and phone.bound

        self._timestamp = datetime.now()

        self._phone_confirmation_datetime = phone_confirmation_datetime or self._timestamp

        if not self._is_bound:
            self._bound_at = self._timestamp
        else:
            self._updated_at = self._timestamp

        self._code = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)

    def submit(self):
        if not self._is_bound:
            if self._washing_enabled:
                self._wash_submit()
            self._bind_submit()
        else:
            self._update_submit()

    def _bind_submit(self):
        if not self._is_new_account:
            with UPDATE(
                self._account,
                self._env,
                {'action': 'acquire_phone', 'consumer': self._consumer},
                allow_nested=True,
            ):
                self._cancel_bind_conflict_operations()
                self._create_bind_operation_commit()
                self._do_extra_work_submit()
            self._statbox.dump_stashes(operation_id=self._logical_op.id)

    def _wash_submit(self):
        if not self._is_new_account:
            bindings_history = self._yasms.phone_bindings_history([self._phone_number.e164])
            self._should_wash_account = does_binding_allow_washing(
                self._account,
                self._phone_number.e164,
                self._bound_at,
                self._should_ignore_binding_limit,
                bindings_history,
            )

    def _update_submit(self):
        if not self._is_new_account:
            with UPDATE(
                self._account,
                self._env,
                {'action': 'cancel_conflict_operations', 'consumer': self._consumer},
                allow_nested=True,
            ):
                self._cancel_update_conflict_operations()
            self._statbox.dump_stashes()

            with UPDATE(
                self._account,
                self._env,
                {'action': 'acquire_phone', 'consumer': self._consumer},
                allow_nested=True,
            ):
                self._create_update_operation_commit()
                self._do_extra_work_submit()
            self._statbox.dump_stashes(operation_id=self._logical_op.id)

    def _cancel_conflict_operations(self, operation_class):
        conflict_ops = operation_class.get_conflict_operations(
            self._account.phones,
            self._phone_number.e164,
            self._statbox,
        )
        for conflict_op in conflict_ops:
            conflict_op.cancel()

    def _create_operation_commit(self, create_operation):
        phone = self._account.phones.by_number(self._phone_number)
        if not phone:
            phone = self._account.phones.create(number=self._phone_number)

        short_ttl = timedelta(seconds=settings.YASMS_MARK_OPERATION_TTL)

        logical_op = create_operation(
            phone_manager=self._account.phones,
            phone_id=phone.id,
            code=self._code,
            statbox=self._statbox,
            ttl=short_ttl,
        )

        self._phone = phone
        self._logical_op = logical_op

    def commit(self, validation_period=0):
        if not self._is_bound:
            self._bind_commit()
            if self._washing_enabled and self._should_wash_account:
                self._wash_commit()
        else:
            self._update_commit()

    def _bind_commit(self):
        if self._is_new_account:
            self._create_bind_operation_commit()

        self._confirm_logical_operation()
        self._logical_op.apply(timestamp=self._bound_at)

    def _wash_commit(self):
        # Обеляем учётную запись
        self._account.karma.prefix = settings.KARMA_PREFIX_WASHED

    def _update_commit(self):
        self._confirm_logical_operation()
        self._logical_op.apply(timestamp=self._updated_at)

    def after_commit(self):
        self._statbox.dump_stashes(
            operation_id=self._logical_op.id,
            uid=self._account.uid,
        )

        if not self._is_bound and not self._should_ignore_binding_limit:
            try:
                unbind_old_phone(
                    subject_phone=self._phone,
                    blackbox_builder=self._blackbox,
                    statbox=self._statbox,
                    consumer=self._consumer,
                    event_timestamp=self._bound_at,
                    environment=self._env,
                )
            except (
                BlackboxInvalidResponseError,
                BlackboxTemporaryError,
                BlackboxUnknownError,
            ) as e:
                log.warning(u'Blackbox error occured while unbind old phones: %s', e)
                return

    def _do_extra_work_submit(self):
        pass


class SaveSecurePhone(_BaseSaveSecurePhone):
    def __init__(
        self,
        aliasify=False,
        language=None,
        password_verification_datetime=None,
        phone_alias_manager_cls=None,
        should_notify_by_email=False,
        notification_language=None,
        should_check_if_alias_allowed=True,
        securify_without_phone_confirmation=False,
        allow_to_take_busy_alias_from_any_account=True,
        allow_to_take_busy_alias_from_neophonish=False,
        allow_to_take_busy_alias_if_not_search=False,
        enable_search_alias=True,
        check_account_type_on_submit=True,
        *args,
        **kwargs
    ):
        super(SaveSecurePhone, self).__init__(*args, **kwargs)

        self._phone_alias_manager_cls = phone_alias_manager_cls
        self._aliasification = None
        self._alias_owner = None
        self._aliasify = aliasify
        self._enable_search_alias = enable_search_alias
        self._allow_to_take_busy_alias_from_any_account = allow_to_take_busy_alias_from_any_account
        self._allow_to_take_busy_alias_from_neophonish = allow_to_take_busy_alias_from_neophonish
        self._allow_to_take_busy_alias_if_not_search = allow_to_take_busy_alias_if_not_search
        self._check_account_type_on_submit = check_account_type_on_submit

        # Язык обязательных сообщений (например, комментарий к коду
        # подтверждения).
        self._language = language

        # Язык необязательных уведомлений.
        self._notification_language = notification_language

        self._should_notify_by_email = should_notify_by_email
        self._should_check_if_alias_allowed = should_check_if_alias_allowed
        self._securify_without_phone_confirmation = securify_without_phone_confirmation

        self._password_verification_datetime = password_verification_datetime or self._timestamp

    def submit(self):
        if self._aliasify:
            self._aliasification = Aliasification(
                self._account,
                self._phone_number,
                self._consumer,
                self._blackbox,
                self._statbox,
                self._language,
                should_check_secure_phone=False,
                phone_alias_manager_cls=self._phone_alias_manager_cls,
                should_check_if_alias_allowed=self._should_check_if_alias_allowed,
                enable_search=self._enable_search_alias,
            )
            self._alias_owner = self._aliasification.get_owner()

            if (
                self._alias_owner and
                self._alias_owner.uid != self._account.uid and
                not (
                    self._allow_to_take_busy_alias_from_any_account or
                    (self._allow_to_take_busy_alias_from_neophonish and self._alias_owner.is_neophonish) or
                    (self._allow_to_take_busy_alias_if_not_search and not self._alias_owner.phonenumber_alias.enable_search)
                )
            ):
                self._aliasification = None
                self._alias_owner = None

        if self._check_account_type_on_submit and not self._is_account_type_allowed():
            raise exceptions.YaSmsSecureNumberNotAllowed()

        if self._account.phones.secure:
            raise exceptions.YaSmsSecureNumberExists()

        super(SaveSecurePhone, self).submit()

    def _cancel_bind_conflict_operations(self):
        self._cancel_conflict_operations(SecureBindOperation)

    def _create_bind_operation_commit(self):
        create_operation = partial(
            SecureBindOperation.create,
            should_ignore_binding_limit=self._should_ignore_binding_limit,
        )
        self._create_operation_commit(create_operation)

    def _cancel_update_conflict_operations(self):
        self._cancel_conflict_operations(SecurifyOperation)

    def _create_update_operation_commit(self):
        self._create_operation_commit(SecurifyOperation.create)

    def _bind_commit(self):
        if not self._is_account_type_allowed():
            raise exceptions.YaSmsSecureNumberNotAllowed()

        if self._is_new_account:
            self._create_bind_operation_commit()

        self._confirm_logical_operation()
        self._logical_op.apply(
            timestamp=self._bound_at,
            need_authenticated_user=self._account.have_password,
        )

    def _update_commit(self):
        self._confirm_logical_operation()
        self._logical_op.apply(
            timestamp=self._updated_at,
            ignore_phones_confirmation=self._securify_without_phone_confirmation,
            need_authenticated_user=self._account.have_password,
        )

    def _confirm_logical_operation(self):
        if not (self._securify_without_phone_confirmation and
                type(self._logical_op) is SecurifyOperation):
            self._logical_op.confirm_phone(
                self._phone.id,
                None,
                timestamp=self._phone_confirmation_datetime,
                should_check_code=False,
                statbox_enabled=self._should_book_confirmation_to_statbox,
            )
        if self._account.is_password_set_or_promised:
            self._logical_op.password_verified = self._password_verification_datetime

    def _is_account_type_allowed(self):
        return (
            self._account.is_lite or
            self._account.is_normal or
            self._account.is_pdd or
            self._account.is_federal or
            self._account.is_social or
            self._account.is_neophonish
        )

    def commit(self, validation_period=0):
        if self._alias_owner:
            try:
                with UPDATE(
                    self._alias_owner,
                    self._env,
                    {'action': 'phone_alias_delete', 'consumer': self._consumer},
                    initiator_uid=self._account.uid,
                ):
                    self._aliasification.take_away()
            except DBError:
                self._statbox.log(error=u'alias.isnt_deleted', operation=u'dealiasify', uid=self._alias_owner.uid)
                raise

        super(SaveSecurePhone, self).commit()

        if self._aliasification:
            self._aliasification.give_out(validation_period=validation_period)

    def after_commit(self):
        if self._should_notify_by_email and self._aliasification:
            self._aliasification.notify()

        if self._should_notify_by_email:
            notify_user_by_email_that_secure_phone_bound(
                self._account,
                language=self._notification_language,
            )

        super(SaveSecurePhone, self).after_commit()

    def _do_extra_work_submit(self):
        if self._aliasification:
            # Восстановление после ошибки. На самом деле телефонного
            # алиаса не может быть на номере, потому что он не защищён.
            self._aliasification.take_old_alias_away()


class ReplaceSecurePhone(object):
    STATUS = [
        'CAN_BIND_SIMPLE_PHONE',
        'CAN_REPLACE_SECURE_PHONE',
        'CAN_START_QUARANTINE',
    ]
    STATUS = namedtuple('STATUS', STATUS)._make(STATUS)

    def __init__(
        self,
        account,
        phone_number,
        consumer,
        env,
        statbox,
        blackbox,
        yasms,
        yasms_builder,
        language=None,
        does_user_admit_secure_number=None,
        is_secure_phone_confirmed=False,
        is_simple_phone_confirmed=False,
        is_password_verified=False,
        is_long_lived=False,
        notification_language=None,
        view=None,
    ):
        """
        does_user_admit_secure_number
            Если ещё не ведётся замена на phone_number, и значение равно None, то
            считаем, что пользователь владеет защищённым номером.
            Если ведётся замена на phone_number, и значение равно None, то
            берётся значение из операции.
            В остальных случаях входной параметр важней значения в операции.

        :raises: InvalidPhoneNumber, может броситься, если задавать phone_number
            строкой.
        """
        self._logical_op = None
        self._should_wash_account = None
        self._is_password_verified = None
        self._status = None
        self._submitted = False

        self._account = account
        self._phone_number = phone_number
        self._does_user_admit_secure_number = does_user_admit_secure_number
        self._statbox = statbox
        self._yasms_builder = yasms_builder
        self._yasms_api = yasms
        self._consumer = consumer
        self._env = env
        self._language = language
        self._is_secure_phone_confirmed = is_secure_phone_confirmed
        self._is_simple_phone_confirmed = is_simple_phone_confirmed
        self._is_password_verified = is_password_verified
        self._blackbox = blackbox
        self._is_long_lived = is_long_lived
        self._notification_language = notification_language or get_preferred_language(self._account)
        self._should_ignore_binding_limit = False
        self._view = view

        phone = self._account.phones.by_number(self._phone_number)
        self._was_bound = phone and phone.bound
        if not self._was_bound:
            self._bound_at = datetime.now()

    def submit(self):
        """
        :raises: OperationInapplicable
        :raises: YaSmsAccountInvalidTypeError
        :raises: YaSmsActionNotRequiredError
        :raises: YaSmsConflictedOperationExists
        :raises: YaSmsLimitExceeded
        :raises: YaSmsSecurePhoneNotFoundError
        :raises: YaSmsTemporaryBlock
        :raises: YaSmsUidLimitExceeded
        """
        with notify_about_phone_changes(
            account=self._account,
            yasms_builder=self._yasms_builder,
            statbox=self._statbox,
            consumer=self._consumer,
            language=self._notification_language,
            client_ip=self._env.user_ip,
            user_agent=self._env.user_agent,
            view=self._view,
        ):
            self._submit()
            self._submitted = True

    def _submit(self):
        if self._account.is_neophonish:
            raise exceptions.YaSmsAccountInvalidTypeError()

        secure_phone = self._account.phones.secure
        simple_phone = self._account.phones.by_number(self._phone_number)
        simple_phone_has_operation = simple_phone and simple_phone.operation

        if simple_phone_has_operation:
            simple_operation = simple_phone.get_logical_operation(self._statbox)
            simple_phone_replaces_secure = type(simple_operation) in {
                ReplaceSecurePhoneWithNonboundPhoneOperation,
                ReplaceSecurePhoneWithBoundPhoneOperation,
            }
            simple_phone_being_bound = (
                simple_operation.is_binding and
                not simple_operation.is_secure and
                not simple_phone.bound
            )
        else:
            simple_phone_replaces_secure = False
            simple_phone_being_bound = False

        if not secure_phone:
            raise exceptions.YaSmsSecurePhoneNotFoundError()

        if secure_phone == simple_phone:
            raise exceptions.YaSmsActionNotRequiredError()

        if self.is_confilict_operation_exist():
            raise exceptions.YaSmsConflictedOperationExists()

        if not (simple_phone and simple_phone.bound):
            self._wash_submit()

        with UPDATE(
            self._account,
            self._env,
            {
                'action': 'phone_secure_replace_submit',
                'consumer': self._consumer,
            },
        ):
            if not simple_phone_replaces_secure:
                if simple_phone_being_bound:
                    simple_operation = simple_phone.get_logical_operation(self._statbox)
                    simple_operation.cancel()
                    simple_phone = self._account.phones.create(
                        self._phone_number,
                        existing_phone_id=simple_phone.id,
                    )

                if not simple_phone:
                    simple_phone = self._account.phones.create(self._phone_number)

                self._logical_op = self._create_operation()

                if self._is_long_lived:
                    self._send_confirmation_codes()
                else:
                    self._status = self._check_operation()
            else:
                self._logical_op = simple_phone.get_logical_operation(self._statbox)
                confirmation_info = self._logical_op.get_confirmation_info(self._account.phones.secure.id)
                if not (self._does_user_admit_secure_number is None or
                        # Если пользователь уже начинал замену на указанный
                        # номер и подтвердил защищённый, то не обращаем
                        # внимание на то, признает ли пользователь защищённый
                        # номер или нет.
                        self._logical_op.does_user_admit_phone and confirmation_info.code_confirmed or
                        self._does_user_admit_secure_number == self._logical_op.does_user_admit_phone):
                    self._logical_op.cancel()
                    simple_phone = self._account.phones.by_number(self._phone_number)
                    if not simple_phone:
                        simple_phone = self._account.phones.create(self._phone_number)
                    self._logical_op = self._create_operation()
                self._status = self._check_operation()

        self._statbox.dump_stashes(operation_id=self._logical_op.id)

    def _check_operation(self):
        reason = self._logical_op.is_inapplicable(
            need_authenticated_user=self._account.have_password,
        )
        if reason is None:
            return self.STATUS.CAN_REPLACE_SECURE_PHONE
        if not isinstance(reason, OperationInapplicable):
            raise reason

        secure_phone = self._account.phones.secure
        simple_phone = self._account.phones.by_number(self._phone_number)

        if self._is_secure_phone_confirmed:
            reason.need_confirmed_phones.discard(secure_phone)

        if self._is_simple_phone_confirmed:
            reason.need_confirmed_phones.discard(simple_phone)

        if self._is_password_verified:
            reason.need_password_verification = False

        is_secure_confirmed = secure_phone not in reason.need_confirmed_phones
        is_simple_confirmed = simple_phone not in reason.need_confirmed_phones
        is_password_verified = not reason.need_password_verification

        if is_secure_confirmed and is_simple_confirmed and is_password_verified:
            return self.STATUS.CAN_REPLACE_SECURE_PHONE
        elif is_simple_confirmed and is_password_verified and not self._logical_op.does_user_admit_phone:
            return self.STATUS.CAN_START_QUARANTINE
        elif is_simple_confirmed and not simple_phone.bound:
            return self.STATUS.CAN_BIND_SIMPLE_PHONE
        else:
            raise reason

    def _send_confirmation_codes(self):
        if not self._is_simple_phone_confirmed:
            simple_phone = self._account.phones.by_number(self._phone_number)
            self._send_confirmation_code(simple_phone)

        if not self._is_secure_phone_confirmed:
            secure_phone = self._account.phones.secure
            if secure_phone.operation and secure_phone.operation.does_user_admit_phone:
                self._send_confirmation_code(secure_phone)

    def _send_confirmation_code(self, phone):
        logical_op = phone.get_logical_operation(self._statbox)
        confirmation_info = logical_op.get_confirmation_info(phone.id)
        send_code = self._build_send_confirmation_code(confirmation_info, phone.number)
        try:
            with self._statbox.make_context(action='phone_secure_replace.submit'):
                send_code()
        except Exception:
            self._statbox.dump_stashes()
            raise
        logical_op.set_confirmation_info(phone.id, confirmation_info)

    def _build_send_confirmation_code(self, confirmation_info, phone_number):
        return build_send_confirmation_code(
            phone_number,
            confirmation_info,
            self._account,
            self._env,
            self._yasms_builder,
            self._consumer,
            self._statbox,
            self._language,
        )

    def _create_operation(self):
        secure_phone = self._account.phones.secure
        simple_phone = self._account.phones.by_number(self._phone_number)

        simple_code = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)

        if (self._does_user_admit_secure_number is None or
                self._does_user_admit_secure_number):
            secure_code = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)
        else:
            secure_code = None

        if self._is_long_lived:
            operation_ttl = None
        else:
            operation_ttl = timedelta(seconds=settings.YASMS_MARK_OPERATION_TTL)

        if not simple_phone.bound:
            operation = ReplaceSecurePhoneWithNonboundPhoneOperation.create(
                phone_manager=self._account.phones,
                secure_phone_id=secure_phone.id,
                being_bound_phone_id=simple_phone.id,
                secure_code=secure_code,
                being_bound_code=simple_code,
                statbox=self._statbox,
                ttl=operation_ttl,
            )
        else:
            operation = ReplaceSecurePhoneWithBoundPhoneOperation.create(
                phone_manager=self._account.phones,
                secure_phone_id=secure_phone.id,
                simple_phone_id=simple_phone.id,
                secure_code=secure_code,
                simple_code=simple_code,
                statbox=self._statbox,
                ttl=operation_ttl
            )
        return operation

    def commit(self):
        with notify_about_phone_changes(
            account=self._account,
            yasms_builder=self._yasms_builder,
            statbox=self._statbox,
            consumer=self._consumer,
            language=self._notification_language,
            client_ip=self._env.user_ip,
            user_agent=self._env.user_agent,
            view=self._view,
        ):
            self._commit()

    def _commit(self):
        if self._is_simple_phone_confirmed:
            try:
                simple_phone = self._account.phones.by_number(self._phone_number)
                self._logical_op.confirm_phone(simple_phone.id, None, should_check_code=False)
            except PhoneConfirmedAlready:
                pass

        if self._is_secure_phone_confirmed:
            try:
                secure_phone = self._account.phones.secure
                self._logical_op.confirm_phone(secure_phone.id, None, should_check_code=False)
            except PhoneConfirmedAlready:
                pass

        if self._is_password_verified and not self._logical_op.password_verified:
            self._logical_op.password_verified = datetime.now()

        if not self._was_bound:
            timestamp = self._bound_at
        else:
            timestamp = None

        try:
            self._next_logical_op, changes = self._logical_op.apply(
                need_authenticated_user=self._account.have_password,
                timestamp=timestamp,
            )
        except OperationInapplicable as e:
            self._next_logical_op = self._logical_op
            if (
                self._logical_op.is_ready_for_quarantine(need_authenticated_user=self._account.have_password) and
                not self._logical_op.in_quarantine
            ):
                self._logical_op.start_quarantine()
            if self._logical_op.in_quarantine:
                return
            raise e

        if (
            self._next_logical_op and
            self._next_logical_op.is_ready_for_quarantine(need_authenticated_user=self._account.have_password) and
            not self._next_logical_op.in_quarantine
        ):
            self._next_logical_op.start_quarantine()

        if changes.bound_numbers and self._should_wash_account:
            self._wash_commit()

        if changes.unbound_numbers:
            has_alias = self._account.phonenumber_alias and self._account.phonenumber_alias.number
            if has_alias:
                aliasification = Aliasification(
                    account=self._account,
                    phone_number=self._phone_number,
                    consumer=self._consumer,
                    blackbox=self._blackbox,
                    statbox=DummyLogger(),
                    language=self._language,
                    # Т.к. собираемся только удалить старый алиас, опустим
                    # проверки на создание алиаса.
                    should_check_if_alias_allowed=False,
                )
                aliasification.take_old_alias_away()

    def after_commit(self):
        self._statbox.dump_stashes(operation_id=self._logical_op.id)

        phone = self._account.phones.by_number(self._phone_number)
        if not self._was_bound and phone.bound:
            try:
                unbind_old_phone(
                    subject_phone=phone,
                    blackbox_builder=self._blackbox,
                    statbox=self._statbox,
                    consumer=self._consumer,
                    event_timestamp=self._bound_at,
                    environment=self._env,
                )
            except (
                BlackboxInvalidResponseError,
                BlackboxTemporaryError,
                BlackboxUnknownError,
            ) as e:
                log.warning(u'Blackbox error occured while unbind old phones: %s', e)
                return

        if self._next_logical_op is not None:
            # Оставшуюся операцию гарантировано нельзя применить, узнаем почему.
            try:
                self._next_logical_op.apply(
                    need_authenticated_user=self._account.have_password,
                )
            except OperationInapplicable as e:
                if self._next_logical_op.in_quarantine:
                    return
                raise e
            finally:
                self._statbox.dump_stashes()

    def _wash_submit(self):
        if not self._was_bound:
            bindings_history = self._yasms_api.phone_bindings_history([self._phone_number.e164])
            self._should_wash_account = does_binding_allow_washing(
                self._account,
                self._phone_number.e164,
                self._bound_at,
                self._should_ignore_binding_limit,
                bindings_history,
            )

    def _wash_commit(self):
        # Обеляем учётную запись
        self._account.karma.prefix = settings.KARMA_PREFIX_WASHED

    def is_confilict_operation_exist(self):
        """
        Признак, что пользователь уже начал какую-то телефонную операцию,
        которая не даст заменить основной телефон.
        """
        secure_phone = self._account.phones.secure
        simple_phone = self._account.phones.by_number(self._phone_number)
        simple_phone_has_operation = simple_phone and simple_phone.operation

        if simple_phone_has_operation:
            simple_operation = simple_phone.get_logical_operation(self._statbox)
            simple_phone_replaces_secure = type(simple_operation) in {
                ReplaceSecurePhoneWithNonboundPhoneOperation,
                ReplaceSecurePhoneWithBoundPhoneOperation,
            }
            simple_phone_being_bound = (
                simple_operation.is_binding and
                not simple_operation.is_secure and
                not simple_phone.bound
            )
        else:
            simple_phone_replaces_secure = False
            simple_phone_being_bound = False

        if secure_phone and secure_phone == simple_phone:
            return False

        if simple_phone_has_operation and not (simple_phone_replaces_secure or simple_phone_being_bound):
            return True

        if (
            secure_phone and
            secure_phone.operation and
            not simple_phone_replaces_secure
        ):
            return True

        return False

    def is_replacing(self):
        """
        Признак, что выполняется замена основного телефона (без карантина)
        """
        self._check_submitted()
        return self._status == self.STATUS.CAN_REPLACE_SECURE_PHONE

    def is_starting_replacement(self):
        """
        Признак, что выполняется замена основного телефона через карантин
        """
        self._check_submitted()
        return self._status == self.STATUS.CAN_START_QUARANTINE

    def is_binding_simple(self):
        """
        Признак, что выполняется привязка простого телефона
        """
        self._check_submitted()
        return self._status == self.STATUS.CAN_BIND_SIMPLE_PHONE

    def _check_submitted(self):
        if not self._submitted:
            raise NotImplementedError('Call method submit first')


class BindSecurestPossiblePhone(object):
    def __init__(
        self,
        account,
        phone_number,
        consumer,
        env,
        statbox,
        blackbox,
        yasms_api,
        yasms_builder,
        language=None,
        check_account_type_on_submit=True,
        view=None,
    ):
        self._executor = None
        self._is_secure_phone_confirmation = None
        self._submitted = False
        self._committed = False

        self._account = account

        if not isinstance(phone_number, PhoneNumber):
            phone_number = PhoneNumber.parse(phone_number)
        self._phone_number = phone_number

        self._consumer = consumer
        self._env = env
        self._statbox = statbox
        self._blackbox = blackbox
        self._yasms_api = yasms_api
        self._yasms_builder = yasms_builder
        self._language = language
        self._check_account_type_on_submit = check_account_type_on_submit
        self._view = view

    def submit(self):
        kwargs = {
            'account': self._account,
            'phone_number': self._phone_number,
            'consumer': self._consumer,
            'env': self._env,
            'statbox': self._statbox,
            'blackbox': self._blackbox,
            'yasms': self._yasms_api,
        }

        executor = SaveSecurePhone(
            check_account_type_on_submit=self._check_account_type_on_submit,
            language=self._language,
            should_notify_by_email=True,
            **kwargs
        )
        try:
            executor.submit()
        except exceptions.YaSmsSecureNumberExists:
            if self._phone_number != self._account.phones.secure.number:
                executor = self._build_replace_secure_phone()
                try:
                    executor.submit()
                except (
                    exceptions.YaSmsAccountInvalidTypeError,
                    exceptions.YaSmsConflictedOperationExists,
                ):
                    executor = SaveSimplePhone(**kwargs)
                    executor.submit()
            else:
                self._is_secure_phone_confirmation = True
                executor = None
        except exceptions.YaSmsSecureNumberNotAllowed:
            executor = SaveSimplePhone(**kwargs)
            executor.submit()

        self._executor = executor
        self._submitted = True

    def commit(self):
        self._check_submitted()

        if self._is_secure_phone_confirmation:
            timestamp = datetime.now()
            self._account.phones.secure.confirmed = timestamp
        else:
            self._executor.commit()

        self._committed = True

    def after_commit(self):
        self._check_committed()

        if not self._is_secure_phone_confirmation:
            self._executor.after_commit()

    def is_updating_old_secure(self):
        """
        Признак, что данный номер уже и так основной и выполняется его
        переподтверждение.
        """
        self._check_submitted()
        return self._executor is None and self._is_secure_phone_confirmation

    def is_binding_secure(self):
        """
        Признак, что выполняется привязка основного номера
        """
        self._check_submitted()
        return type(self._executor) is SaveSecurePhone

    def is_binding_simple(self):
        """
        Признак, что выполняется привязка или переподтверждение дополнительного
        номера.
        """
        self._check_submitted()
        return type(self._executor) is SaveSimplePhone

    def is_replacing(self):
        """
        Признак, что выполняется замена основного номера на данный без
        карантина.
        """
        self._check_submitted()

        if not isinstance(self._executor, ReplaceSecurePhone):
            return False

        return self._executor.is_replacing()

    def is_starting_replacement(self):
        """
        Признак, что выполняется замена основного номера на данный через
        карантин.
        """
        self._check_submitted()

        if not isinstance(self._executor, ReplaceSecurePhone):
            return False

        return self._executor.is_starting_replacement()

    def quarantine_finished_at(self):
        """
        Время, когда закончится карантин на замену основного телефона
        """
        self._check_committed()

        timestamp = None

        if self.is_starting_replacement():
            phone_operation = self._account.phones.secure.get_logical_operation(None)
            if phone_operation.in_quarantine:
                timestamp = phone_operation.finished

        return timestamp

    def is_operation_prevent_to_secure_phone(self):
        """
        Признак, что данный телефон нельзя сделать основным из-за того, что
        пользователь уже начал некую другу телефонную операцию.
        """
        self._check_submitted()

        if (
            self.is_binding_secure() or
            self.is_starting_replacement() or
            self.is_updating_old_secure()
        ):
            return False

        replacer = self._build_replace_secure_phone()
        return replacer.is_confilict_operation_exist()

    def _build_replace_secure_phone(self):
        return ReplaceSecurePhone(
            account=self._account,
            blackbox=self._blackbox,
            consumer=self._consumer,
            does_user_admit_secure_number=False,
            env=self._env,
            is_password_verified=True,
            is_simple_phone_confirmed=True,
            language=self._language,
            phone_number=self._phone_number,
            statbox=self._statbox,
            yasms=self._yasms_api,
            yasms_builder=self._yasms_builder,
            view=self._view,
        )

    def _check_submitted(self):
        if not self._submitted:
            raise NotImplementedError('Call method submit first')

    def _check_committed(self):
        if not self._committed:
            raise NotImplementedError('Call method commit first')
