# -*- coding: utf-8 -*-

from flask import request
from flask.views import View
from passport.backend.api.common.decorators import get_request_values
from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.common.format_response import (
    JsonLoggedResponse,
    XmlLoggedResponse,
)
from passport.backend.api.yasms import (
    errors,
    exceptions as yasms_exceptions,
    forms,
)
from passport.backend.api.yasms.api import Yasms
from passport.backend.api.yasms.utils import get_passport_consumer_from_yasms_consumer
from passport.backend.core.builders.blackbox import (
    exceptions as blackbox_exceptions,
    get_blackbox,
)
from passport.backend.core.builders.yasms import get_yasms
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.grants import (
    get_grants,
    get_grants_config,
    MissingOptionalGrantsError,
    MissingRequiredGrantsError,
    MissingTicketError,
    TicketParsingError,
    UnknownConsumerError,
)
from passport.backend.core.models.phones.phones import OperationExpired
from passport.backend.core.serializers.eav.exceptions import (
    EavDeletedObjectNotFound,
    EavUpdatedObjectNotFound,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import ClassMapping


class YasmsXmlResponse(XmlLoggedResponse):
    """Xml ответ Я.Смса"""


class YasmsJsonResponse(JsonLoggedResponse):
    """Json ответ Я.Смса"""


class XmlErrorResponse(YasmsXmlResponse):
    def __init__(self, code, message, status_code):
        super(XmlErrorResponse, self).__init__(
            {u'code': code, u'message': message},
            format_=format_xml_error,
        )
        self.status_code = status_code


class JsonErrorResponse(YasmsJsonResponse):
    def __init__(self, code, message, status_code):
        super(JsonErrorResponse, self).__init__({
            u'error': code,
        })
        self.status_code = status_code


def format_xml_error(error_dict):
    return u'''<?xml version="1.0" encoding="utf-8"?>
               <doc>
               <error>{message}</error>
               <errorcode>{code}</errorcode>
               </doc>
           '''.format(**error_dict)


class BaseYasmsView(View):
    """Основа для контроллеров паспортного Я.Смса"""

    required_grants = tuple()
    basic_form = None
    sensitive_fields = tuple()

    _root_form = None

    _errors = errors
    _error_response_class = None

    def __init__(self, *args, **kwargs):
        super(BaseYasmsView, self).__init__(*args, **kwargs)

        self._form_field_and_code_to_error = {
            (u'all', u'string'): self._errors.BadArg(u'all'),
            (u'block', u'string'): self._errors.BadArg(u'block'),
            (u'secure', u'string'): self._errors.BadArg(u'secure'),
            (u'revalidate', u'string'): self._errors.BadArg(u'revalidate'),
            (u'withoutsms', u'string'): self._errors.BadArg(u'withoutsms'),
            (u'ignore_bindlimit', u'string'): self._errors.BadArg(u'ignore_bindlimit'),
            (u'format', u'invalid'): self._errors.BadArg(u'format'),

            (u'code', u'empty'): self._errors.NoCode(),
            (u'code', u'missingValue'): self._errors.NoCode(),
            (u'code', u'invalid'): self._errors.NoCode(),

            (u'number', u'badPhone'): self._errors.BadPhone(),
            (u'number', u'empty'): self._errors.NoPhone(),
            (u'number', u'missingValue'): self._errors.NoPhone(),

            (u'number_or_phoneid', u'tooFew'): self._errors.NoPhone(),

            (u'phone', u'badPhone'): self._errors.BadPhone(),
            (u'phone', u'empty'): self._errors.NoPhone(),
            (u'phone', u'missingValue'): self._errors.NoPhone(),

            (u'phoneid', u'empty'): self._errors.NoPhone(),
            (u'phoneid', u'integer'): self._errors.NoPhone(),
            (u'phoneid', u'missingValue'): self._errors.NoPhone(),
            (u'phoneid', u'tooLow'): self._errors.NoPhone(),

            (u'sender', u'empty'): self._errors.NoSender(),
            (u'sender', u'invalid'): self._errors.DontKnowYou(),
            (u'sender', u'missingValue'): self._errors.NoSender(),

            (u'ts', u'integer'): self._errors.BadTsFormat(),
            (u'ts', u'tooHigh'): self._errors.BadTsFormat(),
            (u'ts', u'tooLow'): self._errors.BadTsFormat(),

            (u'uid', u'empty'): self._errors.NoUid(),
            (u'uid', u'integer'): self._errors.NoUid(),
            (u'uid', u'missingValue'): self._errors.NoUid(),
            (u'uid', u'tooLow'): self._errors.NoUid(),
        }

        self._EXCEPTION_MAPPING = ClassMapping([
            (
                yasms_exceptions.YaSmsAccessDenied,
                self._errors.InternalError(u'Old YaSms did not give access to Passport'),
            ),
            (
                yasms_exceptions.YaSmsTemporaryError,
                self._errors.InternalError(u'Old YaSms internal error occured'),
            ),
            (
                yasms_exceptions.YaSmsNoSender,
                self._errors.InternalError(u'Old YaSms says Passport did not provide "sender" argument'),
            ),
            (
                yasms_exceptions.YaSmsSyntaxError,
                self._errors.InternalError(u'Old YaSms replied with invalid response'),
            ),
            (
                blackbox_exceptions.BlackboxTemporaryError,
                self._errors.InternalError(u'Blackbox internal error occured'),
            ),
            (
                blackbox_exceptions.BlackboxUnknownError,
                self._errors.InternalError(u'Blackbox internal error occured'),
            ),
            (
                blackbox_exceptions.BlackboxInvalidResponseError,
                self._errors.InternalError(u'Blackbox replied with invalid response'),
            ),
            (OperationExpired, self._errors.OperationExpired()),
            (DBError, self._errors.DatabaseError()),
            (EavDeletedObjectNotFound, self._errors.InternalError(u'Deleted object not found')),
            (EavUpdatedObjectNotFound, self._errors.InternalError(u'Updated object not found')),
            (
                yasms_exceptions.YaSmsPhoneBindingsLimitExceeded,
                self._errors.InternalError(u'Phonish must has exactly one phone'),
            ),
        ])

    @property
    def consumer(self):
        return self.form_values[u'sender']

    def process_request(self):
        """
        Методу следует заполнить атрибут response_values.
        """
        raise NotImplementedError()  # pragma: no cover

    @classmethod
    def format_response(cls, response):
        """
        Методу следует преобразовать словарь response в закодированный xml
        документ и вернуть результат.
        """
        raise NotImplementedError()  # pragma: no cover

    def respond_success(self):
        """
        Методу следует сформировать Http ответ.
        """
        raise NotImplementedError()  # pragma: no cover

    def setup(self):
        self.request = request
        self.sender_ip = request.env.consumer_ip
        self.client_ip = request.env.user_ip
        self.host = request.env.host
        self.user_agent = request.env.user_agent
        self.service_ticket = request.env.service_ticket
        # словарь с полезной работой ручки
        self.response_values = {}
        # проверенные параметры переданные в ручку
        self.form_values = {}

    def _validate_params(self, form, unsafe_params):
        try:
            return form.to_python(unsafe_params)
        except forms.Invalid as e:
            field, code = forms.form_encode_invalid_to_field_and_code(e)
            error = self._form_field_and_code_to_error[(field, code)]
            raise error

    def process_basic_form(self):
        unsafe_params = get_request_values()
        safe_params = self._validate_params(self.basic_form, unsafe_params)
        self.form_values.update(safe_params)

    def process_root_form(self):
        unsafe_params = get_request_values()
        safe_params = self._validate_params(self._root_form, unsafe_params)
        self.form_values.update(safe_params)

    def grant_is_issued(self, grant):
        try:
            self._check_grants([], [grant])
            ret = True
        except (UnknownConsumerError, MissingOptionalGrantsError):
            ret = False
        return ret

    def require_grants(self):
        try:
            self._check_grants(self.required_grants, [])
        except (UnknownConsumerError, MissingTicketError, TicketParsingError):
            raise self._errors.DontKnowYou()
        except MissingRequiredGrantsError:
            raise self._errors.NoRights()

    def _check_grants(self, required_grants, optional_grants):
        sender = self.form_values[u'sender']
        if (
            sender is not None and
            sender not in settings.YASMS_CONSUMERS_WITHOUT_OLD_PREFIX
        ):
            sender = get_passport_consumer_from_yasms_consumer(sender)

        get_grants().check_access(
            self.sender_ip,
            get_grants_config().get_consumers(
                self.sender_ip,
                sender,
                False,
            ),
            required_grants,
            optional_grants=optional_grants,
            service_ticket=self.service_ticket,
        )

    def dispatch_request(self):
        self.setup()
        try:
            self.process_root_form()
            self.require_grants()
            self.process_basic_form()
            self.process_request()
            response = self.respond_success()
        except Exception as e:
            response = self.respond_error(e)
        return response

    def respond_error(self, exc):
        if not isinstance(exc, errors.BaseError):
            if exc.__class__ in self._EXCEPTION_MAPPING:
                error = self._EXCEPTION_MAPPING[exc.__class__]
            else:
                log_internal_error(exc)
                error = self._errors.InternalError()
        else:
            error = exc
        return self._error_response_class(
            error.code,
            error.message,
            error.http_status_code,
        )

    @classmethod
    def as_view(cls, name=None, *args, **kwargs):
        name = name or cls.__name__
        return super(BaseYasmsView, cls).as_view(name, *args, **kwargs)

    @cached_property
    def yasms(self):
        return Yasms(self.blackbox, get_yasms(), self.request.env, self)

    @cached_property
    def blackbox(self):
        return get_blackbox()


class YasmsXmlView(BaseYasmsView):
    _error_response_class = XmlErrorResponse

    def respond_success(self):
        response = YasmsXmlResponse(
            self.response_values,
            self.format_response,
            sensitive_fields=self.sensitive_fields,
        )
        return response


class YasmsJsonView(BaseYasmsView):
    _error_response_class = JsonErrorResponse

    def respond_success(self):
        formatted_response = self.format_response(self.response_values)
        response = YasmsJsonResponse(
            formatted_response,
            sensitive_fields=self.sensitive_fields,
        )
        return response
