# -*- coding: utf-8 -*-

from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.forms import (
    NormalizablePhoneNumberValidator,
    RequiredSenderForm,
)
from passport.backend.core import validators
from passport.backend.core.builders.yasms.utils import normalize_phone_number

from .base import YasmsJsonView


_USER_ATTR_TO_SERIALIZE = {
    u'uid': serializers.number_to_str,
    u'active': serializers.bool_to_onezero,
    u'phoneid': serializers.number_to_str,
    u'validation_date': serializers.datetime_to_str,
}


class CheckPhoneForm(RequiredSenderForm):
    phone = NormalizablePhoneNumberValidator()
    all = validators.StringBool(if_empty=False, if_missing=False)


class CheckPhoneView(YasmsJsonView):
    required_grants = (grants.CHECK_PHONE,)
    basic_form = CheckPhoneForm
    sensitive_fields = (u'items[].phone',)

    _root_form = RequiredSenderForm

    def process_request(self):
        phone_number = normalize_phone_number(self.form_values[u'phone'])
        response = self.yasms.check_phone(
            phone_number,
            secure_only=not self.form_values[u'all'],
        )
        self.response_values.update({
            u'items': [{
                u'uid': i[u'uid'],
                u'active': i[u'active'],
                u'phoneid': i[u'phoneid'],
                u'phone': i[u'phone'],
                u'valid': i[u'valid'],
                u'validation_date': i[u'validation_date'],
            } for i in response[u'items']],
            u'binding_limit_exceeded': response[u'binding_limit_exceeded'],
        })

    @classmethod
    def format_response(cls, response):
        formatted_response = build_check_phone_dict(
            response[u'items'],
            response[u'binding_limit_exceeded'],
        )
        return formatted_response


def build_check_phone_dict(users, binding_limit_exceeded):
    items = [serializers.stringify_attr_dict(user, _USER_ATTR_TO_SERIALIZE) for user in users]
    # Как правило Я.Смс возвращает дату пустой строкой, но в check_phone всё по
    # особому.
    for item in items:
        if not item['validation_date']:
            item['validation_date'] = None
    return {
        u'binding_limit_exceeded': serializers.bool_to_int(
            binding_limit_exceeded,
        ),
        u'items': items,
    }
