# -*- coding: utf-8 -*-

import elementflow
from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.forms import OptionalSenderForm
from passport.backend.api.yasms.utils import (
    get_account_by_uid,
    old_mask_phone_number,
)
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from six import BytesIO

from .base import YasmsXmlView


class CheckUserForm(OptionalSenderForm):
    uid = validators.Uid()


class CheckUserView(YasmsXmlView):
    required_grants = tuple()
    basic_form = CheckUserForm
    sensitive_fields = (u'number',)

    _root_form = OptionalSenderForm

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        if account is not None:
            user_info = self.yasms.check_user(account)
        else:
            user_info = {
                u'uid': self.form_values[u'uid'],
                u'has_current_phone': False,
                u'number': None,
                u'cyrillic': None,
                u'blocked': None,
            }

        if not self.grant_is_issued(grants.RETURN_FULL_PHONE):
            if user_info[u'has_current_phone']:
                user_info[u'number'] = old_mask_phone_number(user_info[u'number'])

        self.response_values.update(user_info)

    @classmethod
    def format_response(cls, response):
        return build_check_user_xml(
            response[u'uid'],
            response[u'has_current_phone'],
            response[u'number'],
            response[u'cyrillic'],
            response[u'blocked'],
        )


def build_check_user_xml(uid, has_current_phone, phone_number, cyrillic,
                         blocked):
    out = BytesIO()
    with elementflow.xml(out, u'doc') as doc:
        doc.element(u'uid', text=serializers.number_to_str(uid))
        doc.element(
            u'hascurrentphone',
            text=serializers.bool_to_yesno(has_current_phone),
        )
        if has_current_phone:
            doc.element(u'number', text=phone_number)
            doc.element(u'cyrillic', text=serializers.bool_to_yesno(cyrillic))
            doc.element(u'blocked', text=serializers.bool_to_yesno(blocked))
    xml = out.getvalue()
    return xml
