# -*- coding: utf-8 -*-

from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.api import DELETE_PHONE_STATUS
from passport.backend.api.yasms.forms import (
    PhoneValidator,
    RequiredSenderForm,
)
from passport.backend.api.yasms.utils import (
    get_account_by_uid,
    get_operation_id_by_phone_number,
)
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE

from .base import YasmsJsonView


class DeletePhoneForm(RequiredSenderForm):
    uid = validators.Uid()
    number = PhoneValidator()


class DeletePhoneView(YasmsJsonView, BundlePushMixin, KolmogorMixin):
    required_grants = (grants.DELETE_PHONE,)
    basic_form = DeletePhoneForm
    sensitive_fields = tuple()

    _root_form = RequiredSenderForm

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        # Требуется для бандлового push mixin'а
        self.account = account
        self.track_id = None

        if account is not None:
            statbox = StatboxLogger(
                action=u'delete_phone',
                consumer=self.form_values[u'sender'],
                ip=self.client_ip,
            )
            with UPDATE(
                account,
                self.request.env,
                {
                    u'action': u'delete_phone',
                    u'consumer': self.form_values[u'sender'],
                },
            ):
                status = self.yasms.delete_phone(
                    account,
                    self.form_values[u'number'],
                    user_ip=self.client_ip,
                    statbox=statbox,
                    consumer=self.form_values[u'sender'],
                    user_agent=self.user_agent,
                )

            is_removal_started = status == DELETE_PHONE_STATUS.STARTED
            if is_removal_started:
                # И напоследок выполнием простое механическое действие!
                operation_id = get_operation_id_by_phone_number(
                    account,
                    self.form_values[u'number'],
                )
                statbox.dump_stashes(operation_id=operation_id)
        else:
            status = DELETE_PHONE_STATUS.NOT_FOUND

        self.response_values.update({
            u'uid': self.form_values[u'uid'],
            u'status': status,
        })

    @classmethod
    def format_response(cls, response):
        return build_delete_phone_dict(response[u'uid'], response[u'status'])


def build_delete_phone_dict(uid, status):
    return {
        u'uid': serializers.number_to_str(uid),
        u'status': status.upper(),
    }
