# -*- coding: utf-8 -*-

from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.api import DROP_PHONE_STATUS
from passport.backend.api.yasms.errors import drop_phone as errors
from passport.backend.api.yasms.forms import (
    PhoneIdValidator,
    RequiredSenderForm,
)
from passport.backend.api.yasms.utils import get_account_by_uid
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE

from .base import YasmsJsonView


class DropPhoneForm(RequiredSenderForm):
    uid = validators.Uid()
    phoneid = PhoneIdValidator()


class DropPhoneView(YasmsJsonView):
    required_grants = (grants.DROP_PHONE,)
    basic_form = DropPhoneForm
    sensitive_fields = tuple()

    _root_form = RequiredSenderForm

    _errors = errors

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        if account is not None:
            with UPDATE(
                account,
                self.request.env,
                {
                    u'action': u'drop_phone',
                    u'consumer': self.form_values[u'sender'],
                },
            ):
                response = self.yasms.drop_phone(
                    account,
                    self.form_values[u'phoneid'],
                    StatboxLogger(
                        action=u'drop_phone',
                        consumer=self.form_values[u'sender'],
                        ip=self.client_ip,
                    ),
                    self.form_values[u'sender'],
                )
        else:
            response = {u'status': DROP_PHONE_STATUS.NOT_FOUND}
        self.response_values.update({
            u'uid': self.form_values[u'uid'],
            u'status': response[u'status'],
        })

    @classmethod
    def format_response(cls, response):
        return build_drop_phone_dict(response[u'uid'], response[u'status'])


def build_drop_phone_dict(uid, status):
    return {
        u'uid': serializers.number_to_str(uid),
        u'status': status.upper(),
    }
