# -*- coding: utf-8 -*-

import elementflow
from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.forms import OptionalSenderForm
from passport.backend.api.yasms.utils import get_account_by_uid
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from six import BytesIO

from .base import YasmsXmlView


class HaveUserOnceValidatedPhoneForm(OptionalSenderForm):
    uid = validators.Uid()


class HaveUserOnceValidatedPhoneView(YasmsXmlView):
    required_grants = (grants.HAVE_USER_ONCE_VALIDATED_PHONE,)
    basic_form = HaveUserOnceValidatedPhoneForm
    sensitive_fields = tuple()

    _root_form = OptionalSenderForm

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        if account is not None:
            self.response_values.update(
                self.yasms.have_user_once_validated_phone(account),
            )
        else:
            self.response_values.update({
                u'have_user_once_validated_phone': False,
                u'reason': u'no-phone',
            })

    @classmethod
    def format_response(cls, response):
        return build_have_user_once_validated_phone_xml(
            response[u'have_user_once_validated_phone'],
            response[u'reason'],
        )


def build_have_user_once_validated_phone_xml(have_user_once_validated_phone,
                                             reason):
    out = BytesIO()
    with elementflow.xml(out, u'doc') as doc:
        doc.element(
            u'have_user_once_validated_phone',
            attrs={
                u'value': serializers.bool_to_onezero(
                    have_user_once_validated_phone,
                ),
                u'reason': reason,
            },
        )

    xml = out.getvalue()
    return xml
